"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const metadata_1 = require("./metadata");
const dependency_1 = require("./private/dependency");
const stack_trace_1 = require("./private/stack-trace");
const uniqueid_1 = require("./private/uniqueid");
const CONSTRUCT_NODE_PROPERTY_SYMBOL = Symbol.for('constructs.Construct.node');
/**
 * Represents the construct node in the scope tree.
 */
class Node {
    constructor(host, scope, id) {
        this.host = host;
        this._locked = false; // if this is "true", addChild will fail
        this._aspects = [];
        this._children = {};
        this._context = {};
        this._metadata = new Array();
        this._dependencies = new Set();
        this.invokedAspects = [];
        id = id || ''; // if undefined, convert to empty string
        this.id = sanitizeId(id);
        this.scope = scope;
        // We say that scope is required, but root scopes will bypass the type
        // checks and actually pass in 'undefined'.
        if (scope != null) {
            if (id === '') {
                throw new Error('Only root constructs may have an empty name');
            }
            // Has side effect so must be very last thing in constructor
            Node.of(scope).addChild(host, this.id);
        }
        else {
            // This is a root construct.
            this.id = id;
        }
    }
    /**
     * Returns the node associated with a construct.
     * @param construct the construct
     */
    static of(construct) {
        const node = construct[CONSTRUCT_NODE_PROPERTY_SYMBOL];
        if (!node) {
            throw new Error(`construct does not have an associated node. All constructs must extend the "Construct" base class`);
        }
        return node;
    }
    /**
     * The full, absolute path of this construct in the tree.
     *
     * Components are separated by '/'.
     */
    get path() {
        const components = this.scopes.slice(1).map(c => Node.of(c).id);
        return components.join(Node.PATH_SEP);
    }
    /**
     * A tree-global unique alphanumeric identifier for this construct.
     * Includes all components of the tree.
     */
    get uniqueId() {
        const components = this.scopes.slice(1).map(c => Node.of(c).id);
        return components.length > 0 ? uniqueid_1.makeUniqueId(components) : '';
    }
    /**
     * Return a direct child by id, or undefined
     *
     * @param id Identifier of direct child
     * @returns the child if found, or undefined
     */
    tryFindChild(id) {
        return this._children[sanitizeId(id)];
    }
    /**
     * Return a direct child by id
     *
     * Throws an error if the child is not found.
     *
     * @param id Identifier of direct child
     * @returns Child with the given id.
     */
    findChild(id) {
        const ret = this.tryFindChild(id);
        if (!ret) {
            throw new Error(`No child with id: '${id}'`);
        }
        return ret;
    }
    /**
     * Returns the child construct that has the id `Default` or `Resource"`.
     * This is usually the construct that provides the bulk of the underlying functionality.
     * Useful for modifications of the underlying construct that are not available at the higher levels.
     *
     * @throws if there is more than one child
     * @returns a construct or undefined if there is no default child
     */
    get defaultChild() {
        if (this._defaultChild !== undefined) {
            return this._defaultChild;
        }
        const resourceChild = this.tryFindChild('Resource');
        const defaultChild = this.tryFindChild('Default');
        if (resourceChild && defaultChild) {
            throw new Error(`Cannot determine default child for ${this.path}. There is both a child with id "Resource" and id "Default"`);
        }
        return defaultChild || resourceChild;
    }
    /**
     * Override the defaultChild property.
     *
     * This should only be used in the cases where the correct
     * default child is not named 'Resource' or 'Default' as it
     * should be.
     *
     * If you set this to undefined, the default behavior of finding
     * the child named 'Resource' or 'Default' will be used.
     */
    set defaultChild(value) {
        this._defaultChild = value;
    }
    /**
     * All direct children of this construct.
     */
    get children() {
        return Object.values(this._children);
    }
    /**
     * Return this construct and all of its children in the given order
     */
    findAll(order = ConstructOrder.PREORDER) {
        const ret = new Array();
        visit(this.host);
        return ret;
        function visit(c) {
            if (order === ConstructOrder.PREORDER) {
                ret.push(c);
            }
            for (const child of Node.of(c).children) {
                visit(child);
            }
            if (order === ConstructOrder.POSTORDER) {
                ret.push(c);
            }
        }
    }
    /**
     * This can be used to set contextual values.
     * Context must be set before any children are added, since children may consult context info during construction.
     * If the key already exists, it will be overridden.
     * @param key The context key
     * @param value The context value
     */
    setContext(key, value) {
        if (this.children.length > 0) {
            const names = this.children.map(c => Node.of(c).id);
            throw new Error('Cannot set context after children have been added: ' + names.join(','));
        }
        this._context[key] = value;
    }
    /**
     * Retrieves a value from tree context.
     *
     * Context is usually initialized at the root, but can be overridden at any point in the tree.
     *
     * @param key The context key
     * @returns The context value or `undefined` if there is no context value for thie key.
     */
    tryGetContext(key) {
        const value = this._context[key];
        if (value !== undefined) {
            return value;
        }
        return this.scope && Node.of(this.scope).tryGetContext(key);
    }
    /**
     * An immutable array of metadata objects associated with this construct.
     * This can be used, for example, to implement support for deprecation notices, source mapping, etc.
     */
    get metadata() {
        return [...this._metadata];
    }
    /**
     * Adds a metadata entry to this construct.
     * Entries are arbitrary values and will also include a stack trace to allow tracing back to
     * the code location for when the entry was added. It can be used, for example, to include source
     * mapping in CloudFormation templates to improve diagnostics.
     *
     * @param type a string denoting the type of metadata
     * @param data the value of the metadata (can be a Token). If null/undefined, metadata will not be added.
     * @param fromFunction a function under which to restrict the metadata entry's stack trace (defaults to this.addMetadata)
     */
    addMetadata(type, data, fromFunction) {
        if (data == null) {
            return;
        }
        const trace = this.tryGetContext(metadata_1.ConstructMetadata.DISABLE_STACK_TRACE_IN_METADATA)
            ? undefined
            : stack_trace_1.captureStackTrace(fromFunction || this.addMetadata);
        this._metadata.push({ type, data, trace });
    }
    /**
     * Adds a { "info": <message> } metadata entry to this construct.
     * The toolkit will display the info message when apps are synthesized.
     * @param message The info message.
     */
    addInfo(message) {
        this.addMetadata(metadata_1.ConstructMetadata.INFO_METADATA_KEY, message);
    }
    /**
     * Adds a { "warning": <message> } metadata entry to this construct.
     * The toolkit will display the warning when an app is synthesized, or fail
     * if run in --strict mode.
     * @param message The warning message.
     */
    addWarning(message) {
        this.addMetadata(metadata_1.ConstructMetadata.WARNING_METADATA_KEY, message);
    }
    /**
     * Adds an { "error": <message> } metadata entry to this construct.
     * The toolkit will fail synthesis when errors are reported.
     * @param message The error message.
     */
    addError(message) {
        this.addMetadata(metadata_1.ConstructMetadata.ERROR_METADATA_KEY, message);
    }
    /**
     * Applies the aspect to this Constructs node
     */
    applyAspect(aspect) {
        this._aspects.push(aspect);
        return;
    }
    /**
     * All parent scopes of this construct.
     *
     * @returns a list of parent scopes. The last element in the list will always
     * be the current construct and the first element will be the root of the
     * tree.
     */
    get scopes() {
        const ret = new Array();
        let curr = this.host;
        while (curr) {
            ret.unshift(curr);
            curr = Node.of(curr).scope;
        }
        return ret;
    }
    /**
     * Returns the root of the construct tree.
     * @returns The root of the construct tree.
     */
    get root() {
        return this.scopes[0];
    }
    /**
     * Returns true if this construct or the scopes in which it is defined are
     * locked.
     */
    get locked() {
        if (this._locked) {
            return true;
        }
        if (this.scope && Node.of(this.scope).locked) {
            return true;
        }
        return false;
    }
    /**
     * Add an ordering dependency on another Construct.
     *
     * All constructs in the dependency's scope will be deployed before any
     * construct in this construct's scope.
     */
    addDependency(...dependencies) {
        for (const dependency of dependencies) {
            this._dependencies.add(dependency);
        }
    }
    /**
     * Return all dependencies registered on this node or any of its children
     */
    get dependencies() {
        const found = new Map(); // Deduplication map
        const ret = new Array();
        for (const source of this.findAll()) {
            for (const dependable of Node.of(source)._dependencies) {
                for (const target of dependency_1.DependableTrait.get(dependable).dependencyRoots) {
                    let foundTargets = found.get(source);
                    if (!foundTargets) {
                        found.set(source, foundTargets = new Set());
                    }
                    if (!foundTargets.has(target)) {
                        ret.push({ source, target });
                        foundTargets.add(target);
                    }
                }
            }
        }
        return ret;
    }
    /**
     * Remove the child with the given name, if present.
     *
     * @returns Whether a child with the given name was deleted.
     * @experimental
     */
    tryRemoveChild(childName) {
        if (!(childName in this._children)) {
            return false;
        }
        delete this._children[childName];
        return true;
    }
    /**
   * Synthesizes a CloudAssembly from a construct tree.
   * @param options Synthesis options.
   */
    synthesize(options) {
        // the three holy phases of synthesis: prepare, validate and synthesize
        // prepare
        this.prepare();
        // validate
        const validate = options.skipValidation === undefined ? true : !options.skipValidation;
        if (validate) {
            const errors = this.validate();
            if (errors.length > 0) {
                const errorList = errors.map(e => `[${Node.of(e.source).path}] ${e.message}`).join('\n  ');
                throw new Error(`Validation failed with the following errors:\n  ${errorList}`);
            }
        }
        // synthesize (leaves first)
        for (const construct of this.findAll(ConstructOrder.POSTORDER)) {
            const node = Node.of(construct);
            try {
                node._lock();
                const ctx = {
                    ...options.sessionContext,
                    outdir: options.outdir
                };
                construct.onSynthesize(ctx); // "as any" is needed because we want to keep "synthesize" protected
            }
            finally {
                node._unlock();
            }
        }
    }
    /**
     * Invokes "prepare" on all constructs (depth-first, post-order) in the tree under `node`.
     */
    prepare() {
        // Aspects are applied root to leaf
        for (const construct of this.findAll(ConstructOrder.PREORDER)) {
            Node.of(construct).invokeAspects();
        }
        // since constructs can be added to the tree during invokeAspects, call findAll() to recreate the list.
        // use PREORDER.reverse() for backward compatability
        for (const construct of this.findAll(ConstructOrder.PREORDER).reverse()) {
            const cn = construct;
            if ('onPrepare' in cn) {
                if (typeof (cn.onPrepare) !== 'function') {
                    throw new Error(`expecting "onPrepare" to be a function`);
                }
                cn.onPrepare();
            }
        }
    }
    /**
     * Invokes "validate" on all constructs in the tree (depth-first, pre-order) and returns
     * the list of all errors. An empty list indicates that there are no errors.
     */
    validate() {
        let errors = new Array();
        for (const child of this.children) {
            errors = errors.concat(Node.of(child).validate());
        }
        const localErrors = this.host.onValidate(); // "as any" is needed because we want to keep "validate" protected
        return errors.concat(localErrors.map(msg => ({ source: this.host, message: msg })));
    }
    /**
     * Locks this construct from allowing more children to be added. After this
     * call, no more children can be added to this construct or to any children.
     * @internal
     */
    _lock() {
        this._locked = true;
    }
    /**
     * Unlocks this costruct and allows mutations (adding children).
     * @internal
     */
    _unlock() {
        this._locked = false;
    }
    /**
     * Adds a child construct to this node.
     *
     * @param child The child construct
     * @param childName The type name of the child construct.
     * @returns The resolved path part name of the child
     */
    addChild(child, childName) {
        if (this.locked) {
            // special error if root is locked
            if (!this.path) {
                throw new Error('Cannot add children during synthesis');
            }
            throw new Error(`Cannot add children to "${this.path}" during synthesis`);
        }
        if (childName in this._children) {
            const name = this.id || '';
            const typeName = this.host.constructor.name;
            throw new Error(`There is already a Construct with name '${childName}' in ${typeName}${name.length > 0 ? ' [' + name + ']' : ''}`);
        }
        this._children[childName] = child;
    }
    /**
     * Triggers each aspect to invoke visit
     */
    invokeAspects() {
        const descendants = this.findAll();
        for (const aspect of this._aspects) {
            if (this.invokedAspects.includes(aspect)) {
                continue;
            }
            descendants.forEach(member => aspect.visit(member));
            this.invokedAspects.push(aspect);
        }
    }
}
exports.Node = Node;
/**
 * Separator used to delimit construct path components.
 */
Node.PATH_SEP = '/';
/**
 * Represents the building block of the construct graph.
 *
 * All constructs besides the root construct must be created within the scope of
 * another construct.
 */
class Construct {
    /**
     * Creates a new construct node.
     *
     * @param scope The scope in which to define this construct
     * @param id The scoped construct ID. Must be unique amongst siblings. If
     * the ID includes a path separator (`/`), then it will be replaced by double
     * dash `--`.
     * @param options Options
     */
    constructor(scope, id, options = {}) {
        var _a;
        // attach the construct to the construct tree by creating a node
        const nodeFactory = (_a = options.nodeFactory) !== null && _a !== void 0 ? _a : { createNode: (host, scope, id) => new Node(host, scope, id) };
        Object.defineProperty(this, CONSTRUCT_NODE_PROPERTY_SYMBOL, {
            value: nodeFactory.createNode(this, scope, id),
            enumerable: false,
            configurable: false,
        });
        // implement IDependable privately
        dependency_1.DependableTrait.implement(this, {
            dependencyRoots: [this]
        });
    }
    /**
     * Returns a string representation of this construct.
     */
    toString() {
        return Node.of(this).path || '<root>';
    }
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @returns An array of validation error messages, or an empty array if there the construct is valid.
     */
    onValidate() {
        return [];
    }
    /**
     * Perform final modifications before synthesis
     *
     * This method can be implemented by derived constructs in order to perform
     * final changes before synthesis. prepare() will be called after child
     * constructs have been prepared.
     *
     * This is an advanced framework feature. Only use this if you
     * understand the implications.
     */
    onPrepare() {
        return;
    }
    /**
     * Allows this construct to emit artifacts into the cloud assembly during synthesis.
     *
     * This method is usually implemented by framework-level constructs such as `Stack` and `Asset`
     * as they participate in synthesizing the cloud assembly.
     *
     * @param session The synthesis session.
     */
    onSynthesize(session) {
        ignore(session);
    }
}
exports.Construct = Construct;
/**
 * In what order to return constructs
 */
var ConstructOrder;
(function (ConstructOrder) {
    /**
     * Depth-first, pre-order
     */
    ConstructOrder[ConstructOrder["PREORDER"] = 0] = "PREORDER";
    /**
     * Depth-first, post-order (leaf nodes first)
     */
    ConstructOrder[ConstructOrder["POSTORDER"] = 1] = "POSTORDER";
})(ConstructOrder = exports.ConstructOrder || (exports.ConstructOrder = {}));
function ignore(_x) {
    return;
}
// Import this _after_ everything else to help node work the classes out in the correct order...
const PATH_SEP_REGEX = new RegExp(`${Node.PATH_SEP}`, 'g');
/**
 * Return a sanitized version of an arbitrary string, so it can be used as an ID
 */
function sanitizeId(id) {
    // Escape path seps as double dashes
    return id.replace(PATH_SEP_REGEX, '--');
}
//# sourceMappingURL=data:application/json;base64,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