"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const lib_1 = require("../lib");
const util_1 = require("./util");
// tslint:disable:variable-name
// tslint:disable:max-line-length
test('the "Root" construct is a special construct which can be used as the root of the tree', () => {
    const root = new util_1.App();
    const node = lib_1.Node.of(root);
    expect(node.id).toBe('');
    expect(node.scope).toBeUndefined();
    expect(node.children.length).toBe(0);
});
test('constructs cannot be created with an empty name unless they are root', () => {
    const root = new util_1.App();
    expect(() => new lib_1.Construct(root, '')).toThrow(/Only root constructs may have an empty name/);
});
test('construct.name returns the name of the construct', () => {
    const t = createTree();
    expect(lib_1.Node.of(t.child1).id).toBe('Child1');
    expect(lib_1.Node.of(t.child2).id).toBe('Child2');
    expect(lib_1.Node.of(t.child1_1).id).toBe('Child11');
    expect(lib_1.Node.of(t.child1_2).id).toBe('Child12');
    expect(lib_1.Node.of(t.child1_1_1).id).toBe('Child111');
    expect(lib_1.Node.of(t.child2_1).id).toBe('Child21');
});
test('construct id can use any character except the path separator', () => {
    const root = new util_1.App();
    expect(() => new lib_1.Construct(root, 'valid')).not.toThrow();
    expect(() => new lib_1.Construct(root, 'ValiD')).not.toThrow();
    expect(() => new lib_1.Construct(root, 'Va123lid')).not.toThrow();
    expect(() => new lib_1.Construct(root, 'v')).not.toThrow();
    expect(() => new lib_1.Construct(root, '  invalid')).not.toThrow();
    expect(() => new lib_1.Construct(root, 'invalid   ')).not.toThrow();
    expect(() => new lib_1.Construct(root, '123invalid')).not.toThrow();
    expect(() => new lib_1.Construct(root, 'in valid')).not.toThrow();
    expect(() => new lib_1.Construct(root, 'in_Valid')).not.toThrow();
    expect(() => new lib_1.Construct(root, 'in-Valid')).not.toThrow();
    expect(() => new lib_1.Construct(root, 'in\\Valid')).not.toThrow();
    expect(() => new lib_1.Construct(root, 'in.Valid')).not.toThrow();
});
test('if construct id contains path seperators, they will be replaced by double-dash', () => {
    const root = new util_1.App();
    const c = new lib_1.Construct(root, 'Boom/Boom/Bam');
    expect(lib_1.Node.of(c).id).toBe('Boom--Boom--Bam');
});
test('if "undefined" is forcefully used as an "id", it will be treated as an empty string', () => {
    const c = new lib_1.Construct(undefined, undefined);
    expect(lib_1.Node.of(c).id).toBe('');
});
test('construct.uniqueId returns a tree-unique alphanumeric id of this construct', () => {
    const root = new util_1.App();
    const child1 = new lib_1.Construct(root, 'This is the first child');
    const child2 = new lib_1.Construct(child1, 'Second level');
    const c1 = new lib_1.Construct(child2, 'My construct');
    const c2 = new lib_1.Construct(child1, 'My construct');
    expect(lib_1.Node.of(c1).path).toBe('This is the first child/Second level/My construct');
    expect(lib_1.Node.of(c2).path).toBe('This is the first child/My construct');
    expect(lib_1.Node.of(c1).uniqueId).toBe('ThisisthefirstchildSecondlevelMyconstruct202131E0');
    expect(lib_1.Node.of(c2).uniqueId).toBe('ThisisthefirstchildMyconstruct8C288DF9');
});
test('cannot calculate uniqueId if the construct path is ["Default"]', () => {
    const root = new util_1.App();
    const c = new lib_1.Construct(root, 'Default');
    expect(() => lib_1.Node.of(c).uniqueId).toThrow(/Unable to calculate a unique id for an empty set of components/);
});
test('construct.getChildren() returns an array of all children', () => {
    const root = new util_1.App();
    const child = new lib_1.Construct(root, 'Child1');
    new lib_1.Construct(root, 'Child2');
    expect(lib_1.Node.of(child).children.length).toBe(0);
    expect(lib_1.Node.of(root).children.length).toBe(2);
});
test('construct.findChild(name) can be used to retrieve a child from a parent', () => {
    const root = new util_1.App();
    const child = new lib_1.Construct(root, 'Contruct');
    expect(lib_1.Node.of(root).tryFindChild(lib_1.Node.of(child).id)).toBe(child);
    expect(lib_1.Node.of(root).tryFindChild('NotFound')).toBeUndefined();
});
test('construct.getChild(name) can be used to retrieve a child from a parent', () => {
    const root = new util_1.App();
    const child = new lib_1.Construct(root, 'Contruct');
    expect(lib_1.Node.of(root).findChild(lib_1.Node.of(child).id)).toBe(child);
    expect(() => lib_1.Node.of(root).findChild('NotFound')).toThrow(/No child with id: 'NotFound'/);
});
test('construct.getContext(key) can be used to read a value from context defined at the root level', () => {
    const context = {
        ctx1: 12,
        ctx2: 'hello'
    };
    const t = createTree(context);
    expect(lib_1.Node.of(t.child1_2).tryGetContext('ctx1')).toBe(12);
    expect(lib_1.Node.of(t.child1_1_1).tryGetContext('ctx2')).toBe('hello');
});
// tslint:disable-next-line:max-line-length
test('construct.setContext(k,v) sets context at some level and construct.getContext(key) will return the lowermost value defined in the stack', () => {
    const root = new util_1.App();
    const highChild = new lib_1.Construct(root, 'highChild');
    lib_1.Node.of(highChild).setContext('c1', 'root');
    lib_1.Node.of(highChild).setContext('c2', 'root');
    const child1 = new lib_1.Construct(highChild, 'child1');
    lib_1.Node.of(child1).setContext('c2', 'child1');
    lib_1.Node.of(child1).setContext('c3', 'child1');
    const child2 = new lib_1.Construct(highChild, 'child2');
    const child3 = new lib_1.Construct(child1, 'child1child1');
    lib_1.Node.of(child3).setContext('c1', 'child3');
    lib_1.Node.of(child3).setContext('c4', 'child3');
    expect(lib_1.Node.of(highChild).tryGetContext('c1')).toBe('root');
    expect(lib_1.Node.of(highChild).tryGetContext('c2')).toBe('root');
    expect(lib_1.Node.of(highChild).tryGetContext('c3')).toBeUndefined();
    expect(lib_1.Node.of(child1).tryGetContext('c1')).toBe('root');
    expect(lib_1.Node.of(child1).tryGetContext('c2')).toBe('child1');
    expect(lib_1.Node.of(child1).tryGetContext('c3')).toBe('child1');
    expect(lib_1.Node.of(child2).tryGetContext('c1')).toBe('root');
    expect(lib_1.Node.of(child2).tryGetContext('c2')).toBe('root');
    expect(lib_1.Node.of(child2).tryGetContext('c3')).toBeUndefined();
    expect(lib_1.Node.of(child3).tryGetContext('c1')).toBe('child3');
    expect(lib_1.Node.of(child3).tryGetContext('c2')).toBe('child1');
    expect(lib_1.Node.of(child3).tryGetContext('c3')).toBe('child1');
    expect(lib_1.Node.of(child3).tryGetContext('c4')).toBe('child3');
});
test('construct.setContext(key, value) can only be called before adding any children', () => {
    const root = new util_1.App();
    new lib_1.Construct(root, 'child1');
    expect(() => lib_1.Node.of(root).setContext('k', 'v')).toThrow(/Cannot set context after children have been added: child1/);
});
test('construct.pathParts returns an array of strings of all names from root to node', () => {
    const tree = createTree();
    expect(lib_1.Node.of(tree.root).path).toBe('');
    expect(lib_1.Node.of(tree.child1_1_1).path).toBe('HighChild/Child1/Child11/Child111');
    expect(lib_1.Node.of(tree.child2).path).toBe('HighChild/Child2');
});
test('if a root construct has a name, it should be included in the path', () => {
    const tree = createTree({});
    expect(lib_1.Node.of(tree.root).path).toBe('');
    expect(lib_1.Node.of(tree.child1_1_1).path).toBe('HighChild/Child1/Child11/Child111');
});
test('construct can not be created with the name of a sibling', () => {
    const root = new util_1.App();
    // WHEN
    new lib_1.Construct(root, 'SameName');
    // THEN: They have different paths
    expect(() => new lib_1.Construct(root, 'SameName')).toThrow(/There is already a Construct with name 'SameName' in App/);
    // WHEN
    const c0 = new lib_1.Construct(root, 'c0');
    new lib_1.Construct(c0, 'SameName');
    // THEN: They have different paths
    expect(() => new lib_1.Construct(c0, 'SameName')).toThrow(/There is already a Construct with name 'SameName' in Construct \[c0\]/);
});
test('addMetadata(type, data) can be used to attach metadata to constructs', () => {
    var _a;
    const root = new util_1.App();
    const con = new lib_1.Construct(root, 'MyConstruct');
    expect(lib_1.Node.of(con).metadata).toEqual([]);
    const node = lib_1.Node.of(con);
    (function FIND_ME() {
        node.addMetadata('key', 'value');
        node.addMetadata('number', 103);
        node.addMetadata('array', [123, 456]);
    })();
    expect(node.metadata[0].type).toBe('key');
    expect(node.metadata[0].data).toBe('value');
    expect(node.metadata[1].data).toBe(103);
    expect(node.metadata[2].data).toEqual([123, 456]);
    expect((_a = node.metadata[0].trace) === null || _a === void 0 ? void 0 : _a[0]).toContain('FIND_ME');
});
test('addMetadata(type, undefined/null) is ignored', () => {
    const root = new util_1.App();
    const con = new lib_1.Construct(root, 'Foo');
    const node = lib_1.Node.of(con);
    node.addMetadata('Null', null);
    node.addMetadata('Undefined', undefined);
    node.addMetadata('True', true);
    node.addMetadata('False', false);
    node.addMetadata('Empty', '');
    const exists = (key) => node.metadata.find(x => x.type === key);
    expect(exists('Null')).toBeFalsy();
    expect(exists('Undefined')).toBeFalsy();
    expect(exists('True')).toBeTruthy();
    expect(exists('False')).toBeTruthy();
    expect(exists('Empty')).toBeTruthy();
});
test('addWarning(message) can be used to add a "WARNING" message entry to the construct', () => {
    var _a;
    const root = new util_1.App();
    const con = new lib_1.Construct(root, 'MyConstruct');
    const node = lib_1.Node.of(con);
    node.addWarning('This construct is deprecated, use the other one instead');
    expect(node.metadata[0].type).toBe(lib_1.ConstructMetadata.WARNING_METADATA_KEY);
    expect(node.metadata[0].data).toBe('This construct is deprecated, use the other one instead');
    expect((_a = node.metadata[0].trace) === null || _a === void 0 ? void 0 : _a.length).toBeGreaterThan(0);
});
test('addError(message) can be used to add a "ERROR" message entry to the construct', () => {
    var _a;
    const root = new util_1.App();
    const con = new lib_1.Construct(root, 'MyConstruct');
    const node = lib_1.Node.of(con);
    node.addError('Stop!');
    expect(node.metadata[0].type).toBe(lib_1.ConstructMetadata.ERROR_METADATA_KEY);
    expect(node.metadata[0].data).toBe('Stop!');
    expect((_a = node.metadata[0].trace) === null || _a === void 0 ? void 0 : _a.length).toBeGreaterThan(0);
});
test('addInfo(message) can be used to add an "INFO" message entry to the construct', () => {
    var _a;
    const root = new util_1.App();
    const con = new lib_1.Construct(root, 'MyConstruct');
    const node = lib_1.Node.of(con);
    node.addInfo('Hey there, how do you do?');
    expect(node.metadata[0].type).toBe(lib_1.ConstructMetadata.INFO_METADATA_KEY);
    expect(node.metadata[0].data).toBe('Hey there, how do you do?');
    expect((_a = node.metadata[0].trace) === null || _a === void 0 ? void 0 : _a.length).toBeGreaterThan(0);
});
test('multiple children of the same type, with explicit names are welcome', () => {
    const root = new util_1.App();
    new MyBeautifulConstruct(root, 'mbc1');
    new MyBeautifulConstruct(root, 'mbc2');
    new MyBeautifulConstruct(root, 'mbc3');
    new MyBeautifulConstruct(root, 'mbc4');
    expect(lib_1.Node.of(root).children.length).toBeGreaterThanOrEqual(4);
});
// tslint:disable-next-line:max-line-length
test('construct.onValidate() can be implemented to perform validation, node.validate() will return all errors from the subtree (DFS)', () => {
    class MyConstruct extends lib_1.Construct {
        onValidate() {
            return ['my-error1', 'my-error2'];
        }
    }
    class YourConstruct extends lib_1.Construct {
        onValidate() {
            return ['your-error1'];
        }
    }
    class TheirConstruct extends lib_1.Construct {
        constructor(scope, id) {
            super(scope, id);
            new YourConstruct(this, 'YourConstruct');
        }
        onValidate() {
            return ['their-error'];
        }
    }
    class TestStack extends util_1.App {
        constructor() {
            super();
            new MyConstruct(this, 'MyConstruct');
            new TheirConstruct(this, 'TheirConstruct');
        }
        onValidate() {
            return ['stack-error'];
        }
    }
    const stack = new TestStack();
    const errors = lib_1.Node.of(stack).validate()
        .map((v) => ({ path: lib_1.Node.of(v.source).path, message: v.message }));
    // validate DFS
    expect(errors).toEqual([
        { path: 'MyConstruct', message: 'my-error1' },
        { path: 'MyConstruct', message: 'my-error2' },
        { path: 'TheirConstruct/YourConstruct', message: 'your-error1' },
        { path: 'TheirConstruct', message: 'their-error' },
        { path: '', message: 'stack-error' }
    ]);
});
test('construct.lock() protects against adding children anywhere under this construct (direct or indirect)', () => {
    class LockableConstruct extends lib_1.Construct {
        lockMe() {
            lib_1.Node.of(this)._lock();
        }
        unlockMe() {
            lib_1.Node.of(this)._unlock();
        }
    }
    const stack = new util_1.App();
    const c0a = new LockableConstruct(stack, 'c0a');
    const c0b = new lib_1.Construct(stack, 'c0b');
    const c1a = new lib_1.Construct(c0a, 'c1a');
    const c1b = new lib_1.Construct(c0a, 'c1b');
    c0a.lockMe();
    // now we should still be able to add children to c0b, but not to c0a or any its children
    new lib_1.Construct(c0b, 'c1a');
    expect(() => new lib_1.Construct(c0a, 'fail1')).toThrow(/Cannot add children to "c0a" during synthesis/);
    expect(() => new lib_1.Construct(c1a, 'fail2')).toThrow(/Cannot add children to "c0a\/c1a" during synthesis/);
    expect(() => new lib_1.Construct(c1b, 'fail3')).toThrow(/Cannot add children to "c0a\/c1b" during synthesis/);
    c0a.unlockMe();
    new lib_1.Construct(c0a, 'c0aZ');
    new lib_1.Construct(c1a, 'c1aZ');
    new lib_1.Construct(c1b, 'c1bZ');
});
test('findAll returns a list of all children in either DFS or BFS', () => {
    // GIVEN
    const c1 = new lib_1.Construct(undefined, '1');
    const c2 = new lib_1.Construct(c1, '2');
    new lib_1.Construct(c1, '3');
    new lib_1.Construct(c2, '4');
    new lib_1.Construct(c2, '5');
    // THEN
    const node = lib_1.Node.of(c1);
    expect(node.findAll().map(x => lib_1.Node.of(x).id)).toEqual(lib_1.Node.of(c1).findAll(lib_1.ConstructOrder.PREORDER).map(x => lib_1.Node.of(x).id)); // default is PreOrder
    expect(node.findAll(lib_1.ConstructOrder.PREORDER).map(x => lib_1.Node.of(x).id)).toEqual(['1', '2', '4', '5', '3']);
    expect(node.findAll(lib_1.ConstructOrder.POSTORDER).map(x => lib_1.Node.of(x).id)).toEqual(['4', '5', '2', '3', '1']);
});
test('ancestors returns a list of parents up to root', () => {
    const { child1_1_1 } = createTree();
    expect(lib_1.Node.of(child1_1_1).scopes.map(x => lib_1.Node.of(x).id)).toEqual(['', 'HighChild', 'Child1', 'Child11', 'Child111']);
});
test('"root" returns the root construct', () => {
    const { child1, child2, child1_1_1, root } = createTree();
    expect(lib_1.Node.of(child1).root).toBe(root);
    expect(lib_1.Node.of(child2).root).toBe(root);
    expect(lib_1.Node.of(child1_1_1).root).toBe(root);
});
describe('defaultChild', () => {
    test('returns the child with id "Resource"', () => {
        const root = new util_1.App();
        new lib_1.Construct(root, 'child1');
        const defaultChild = new lib_1.Construct(root, 'Resource');
        new lib_1.Construct(root, 'child2');
        expect(lib_1.Node.of(root).defaultChild).toBe(defaultChild);
    });
    test('returns the child with id "Default"', () => {
        const root = new util_1.App();
        new lib_1.Construct(root, 'child1');
        const defaultChild = new lib_1.Construct(root, 'Default');
        new lib_1.Construct(root, 'child2');
        expect(lib_1.Node.of(root).defaultChild).toBe(defaultChild);
    });
    test('can override defaultChild', () => {
        const root = new util_1.App();
        new lib_1.Construct(root, 'Resource');
        const defaultChild = new lib_1.Construct(root, 'OtherResource');
        lib_1.Node.of(root).defaultChild = defaultChild;
        expect(lib_1.Node.of(root).defaultChild).toBe(defaultChild);
    });
    test('returns "undefined" if there is no default', () => {
        const root = new util_1.App();
        new lib_1.Construct(root, 'child1');
        new lib_1.Construct(root, 'child2');
        expect(lib_1.Node.of(root).defaultChild).toBeUndefined();
    });
    test('fails if there are both "Resource" and "Default"', () => {
        const root = new util_1.App();
        new lib_1.Construct(root, 'child1');
        new lib_1.Construct(root, 'Default');
        new lib_1.Construct(root, 'child2');
        new lib_1.Construct(root, 'Resource');
        expect(() => lib_1.Node.of(root).defaultChild)
            .toThrow(/Cannot determine default child for . There is both a child with id "Resource" and id "Default"/);
    });
    test('constructs created in an Aspect are prepared', () => {
        const root = new util_1.App();
        const construct = new lib_1.Construct(root, 'Resource');
        lib_1.Node.of(construct).applyAspect(new AddConstructAspect(construct));
        lib_1.Node.of(root).prepare();
        // THEN
        const addedConstruct = lib_1.Node.of(root).findAll(lib_1.ConstructOrder.PREORDER)
            .find(child => lib_1.Node.of(child).id === `AspectAdded-${lib_1.Node.of(construct).id}`);
        expect(addedConstruct.status).toBe('Prepared');
    });
});
describe('construct prepare', () => {
    it('created constructs are prepared', () => {
        const root = new util_1.App();
        const construct01 = new MyAlmostBeautifulConstruct(root, 'Resource01');
        const construct02 = new MyAlmostBeautifulConstruct(root, 'Resource02');
        lib_1.Node.of(root).prepare();
        // THEN
        expect(construct01.status).toEqual('Prepared');
        expect(construct02.status).toEqual('Prepared');
    });
    it('only constructs with onPrepare function are prepared', () => {
        const root = new util_1.App();
        const construct01 = new MyAlmostBeautifulConstruct(root, 'Resource01');
        const construct02 = new MyMissingPrepareConstruct(root, 'Resource02');
        lib_1.Node.of(root).prepare();
        // THEN
        expect(construct01.status).toEqual('Prepared');
        expect(construct02.status).not.toEqual('Prepared');
    });
    it('only constructs with onPrepare function are prepared', () => {
        const root = new util_1.App();
        const construct01 = new MyAlmostBeautifulConstruct(root, 'Resource01');
        // we try to force the error
        construct01.onPrepare = undefined;
        expect(() => lib_1.Node.of(root).prepare())
            .toThrow(/expecting "onPrepare" to be a function/);
    });
});
function createTree(context) {
    const root = new util_1.App();
    const highChild = new lib_1.Construct(root, 'HighChild');
    if (context) {
        Object.keys(context).forEach(key => lib_1.Node.of(highChild).setContext(key, context[key]));
    }
    const child1 = new lib_1.Construct(highChild, 'Child1');
    const child2 = new lib_1.Construct(highChild, 'Child2');
    const child1_1 = new lib_1.Construct(child1, 'Child11');
    const child1_2 = new lib_1.Construct(child1, 'Child12');
    const child1_1_1 = new lib_1.Construct(child1_1, 'Child111');
    const child2_1 = new lib_1.Construct(child2, 'Child21');
    return {
        root, child1, child2, child1_1, child1_2, child1_1_1, child2_1
    };
}
class MyMissingPrepareConstruct extends lib_1.Construct {
    constructor(scope, id) {
        super(scope, id);
        this.status = 'PrePrepared';
    }
}
class MyBeautifulConstruct extends lib_1.Construct {
    constructor(scope, id) {
        super(scope, id);
    }
}
class MyAlmostBeautifulConstruct extends lib_1.Construct {
    constructor(scope, id) {
        super(scope, id);
        this.status = 'PrePrepared';
    }
    onPrepare() {
        this.status = 'Prepared';
    }
}
class AddConstructAspect {
    constructor(scope) {
        this.scope = scope;
    }
    visit(node) {
        new MyAlmostBeautifulConstruct(this.scope, `AspectAdded-${lib_1.Node.of(node).id}`);
    }
}
//# sourceMappingURL=data:application/json;base64,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