# Experiment Framework with Docker
This repository provides a reproducible Docker-based environment for running experiments.  
It ensures that datasets are properly downloaded/prepared and that results are stored locally.

---

## 📂 Project Structure

```
01_Apr_02_compare_recon_method/
├── conf/                  # Configurations 
│   └── setup.sh           # Script to create virtual environment & install deps
│   └── requirements.txt   # Dependencies to install
├── data/                  # Datasets (mounted from host)
├── images/                # Container definition files (Dockerfile, experiment.def)
│   ├── scripts/           # Container utility scripts
│       └── entrypoint.sh 
│   ├── Dockerfile
│   └── experiment.def
├── src/                   # Source code (Python modules)
├── results/               # Experiment outputs (mounted from host)
├── notebook/              # Jupyter notebooks
├── scripts/               # Utility scripts
│   ├── download_blend_mean_temperature.sh
│   ├── prepare_ecad_observations.py
```

---

## 🐳 Docker Setup

### 1. Build the Docker Image

From inside the `images/` directory:

```bash
cd 01_Apr_02_compare_recon_method
docker build -t my-experiments -f images/Dockerfile .
```

This creates an image named `my-experiments`.

---

### 2. Run the Container

To run experiments while keeping **data** and **results** synced with your host:

```bash
docker run --rm \
  -v $(pwd)/../data:/app/data \
  -v $(pwd)/../results:/app/results \
  my-experiments
```

- `-v $(pwd)/../data:/app/data` → Mounts host `01_Apr_02_compare_recon_method/data` into container `/app/data`  
- `-v $(pwd)/../results:/app/results` → Mounts host `01_Apr_02_compare_recon_method/results` into container `/app/results`  
- `--rm` → Automatically removes the container after execution  

---

# 🛰️ Using Apptainer (Singularity)

### 1. Build the Apptainer Image

From inside the `images/` directory:

```bash
cd 01_Apr_02_compare_recon_method/images
apptainer build experiment.sif experiment.def
```

If you don’t have root privileges, you may need:
```bash
apptainer build --fakeroot experiment.sif experiment.def
```

---

### 2. Run the Container

```bash
apptainer run \
  --bind ../data:/app/data \
  --bind ../results:/app/results \
  experiment.sif
```

- `--bind ../data:/app/data` → Mounts host `01_Apr_02_compare_recon_method/data`  
- `--bind ../results:/app/results` → Mounts host `01_Apr_02_compare_recon_method/results`  
- Results will appear in your host `01_Apr_02_compare_recon_method/results`.

---

## ⚙️ What Happens Inside the Container

When the container starts (`CMD` in Dockerfile):

1. **Download data** (if not already present)  
   ```bash
   scripts/download_blend_mean_temperature.sh
   ```
   - Downloads data into `/app/data` (mapped to host `01_Apr_02_compare_recon_method/data`).

2. **Prepare dataset**  
   ```bash
   python scripts/prepare_ecad_observations.py
   ```
   - Processes raw data and stores ready-to-use files in `/app/data`.

3. **Run experiments**  
   ```bash
   bash scripts/run_experiments.sh
   ```
   - Executes experiments and saves outputs in `/app/results` (mapped to host `01_Apr_02_compare_recon_method/results`).

---

## 📌 Virtual Environment

- A Python virtual environment is created in `/app/conf/exp1` during image build.  
- It’s automatically used for all Python commands via environment variables:
  ```dockerfile
  ENV VIRTUAL_ENV=/app/conf/exp1
  ENV PATH="$VIRTUAL_ENV/bin:$PATH"
  ```
- You do **not** need to run `source activate`; `python` and `pip` already refer to the venv.

---

## 📊 Viewing Results

After the run, check results on your host machine:

```bash
ls 01_Apr_02_compare_recon_method/results
```

Outputs generated by the experiments will appear here.

---

## 📝 License

The code and experiments are published under MIT license.