import numpy as np
from sklearn.metrics.pairwise import rbf_kernel


def predict_binary_classifier(X):
    ''''
    X: num test data by 13 features
    '''

    # Get params
    params = binary_classifier()

    # compute kernel for all data points
    K = rbf_kernel(params['support_vec'], X, gamma=params['gamma'])

    # Prediction = sign((sum_{i=1}^n y_i*alpha*K(x_i,x)) + rho)
    predictions = np.zeros(X.shape[0])
    for i in range(X.shape[0]):
        predictions[i] = np.sign(np.sum(params['dual_coef']*K[:, i]) + params['intercept'])

    return predictions


def predict_multiclass_classifier(X):
    '''
        X: num test data by 10 features
    '''
    # Get params
    params = multiclass_classifier()

    K = rbf_kernel(params['support_vec'], X, gamma=params['gamma'])

    # define the start and end index for support vectors for each class
    nv = params['num_support_vec']
    start = [sum(nv[:i]) for i in range(len(nv))]
    end = [start[i] + nv[i] for i in range(len(nv))]

    # calculate: sum(a_p * k(x_p, x)) between every 2 classes
    dual_coef = params['dual_coef'].T
    predictions_0_1 = np.zeros(X.shape[0])
    for i in range(X.shape[0]):
        temp_prediction = np.sum(dual_coef[start[0]:end[0], 0] * K[start[0]:end[0], i]) + \
                          np.sum(dual_coef[start[1]:end[1], 0] * K[start[1]:end[1], i]) + params['intercept'][0]
        predictions_0_1[i] = 0 if temp_prediction > 0 else 1

    predictions_0_2 = np.zeros(X.shape[0])
    for i in range(X.shape[0]):
        temp_prediction = np.sum(dual_coef[start[0]:end[0], 1] * K[start[0]:end[0], i]) + \
                          np.sum(dual_coef[start[2]:end[2], 0] * K[start[2]:end[2], i]) + params['intercept'][1]
        predictions_0_2[i] = 0 if temp_prediction > 0 else 2

    predictions_1_2 = np.zeros(X.shape[0])
    for i in range(X.shape[0]):
        temp_prediction = np.sum(dual_coef[start[1]:end[1], 1] * K[start[1]:end[1], i]) + \
                          np.sum(dual_coef[start[2]:end[2], 1] * K[start[2]:end[2], i]) + params['intercept'][2]
        predictions_1_2[i] = 1 if temp_prediction > 0 else 2

    decision_function = np.vstack([predictions_0_1, predictions_0_2, predictions_1_2]).T

    # Majority Vote to find the best class
    predictions = np.zeros(X.shape[0])
    for i in range(X.shape[0]):
        lst = decision_function[i,:].tolist()
        predictions[i] = max(set(lst), key=lst.count)-1

    return predictions



def binary_classifier():
    gamma = 0.1

    # dual coef = y_i*alpha_i
    dual_coef = np.array([[-1.12775599e+02,  -1.00000000e+03,  -1.00000000e+03,
                           -1.00000000e+03,  -1.00000000e+03,  -1.00000000e+03,
                           -1.00000000e+03,  -1.00000000e+03,  -1.00000000e+03,
                           -1.00000000e+03,  -1.00000000e+03,  -1.00000000e+03,
                           -4.65947457e+02,  -1.00000000e+03,  -1.00000000e+03,
                           -1.00000000e+03,  -1.17935400e+02,  -1.00000000e+03,
                           -1.00000000e+03,  -1.00000000e+03,  -1.00000000e+03,
                           -1.00000000e+03,  -1.00000000e+03,  -1.00000000e+03,
                           -1.00000000e+03,  -2.92534132e+02,  -1.00000000e+03,
                           -1.00000000e+03,  -3.69965631e+01,  -1.00000000e+03,
                           -1.00000000e+03,  -1.00000000e+03,  -1.00000000e+03,
                           -1.00000000e+03,  -1.00000000e+03,  -1.00000000e+03,
                           -1.00000000e+03,  -1.00000000e+03,   1.00000000e+03,
                           1.00000000e+03,   1.00000000e+03,   1.00000000e+03,
                           7.92366387e+02,   3.00553142e+02,   2.22950860e-01,
                           1.00000000e+03,   1.00000000e+03,   5.58636056e+02,
                           1.21751544e+02,   1.00000000e+03,   1.00000000e+03,
                           2.61920652e+00,   9.96570403e+02,   1.00000000e+03,
                           1.00000000e+03,   1.00000000e+03,   1.00000000e+03,
                           1.00000000e+03,   1.00000000e+03,   1.02270060e+02,
                           5.41288840e+01,   1.91650287e+02,   1.00000000e+03,
                           1.00000000e+03,   1.00000000e+03,   1.00000000e+03,
                           1.00000000e+03,   2.45152637e+02,   7.53766346e+02,
                           1.00000000e+03,   1.00000000e+03,   3.63211198e+00,
                           1.00000000e+03,   3.31675798e+01,   5.64620367e+02,
                           1.00000000e+03,   1.00000000e+03,   1.00000000e+03,
                           2.66900636e+02,   1.00000000e+03,   6.54763900e+02,
                           3.38216549e+02,   6.86434772e+01,   2.78998678e+02,
                           6.97557950e+02,   1.00000000e+03]])

    # intercept = rho
    intercept = np.array([-2.63232929])

    # support vectors = x_i
    support_vec = np.array([[0.02809756, 0.0455, 0.025, 0.00866667, 0.03799132, -0.00799413, 0.01061208, 0.016263, 0.00671743, 0.00572262, 0.00578504, 0.00542415, 0.00318195],
                            [0.00060976, 0.0035, 0.007, 0.00087179, 0.00024191, -0.0005069, 0.0005069, 0.0070711, 0.00306413, 0.0031833, 0.0107827, 0.0066959, 0.0022981],
                            [3.49731707, 0.092, 0.054, 0.01923077, 3.53815367, -0.02236652, 0.02659884, 0.062225, 0.0316782, 0.01818914, 0.06607571, 0.03342241, 0.099702],
                            [2.52643902, 0.058, 0.055, 0.0114359, 2.54031008, -0.01070662, 0.01296803, 0.043134, 0.01649923, 0.01579683, 0.03326171, 0.05004163, 0.013965],
                            [0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, -2.74622599e-18, -2.42947453e-17, 3.36047450e-17, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00],
                            [3.89758537, 0.167, 0.27, 0.06717949, 3.87923565, -0.04130143, 0.05403825, 0.047376, 0.0328098, 0.01255584, 0.03676955, 0.14237773, 0.11031],
                            [0.93326829, 0.0855, 0.106, 0.01169231, 0.92669874, -0.02740927, 0.02740927, 0.043841, 0.01131377, 0.01595008, 0.0231871, 0.02414775, 0.0139655],
                            [4.64253659, 0.106, 0.13, 0.03661538, 4.63806066, -0.03168223, 0.03168223, 0.10182, 0.0559785, 0.03369301, 0.06341563, 0.08583294, 0.0251025],
                            [0.29312195, 0.028, 0.039, 0.00682051, 0.28575076, -0.00648365, 0.00648365, 0.0056569, 0.00367694, 0.00126494, 0.00364005, 0.01814984, 0.006364],
                            [3.08187805, 0.0615, 0.123, 0.03435897, 3.11862292, -0.02260403, 0.02260403, 0.053033, 0.0397394, 0.01570345, 0.0338851, 0.10069204, 0.16652],
                            [2.43902439e-05, 5.00000000e-04, 1.00000000e-03, 1.02564103e-04, 2.43769719e-05, -7.19856842e-05, 7.19856842e-05, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00],
                            [0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, -4.05052739e-10, -2.77557303e-09, 5.77955577e-09, 7.07110000e-04, 1.17851667e-04, 2.88676449e-04, 2.04124145e-04, 1.44336183e-04, 0.00000000e+00],
                            [0.83290244, 0.099, 0.172, 0.02610256, 0.82408369, -0.0168393, 0.0168393, 0.13011, 0.02875613, 0.04987211, 0.03786379, 0.02684837, 0.0155565],
                            [0.92597561, 0.017, 0.009, 0.00369231, 0.92583814, -0.00670974, 0.00670974, 0.012021, 0.00506763, 0.00420523, 0.01259266, 0.0115391, 0.00265165],
                            [2.43902439e-05, 5.00000000e-04, 1.00000000e-03, 2.56410256e-05, 2.18000765e-04, -5.56411248e-04, 5.56411248e-04, 9.19240000e-03, 2.71058333e-03, 4.25246049e-03, 2.49833278e-03, 7.64311464e-03, 0.00000000e+00],
                            [0.88760976, 0.0205, 0.022, 0.00489744, 0.88799505, -0.00346772, 0.00461828, 0.011314, 0.00447838, 0.00394135, 0.01327278, 0.01434142, 0.00406585],
                            [9.21263415, 0.118, 0.472, 0.0695641, 9.19153391, -0.02181738, 0.02181738, 0.16688, 0.07130037, 0.06135461, 0.04328934, 0.04277416, 0.0829085],
                            [0.48378049, 0.017, 0.026, 0.00794872, 0.48333175, -0.00337375, 0.00350864, 0.016971, 0.0089568, 0.00472601, 0.01168189, 0.01629524, 0.0226275],
                            [0.00000000e+000, 0.00000000e+000, 0.00000000e+000, 0.00000000e+000, 9.65026603e-122, -2.00921455e-120, 4.22507597e-120, 0.00000000e+000, 0.00000000e+000, 0.00000000e+000, 0.00000000e+000, 0.00000000e+000, 0.00000000e+000],
                            [0.10897561, 0.03, 0.033, 0.00553846, 0.12761266, -0.00442938, 0.00556735, 0.025456, 0.00872107, 0.00870258, 0.01130487, 0.01554551, 0.0123745],
                            [0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, -1.38812548e-09, -2.34438020e-08, 2.34438020e-08, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00],
                            [0.66663415, 0.052, 0.05, 0.00510256, 0.66182973, -0.01361869, 0.01361869, 0.0049497, 0.00296982, 0.00208565, 0.00424264, 0.00961131, 0.012374],
                            [3.74146341e+00, 6.60000000e-02, 7.00000000e-02, 2.41025641e-02, 3.72790310e+00, -1.65194036e-02, 1.65194036e-02, 2.33350000e-02, 2.29102000e-02, 3.87787571e-04, 7.25086202e-03, 8.04828002e-03, 2.26270000e-02],
                            [2.43902439e-05, 5.00000000e-04, 1.00000000e-03, 1.02564103e-04, 2.44149661e-05, -7.19856850e-05, 7.19856850e-05, 7.07110000e-04, 1.17851667e-04, 2.88676449e-04, 2.04124145e-04, 1.44336183e-04, 0.00000000e+00],
                            [1.14713659e+01, 1.68000000e-01, 3.24000000e-01, 8.83589744e-02, 1.13977278e+01, -4.35202063e-02, 4.35202063e-02, 1.20920000e-01, 1.15826000e-01, 5.32593935e-03, 4.29825546e-02, 1.11681949e-01, 1.82080000e-01],
                            [1.63631707, 0.0825, 0.138, 0.02410256, 1.65473267, -0.02914746, 0.02927458, 0.074953, 0.02899134, 0.03271076, 0.02718317, 0.09610564, 0.012728],
                            [0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 6.01460518e-42, -2.71490067e-40, 2.71490067e-40, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00],
                            [0.52358537, 0.038, 0.03, 0.00769231, 0.52319376, -0.01066405, 0.01066405, 0.026163, 0.01025307, 0.00912966, 0.02678697, 0.04011893, 0.00866185],
                            [0.10931707, 0.103, 0.407, 0.04461538, 0.13188551, -0.01686662, 0.02506229, 0.1492, 0.0384195, 0.06327203, 0.06411448, 0.05508901, 0],
                            [0.0444878, 0.0245, 0.04, 0.00984615, 0.03577326, -0.00573919, 0.00573919, 0.013435, 0.0078961, 0.00418135, 0.01136515, 0.01291603, 0.0134352],
                            [0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 1.03127202e-08, -2.56175141e-07, 5.37317466e-07, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00],
                            [0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0.],
                            [0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 3.27917545e-05, -7.79437718e-04, 7.79437718e-04, 3.04060000e-02, 5.06766667e-03, 1.24131975e-02, 1.34721936e-02, 5.34029589e-02, 0.00000000e+00],
                            [2.43902439e-05, 5.00000000e-04, 1.00000000e-03, 1.02564103e-04, 2.60691650e-05, -7.19856850e-05, 7.19856850e-05, 7.07110000e-04, 1.17851667e-04, 2.88676449e-04, 2.04124145e-04, 1.44336183e-04, 0.00000000e+00],
                            [0.46446341, 0.033, 0.03, 0.00933333, 0.46299034, -0.00866364, 0.00866364, 0.033941, 0.01357644, 0.01214903, 0.02164486, 0.02701617, 0.012374],
                            [5.89978049, 0.117, 0.112, 0.04453846, 5.88525247, -0.02253416, 0.02253416, 0.084146, 0.0492146, 0.01985341, 0.06802812, 0.09041259, 0.045255],
                            [0.01317073, 0.0195, 0.015, 0.00538462, 0.00829287, -0.00622806, 0.00622806, 0.026163, 0.01145514, 0.00926554, 0.00690652, 0.02540613, 0.018031],
                            [1.16509756, 0.028, 0.02, 0.01051282, 1.16338281, -0.01379371, 0.01379371, 0.020506, 0.01461345, 0.00563317, 0.01416569, 0.01971055, 0.0281075],
                            [3.67914634, 0.1235, 0.126, 0.02676923, 3.67052968, -0.04266586, 0.04266586, 0.041719, 0.0233342, 0.0106888, 0.03232337, 0.07260248, 0.050912],
                            [0.11331707, 0.0015, 0.004, 0.0014359, 0.11329803, -0.00042144, 0.00042144, 0.0021213, 0.0014142, 0.00109543, 0.00124164, 0.00053231, 0.00070713],
                            [1.11256098, 0.026, 0.016, 0.00561538, 1.09093248, -0.00174647, 0.00490015, 0.02192, 0.01272782, 0.00816993, 0.02111102, 0.04921207, 0.012021],
                            [0.06846341, 0.007, 0.01, 0.00307692, 0.06774886, -0.00179795, 0.00190969, 0.0056569, 0.00311126, 0.00162791, 0.00195576, 0.00721732, 0.01096],
                            [1.16454634e+01, 1.78500000e-01, 3.20000000e-01, 8.94615385e-02, 1.15869935e+01, -1.15451745e-02, 1.59897956e-02, 1.37890000e-01, 1.23393333e-01, 1.01170444e-02, 3.66151153e-02, 1.46607419e-01, 1.94455000e-01],
                            [3.45158537, 0.1375, 0.052, 0.01676923, 3.44594643, -0.03141983, 0.03141983, 0.038184, 0.0272946, 0.00958649, 0.01698014, 0.06290749, 0.1393],
                            [3.12563415, 0.0535, 0.111, 0.02897436, 3.17337638, -0.02835417, 0.02835417, 0.054447, 0.0278601, 0.0188188, 0.00755315, 0.03628251, 0.055154],
                            [8.50975610e-02, 1.00000000e-03, 4.00000000e-03, 8.20512821e-04, 8.50491997e-02, -1.84870042e-04, 2.35933619e-04, 1.41420000e-03, 1.41420000e-03, 2.60312573e-11, 4.08248290e-04, 2.88668284e-04, 7.07110000e-04],
                            [0.82373171, 0.048, 0.121, 0.01853846, 0.82149219, -0.0053288, 0.00684639, 0.041012, 0.0208598, 0.01423898, 0.02609294, 0.02676908, 0.01078335],
                            [4.39680488, 0.223, 0.354, 0.09258974, 4.35973108, -0.03206468, 0.03450864, 0.20506, 0.0971572, 0.07235446, 0.13713059, 0.23019854, 0.32138],
                            [5.66058537, 0.0285, 0.093, 0.01282051, 5.66682734, -0.00633008, 0.00633008, 0.040305, 0.01513214, 0.01889847, 0.01503912, 0.03383458, 0],
                            [0.13329268, 0.011, 0.021, 0.00338462, 0.13419267, -0.00262455, 0.00262455, 0.0035355, 0.00226272, 0.00092195, 0.00772172, 0.00411547, 0.0038891],
                            [0.15463415, 0.0325, 0.065, 0.01617949, 0.15422134, -0.00766504, 0.00766504, 0.067882, 0.02286322, 0.02270081, 0.02939288, 0.0224428, 0.017501],
                            [1.47902439e-01, 1.50000000e-03, 2.00000000e-03, 3.84615385e-04, 1.48269290e-01, -1.36058722e-04, 1.36058722e-04, 2.12130000e-03, 8.24950000e-04, 9.39849132e-04, 5.16397779e-04, 5.91603500e-04, 0.00000000e+00],
                            [2.76797561, 0.071, 0.17, 0.03212821, 2.84223399, -0.01692731, 0.01692731, 0.04879, 0.03441267, 0.00934515, 0.03221283, 0.05768286, 0.092806],
                            [1.30939024, 0.044, 0.066, 0.0165641, 1.2967273, -0.01727205, 0.01727205, 0.03182, 0.01456652, 0.01056655, 0.00732632, 0.02987207, 0.038891],
                            [0.0914878, 0.038, 0.028, 0.00364103, 0.08295897, -0.00877545, 0.00877545, 0.032527, 0.00648182, 0.01277828, 0.01289089, 0.01040763, 0.0042426],
                            [0.13621951, 0.0015, 0.006, 0.00174359, 0.13689296, -0.00036169, 0.00040731, 0.0021213, 0.00153205, 0.00082663, 0.00058452, 0.00069522, 0.00088391],
                            [0.05692683, 0.007, 0.006, 0.00189744, 0.05532006, -0.00145672, 0.00145672, 0.0056569, 0.00311126, 0.00184393, 0.00420714, 0.00465287, 0.0070711],
                            [0.07460976, 0.002, 0.006, 0.00097436, 0.07430141, -0.00035004, 0.00038011, 0.0028284, 0.00113136, 0.0011832, 0.00070711, 0.0005916, 0.00070711],
                            [0.04782927, 0.006, 0.011, 0.00353846, 0.04406202, -0.00232859, 0.00232859, 0.012021, 0.00438408, 0.00442728, 0.00363318, 0.00540593, 0.0091924],
                            [4.443, 0.141, 0.076, 0.02310256, 4.40858239, -0.03710778, 0.03710778, 0.03182, 0.0271528, 0.00465324, 0.03506173, 0.07970664, 0.11278],
                            [8.79678049, 0.057, 0.208, 0.04194872, 8.784878, -0.01132933, 0.01132933, 0.08061, 0.04695182, 0.039817, 0.0405623, 0.01937402, 0.033234],
                            [2.58236585, 0.063, 0.128, 0.02112821, 2.5705713, -0.0079298, 0.01979542, 0.062225, 0.0309712, 0.02172778, 0.02949491, 0.02741888, 0.02687],
                            [0.08992683, 0.0015, 0.006, 0.00030769, 0.09000535, -0.00020308, 0.00020308, 0.0021213, 0.00106065, 0.00116188, 0.0007746, 0.00086603, 0.00053035],
                            [0.09085366, 0.0175, 0.037, 0.00694872, 0.09607742, -0.00456388, 0.00456388, 0.0098995, 0.00523258, 0.00310646, 0.01357571, 0.0133944, 0.0056569],
                            [1.34473171, 0.0255, 0.022, 0.00953846, 1.37010789, -0.00558419, 0.00558419, 0.030406, 0.0134351, 0.00877511, 0.00929516, 0.03188089, 0.0265165],
                            [0.14253659, 0.001, 0.004, 0.00097436, 0.14237889, -0.0002998, 0.0002998, 0.0014142, 0.0011785, 0.00057734, 0.0005164, 0.00069521, 0.00106066],
                            [0.07617073, 0.001, 0.004, 0.00179487, 0.07597272, -0.00025949, 0.00025949, 0.0014142, 0.0011785, 0.00057734, 0.0005164, 0.00063245, 0.00070711],
                            [0.28502439, 0.0025, 0.01, 0.00241026, 0.28596915, -0.000355, 0.000355, 0.12869, 0.02333393, 0.05162999, 0.0313152, 0.13233722, 0.0044194],
                            [5.97658537, 0.0645, 0.106, 0.02925641, 5.95365623, -0.01454886, 0.01454886, 0.045962, 0.02913296, 0.02145587, 0.04602717, 0.06410626, 0.053033],
                            [4.19787805, 0.0405, 0.072, 0.02764103, 4.21230508, -0.01456906, 0.01468492, 0.030406, 0.02206174, 0.01003006, 0.02031748, 0.03873656, 0.034295],
                            [0.06904878, 0.0025, 0.005, 0.00117949, 0.06819891, -0.00023428, 0.00033805, 0.0035355, 0.00098994, 0.00154918, 0.001, 0.0007071, 0.00070711],
                            [2.07410488e+01, 1.10000000e-02, 4.40000000e-02, 1.24102564e-02, 2.07288498e+01, -5.11402880e-02, 5.11402880e-02, 1.55560000e-02, 1.55560000e-02, 0.00000000e+00, 5.68037557e-03, 3.17543685e-03, 7.77820000e-03],
                            [0.15141463, 0.0025, 0.008, 0.00161538, 0.15286961, -0.00066236, 0.00066236, 0.0049497, 0.0021213, 0.00180276, 0.00235584, 0.01268589, 0.0021213],
                            [1.07970732, 0.0275, 0.046, 0.00725641, 1.0819483, -0.0025949, 0.00261392, 0.026163, 0.00754248, 0.00945165, 0.01400506, 0.00566908, 0.011137],
                            [1.45278049e+00, 2.50000000e-02, 3.40000000e-02, 8.23076923e-03, 1.46401853e+00, -5.22375992e-03, 7.56803574e-03, 8.48530000e-03, 6.71755000e-03, 1.39641061e-03, 4.14024959e-03, 1.47976972e-02, 2.03295000e-02],
                            [1.18829268e-01, 1.00000000e-03, 4.00000000e-03, 1.17948718e-03, 1.18657803e-01, -3.33958979e-04, 3.55599268e-04, 1.41420000e-03, 1.41420000e-03, 2.60312573e-11, 6.32455532e-04, 5.32284214e-04, 7.07110000e-04],
                            [0.09217073, 0.0085, 0.007, 0.00258974, 0.07952256, -0.00104703, 0.00138337, 0.006364, 0.00466692, 0.00203719, 0.00509166, 0.01307342, 0.021213],
                            [0.06936585, 0.0095, 0.015, 0.00394872, 0.06837444, -0.00205373, 0.00205373, 0.0084853, 0.00296984, 0.0030984, 0.00234521, 0.00419839, 0.0017678],
                            [5.05807317, 0.049, 0.082, 0.02402564, 5.06327737, -0.01120311, 0.01120311, 0.031113, 0.0239, 0.01338272, 0.01117139, 0.04351642, 0.020506],
                            [0.26421951, 0.04, 0.068, 0.00902564, 0.2587529, -0.01040894, 0.01040894, 0.025456, 0.01060666, 0.00890233, 0.01111643, 0.04563416, 0.011314],
                            [3.59336585, 0.0575, 0.054, 0.02094872, 3.58195886, -0.01804095, 0.01838506, 0.043134, 0.0336584, 0.01240579, 0.01683523, 0.04717173, 0.038184],
                            [1.29187805, 0.026, 0.016, 0.00689744, 1.27916244, -0.00322078, 0.00490015, 0.025456, 0.01032378, 0.00861112, 0.01863263, 0.0636921, 0.038537],
                            [6.28670732, 0.1245, 0.127, 0.03102564, 6.35501978, -0.01747513, 0.02813757, 0.084146, 0.04690465, 0.0254467, 0.06541464, 0.18275149, 0.15008],
                            [10.64578049, 0.079, 0.284, 0.04564103, 10.64447668, -0.01946271, 0.01947497, 0.10889, 0.04186, 0.05739752, 0.06891299, 0.05417812, 0.050205],
                            [3.32470732, 0.092, 0.046, 0.01687179, 3.32977984, -0.02794509, 0.02794509, 0.072125, 0.0288498, 0.02428699, 0.06277798, 0.10343739, 0.061518],
                            [0.07358537, 0.001, 0.004, 0.00153846, 0.0735262, -0.00027514, 0.00027514, 0.0014142, 0.0009428, 0.00073029, 0.00075277, 0.00053228, 0.00070711]])

    return {'dual_coef': dual_coef,
            'support_vec': support_vec,
            'intercept': intercept,
            'gamma': gamma}


def multiclass_classifier():
    gamma = 0.1

    # dual coef = y_i*alpha_i
    dual_coef = np.array([[1.00000000e+02, 0.00000000e+00, 0.00000000e+00, 1.00000000e+02, 2.19164051e-01,
                           1.00000000e+02, 1.00000000e+02, 0.00000000e+00, 0.00000000e+00, 1.00000000e+02,
                           2.73972798e+00, 1.00000000e+02, 0.00000000e+00, 1.00000000e+02, 1.00000000e+02,
                           1.00000000e+02, 0.00000000e+00, 0.00000000e+00, 1.00000000e+02, 1.00000000e+02,
                           1.00000000e+02, 0.00000000e+00, 1.00000000e+02, 1.00000000e+02, 5.78184818e+01,
                           1.00000000e+02, 1.00000000e+02, 0.00000000e+00, 1.00000000e+02, 1.00000000e+02,
                           1.00000000e+02, 0.00000000e+00, 1.00000000e+02, 1.00000000e+02, 1.00000000e+02,
                           0.00000000e+00, 1.00000000e+02, 1.00000000e+02, 1.00000000e+02, 4.43824790e+01,
                           0.00000000e+00, 0.00000000e+00, 8.90021137e+01, 1.00000000e+02, 3.38829336e+01,
                           1.00000000e+02, 7.35308055e+01, 5.00832282e+01, 1.00000000e+02, 1.00000000e+02,
                           1.00000000e+02, 9.04295253e+01, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00,
                           1.00000000e+02, 1.00000000e+02, 7.37255035e+01, 1.00000000e+02, 0.00000000e+00,
                           1.00000000e+02, -1.00000000e+02, -4.59726588e+01, -9.10060871e+01, -0.00000000e+00,
                           -1.00000000e+02, -1.00000000e+02, -0.00000000e+00, -0.00000000e+00, -0.00000000e+00,
                           -1.00000000e+02, -1.00000000e+02, -1.00000000e+02, -0.00000000e+00, -0.00000000e+00,
                           -1.00000000e+02, -0.00000000e+00, -0.00000000e+00, -1.00000000e+02, -0.00000000e+00,
                           -0.00000000e+00, -1.00000000e+02, -0.00000000e+00, -1.00000000e+02, -1.00000000e+02,
                           -1.00000000e+02, -1.00000000e+02, -2.32473120e-01, -1.00000000e+02, -0.00000000e+00,
                           -0.00000000e+00, -1.00000000e+02, -1.00000000e+02, -1.00000000e+02, -1.00000000e+02,
                           -0.00000000e+00, -0.00000000e+00, -0.00000000e+00, -1.00000000e+02, -0.00000000e+00,
                           -2.01478019e-01, -1.00000000e+02, -5.32795432e+01, -0.00000000e+00, -0.00000000e+00,
                           -1.00000000e+02, -0.00000000e+00, -0.00000000e+00, -0.00000000e+00, -0.00000000e+00,
                           -1.00000000e+02, -2.05233000e+01, -0.00000000e+00, -9.58435547e-02, -0.00000000e+00,
                           -0.00000000e+00, -0.00000000e+00, -1.00000000e+02, -0.00000000e+00, -1.00000000e+02,
                           -1.00000000e+02, -0.00000000e+00, -0.00000000e+00, -0.00000000e+00, -0.00000000e+00,
                           -1.00000000e+02, -0.00000000e+00, -1.14900102e+01, -7.73085905e+01, -1.00000000e+02,
                           -0.00000000e+00, -1.00000000e+02, -0.00000000e+00, -1.00000000e+02, -0.00000000e+00,
                           -0.00000000e+00, -8.64770605e+01, -1.00000000e+02, -1.18090663e-01, -1.00000000e+02,
                           -1.00000000e+02, -0.00000000e+00, -0.00000000e+00, -1.00000000e+02, -0.00000000e+00,
                           -0.00000000e+00, -6.27523608e+01, -0.00000000e+00, -4.38003436e+01, -0.00000000e+00,
                           -0.00000000e+00, -5.36807440e-02, -0.00000000e+00, -0.00000000e+00, -1.00000000e+02,
                           -0.00000000e+00, -1.51862509e-01, -2.23505792e+01, -0.00000000e+00, -1.71549400e+00,
                           -0.00000000e+00, -0.00000000e+00, -1.00000000e+02, -1.00000000e+02, -0.00000000e+00,
                           -1.00000000e+02, -0.00000000e+00, -1.00000000e+02, -1.00000000e+02, -0.00000000e+00,
                           -6.48908553e+01, -5.45079781e+01, -0.00000000e+00, -1.00000000e+02, -1.00000000e+02,
                           -1.00000000e+02, -4.15526000e+01, -1.00000000e+02, -0.00000000e+00, -0.00000000e+00,
                           -3.97322757e+01, -1.00000000e+02, -1.00000000e+02, -0.00000000e+00, -1.00000000e+02,
                           -1.00000000e+02, -8.51452564e+01, -1.00000000e+02, -0.00000000e+00, -0.00000000e+00,
                           -0.00000000e+00, -0.00000000e+00, -0.00000000e+00, -1.00000000e+02, -1.00000000e+02,
                           -1.00000000e+02, -1.00000000e+02, -0.00000000e+00, -1.00000000e+02, -1.00000000e+02,
                           -1.00000000e+02, -1.36707150e+01, -2.28944671e+00, -1.00000000e+02, -1.00000000e+02,
                           -0.00000000e+00, -1.00000000e+02, -0.00000000e+00, -1.00000000e+02, -9.70237576e+01,
                           -0.00000000e+00, -1.00000000e+02, -8.98901380e+00, -1.00000000e+02, -0.00000000e+00,
                           -1.00000000e+02, -1.00000000e+02, -0.00000000e+00, -0.00000000e+00, -1.00000000e+02,
                           -0.00000000e+00, -0.00000000e+00, -0.00000000e+00, -2.31872364e+00, -0.00000000e+00,
                           -1.00000000e+02, -1.00000000e+02, -6.81207558e+01, -0.00000000e+00, -1.00000000e+02,
                           -1.00000000e+02, -1.00000000e+02, -0.00000000e+00, -0.00000000e+00, -1.00000000e+02,
                           -1.25804913e+01, -1.00000000e+02, -1.00000000e+02, -0.00000000e+00, -1.00000000e+02,
                           -0.00000000e+00, -0.00000000e+00, -5.79636185e+01, -0.00000000e+00, -3.60349193e+01,
                           -1.00000000e+02, -1.00000000e+02],
                          [1.00000000e+02, 1.00000000e+02, 1.00000000e+02, 1.00000000e+02, 0.00000000e+00,
                           1.00000000e+02, 0.00000000e+00, 1.22133880e+01, 1.00000000e+02, 1.00000000e+02,
                           1.00000000e+02, 1.00000000e+02, 1.00000000e+02, 1.00000000e+02, 1.00000000e+02,
                           1.00000000e+02, 1.00000000e+02, 1.00000000e+02, 0.00000000e+00, 1.00000000e+02,
                           0.00000000e+00, 1.00000000e+02, 1.00000000e+02, 5.45699567e+01, 0.00000000e+00,
                           1.00000000e+02, 1.00000000e+02, 1.00000000e+02, 1.00000000e+02, 1.00000000e+02,
                           1.00000000e+02, 1.00000000e+02, 1.00000000e+02, 1.00000000e+02, 1.00000000e+02,
                           1.00000000e+02, 1.00000000e+02, 0.00000000e+00, 1.00000000e+02, 0.00000000e+00,
                           1.00000000e+02, 1.00000000e+02, 5.30198194e+01, 1.00000000e+02, 0.00000000e+00,
                           8.10028022e+01, 0.00000000e+00, 1.00000000e+02, 1.00000000e+02, 8.57299348e+01,
                           0.00000000e+00, 1.00000000e+02, 1.00000000e+02, 1.00000000e+02, 1.00000000e+02,
                           1.00000000e+02, 0.00000000e+00, 0.00000000e+00, 1.00000000e+02, 1.00000000e+02,
                           1.00000000e+02, 1.00000000e+02, 1.00000000e+02, 1.00000000e+02, 1.00000000e+02,
                           0.00000000e+00, 1.00000000e+02, 1.00000000e+02, 1.00000000e+02, 1.00000000e+02,
                           1.00000000e+02, 1.00000000e+02, 1.00000000e+02, 6.98226850e+00, 1.00000000e+02,
                           1.00000000e+02, 2.28942244e+00, 1.00000000e+02, 0.00000000e+00, 3.10951756e+00,
                           1.00000000e+02, 1.00000000e+02, 1.00000000e+02, 0.00000000e+00, 2.43965458e+01,
                           5.54247795e+01, 4.89715327e+01, 0.00000000e+00, 1.00000000e+02, 1.00000000e+02,
                           1.00000000e+02, 0.00000000e+00, 1.00000000e+02, 1.00000000e+02, 1.00000000e+02,
                           1.00000000e+02, 1.00000000e+02, 1.00000000e+02, 1.00000000e+02, 8.77648862e-01,
                           1.41352297e+00, 1.00000000e+02, 0.00000000e+00, 1.00000000e+02, 5.87399500e+01,
                           1.00000000e+02, 7.89673831e+01, 7.17216921e-01, 7.08622898e+01, 1.00000000e+02,
                           1.00000000e+02, 0.00000000e+00, 1.00000000e+02, 0.00000000e+00, 7.08652210e+01,
                           1.00000000e+02, 1.00000000e+02, 1.00000000e+02, 2.28740165e+00, 1.00000000e+02,
                           1.00000000e+02, 1.00000000e+02, 6.26644343e+01, 1.51915932e+01, 9.33156003e+01,
                           1.00000000e+02, 5.73480226e-01, 0.00000000e+00, 0.00000000e+00, 1.00000000e+02,
                           6.51947143e+01, 0.00000000e+00, 1.00000000e+02, 3.61854680e+01, 1.50700439e+00,
                           3.93114839e+01, 1.00000000e+02, 1.00000000e+02, 0.00000000e+00, 0.00000000e+00,
                           1.00000000e+02, 1.62942145e+01, 1.00000000e+02, 1.00000000e+02, 3.65697187e+01,
                           3.32328741e+01, 1.00000000e+02, 1.00000000e+02, 1.00000000e+02, 1.00000000e+02,
                           1.00000000e+02, 0.00000000e+00, 3.84017861e-02, 3.27497129e+00, 1.00000000e+02,
                           1.00000000e+02, 0.00000000e+00, 1.00000000e+02, -1.00000000e+02, -1.00000000e+02,
                           -1.00000000e+02, -1.00000000e+02, -1.00000000e+02, -0.00000000e+00, -7.87287696e+01,
                           -1.00000000e+02, -2.17133274e+01, -1.00000000e+02, -0.00000000e+00, -1.00000000e+02,
                           -1.00000000e+02, -1.00000000e+02, -1.00000000e+02, -0.00000000e+00, -1.00000000e+02,
                           -1.00000000e+02, -1.00000000e+02, -1.00000000e+02, -1.00000000e+02, -1.00000000e+02,
                           -1.00000000e+02, -4.03561653e+01, -1.00000000e+02, -1.00000000e+02, -0.00000000e+00,
                           -1.00000000e+02, -0.00000000e+00, -0.00000000e+00, -8.73885349e+01, -1.00000000e+02,
                           -1.00000000e+02, -1.00000000e+02, -1.00000000e+02, -1.00000000e+02, -1.00000000e+02,
                           -0.00000000e+00, -0.00000000e+00, -1.00000000e+02, -1.00000000e+02, -0.00000000e+00,
                           -1.00000000e+02, -1.00000000e+02, -1.00000000e+02, -1.00000000e+02, -1.00000000e+02,
                           -1.00000000e+02, -0.00000000e+00, -1.00000000e+02, -1.00000000e+02, -0.00000000e+00,
                           -1.00000000e+02, -1.00000000e+02, -1.00000000e+02, -0.00000000e+00, -1.00000000e+02,
                           -0.00000000e+00, -1.00000000e+02, -1.00000000e+02, -7.69821289e+01, -0.00000000e+00,
                           -1.00000000e+02, -8.28241499e+01, -1.00000000e+02, -6.27852100e+00, -8.74723914e+01,
                           -0.00000000e+00, -0.00000000e+00, -1.00000000e+02, -1.00000000e+02, -1.00000000e+02,
                           -0.00000000e+00, -0.00000000e+00, -1.00000000e+02, -1.00000000e+02, -0.00000000e+00,
                           -1.00000000e+02, -1.00000000e+02, -1.00000000e+02, -2.15412985e+01, -0.00000000e+00,
                           -2.97074994e+01, -9.62658735e+01, -1.00000000e+02, -1.00000000e+02, -1.00000000e+02,
                           -1.00000000e+02, -0.00000000e+00]])

    # intercept = rho
    intercept = np.array([-0.62674907,  1.31994877,  0.67252991])

    # support vectors = x_i
    support_vec = np.array([[5.49570019e-07, -2.58632551e-07, 3.16229206e-02, 0.00000000e+00, 0.00000000e+00, 7.07110000e-02, 0.00000000e+00, 0.00000000e+00, 5.49570019e-07,-1.79132036e-08],
                            [0.01061208, -0.00799413, 0.00572262, 0.025, 0.02809756,0.016263, 0.058, 0.00866667, 0.01061208, 0.03799132],
                            [0.0005069, -0.0005069, 0.0031833, 0.007, 0.00060976,0.0070711, 0.014, 0.00087179, 0.00039282, 0.00024191],
                            [0.02659884, -0.02236652, 0.01818914, 0.054, 3.49731707,0.062225, 0.063, 0.01923077, 0.02659884, 3.53815367],
                            [0.1959552, -0.19377234, 0.49935644, 2.567, 0.212,1.2473, 4.086, 0.27730769, 0.1959552, 0.21128449],
                            [0.01296803, -0.01070662, 0.01579683, 0.055, 2.52643902,0.043134, 0.057, 0.0114359, 0.01296803, 2.54031008],
                            [0.04634941, -0.03616377, 0.03342396, 0.285, 1.25278049,0.11031, 0.285, 0.05482051, 0.04634941, 1.24439126],
                            [0.01161685, -0.01161685, 0.01472225, 0.061, 0.00495122,0.036062, 0.131, 0.00758974, 0.00936955, 0.00494698],
                            [3.36047450e-17, -2.42947453e-17, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 3.36047450e-17,-2.74622599e-18],
                            [0.05403825, -0.04130143, 0.01255584, 0.27, 3.89758537,0.047376, 0.27, 0.06717949, 0.05403825, 3.87923565],
                            [0.06322635, -0.0450853, 0.0069893, 0.107, 0.52617073,0.019799, 0.166, 0.01469231, 0.06322635, 0.51547654],
                            [0.02740927, -0.02740927, 0.01595008, 0.106, 0.93326829,0.043841, 0.106, 0.01169231, 0.01663183, 0.92669874],
                            [0.02181645, -0.02181645, 0.00031623, 0.045, 0.04685366,0.00070711, 0.045, 0.00605128, 0.01964941, 0.05939633],
                            [0.03168223, -0.03168223, 0.03369301, 0.13, 4.64253659,0.10182, 0.13, 0.03661538, 0.02589546, 4.63806066],
                            [0.00648365, -0.00648365, 0.00126494, 0.039, 0.29312195,0.0056569, 0.039, 0.00682051, 0.0043318, 0.28575076],
                            [0.02260403, -0.02260403, 0.01570345, 0.123, 3.08187805,0.053033, 0.123, 0.03435897, 0.01139526, 3.11862292],
                            [7.19856842e-05, -7.19856842e-05, 0.00000000e+00, 1.00000000e-03, 2.43902439e-05, 0.00000000e+00, 2.00000000e-03, 1.02564103e-04, 5.59639155e-05, 2.43769719e-05],
                            [5.77955577e-09, -2.77557303e-09, 2.88676449e-04, 0.00000000e+00, 0.00000000e+00, 7.07110000e-04, 0.00000000e+00, 0.00000000e+00, 5.77955577e-09,-4.05052739e-10],
                            [0.0168393, -0.0168393, 0.04987211, 0.172, 0.83290244,0.13011, 0.287, 0.02610256, 0.01063438, 0.82408369],
                            [0.00670974, -0.00670974, 0.00420523, 0.009, 0.92597561,0.012021, 0.012, 0.00369231, 0.00394705, 0.92583814],
                            [0.04881422, -0.04881422, 0.06039519, 0.128, 0.84173171,0.15274, 0.166, 0.0145641, 0.02705203, 0.83114509],
                            [5.56411248e-04, -5.56411248e-04, 4.25246049e-03, 1.00000000e-03, 2.43902439e-05, 9.19240000e-03, 1.00000000e-03, 2.56410256e-05, 4.16114259e-04, 2.18000765e-04],
                            [0.00461828, -0.00346772, 0.00394135, 0.022, 0.88760976,0.011314, 0.022, 0.00489744, 0.00461828, 0.88799505],
                            [0.02181738, -0.02181738, 0.06135461, 0.472, 9.21263415,0.16688, 0.472, 0.0695641, 0.01361679, 9.19153391],
                            [0.18064104, -0.18064104, 0.02243327, 0.141, 1.74753659,0.065761, 0.141, 0.02587179, 0.08614869, 1.89288442],
                            [0.04502684, -0.04502684, 0.03092595, 0.075, 0.66726829,0.070004, 0.075, 0.01507692, 0.02703687, 0.6524218 ],
                            [0.00350864, -0.00337375, 0.00472601, 0.026, 0.48378049,0.016971, 0.038, 0.00794872, 0.00350864, 0.48333175],
                            [4.22507597e-120, -2.00921455e-120, 0.00000000e+000, 0.00000000e+000, 0.00000000e+000, 0.00000000e+000, 0.00000000e+000, 0.00000000e+000, 4.22507597e-120, 9.65026603e-122],
                            [0.0511954, -0.0511954, 0.00917865, 0.032, 1.50741463,0.028284, 0.056, 0.00958974, 0.01612377, 1.43087637],
                            [0.0504243, -0.03682241, 0.03693438, 0.147, 3.4434878,0.086267, 0.147, 0.02051282, 0.0504243, 3.49633275],
                            [0.00556735, -0.00442938, 0.00870258, 0.033, 0.10897561,0.025456, 0.033, 0.00553846, 0.00556735, 0.12761266],
                            [2.34438020e-08, -2.34438020e-08, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 1.15475168e-08,-1.38812548e-09],
                            [0.01361869, -0.01361869, 0.00208565, 0.05, 0.66663415,0.0049497, 0.05, 0.00510256, 0.0087315, 0.66182973],
                            [0.00526982, -0.00287197, 0.016589, 0.06, 0.02595122,0.042426, 0.06, 0.00515385, 0.00526982, 0.02530658],
                            [1.65194036e-02, -1.65194036e-02, 3.87787571e-04, 7.00000000e-02, 3.74146341e+00, 2.33350000e-02, 1.06000000e-01, 2.41025641e-02, 1.58008422e-02, 3.72790310e+00],
                            [7.19856850e-05, -7.19856850e-05, 2.88676449e-04, 1.00000000e-03, 2.43902439e-05, 7.07110000e-04, 2.00000000e-03, 1.02564103e-04, 5.59639159e-05, 2.44149661e-05],
                            [4.35202063e-02, -4.35202063e-02, 5.32593935e-03, 3.24000000e-01, 1.14713659e+01, 1.20920000e-01, 3.24000000e-01, 8.83589744e-02, 3.31111507e-02, 1.13977278e+01],
                            [0.0767267, -0.0767267, 0.07121201, 0.446, 8.87180488,0.24324, 0.446, 0.10620513, 0.0720187, 8.83162683],
                            [0.02927458, -0.02914746, 0.03271076, 0.138, 1.63631707,0.074953, 0.138, 0.02410256, 0.02927458, 1.65473267],
                            [0.02124579, -0.00660226, 0.05683001, 0.218, 0.16541463,0.13718, 0.241, 0.04171795, 0.02124579, 0.07039812],
                            [2.71490067e-40, -2.71490067e-40, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 1.26462710e-40, 6.01460518e-42],
                            [0.0147575, -0.0147575, 0.016589, 0.06, 0.03368293,0.042426, 0.06, 0.00917949, 0.01001548, 0.03172037],
                            [0.05563514, -0.05003055, 0.0377493, 0.474, 3.34209756,0.10819, 0.474, 0.04441026, 0.05563514, 3.34310614],
                            [0.01066405, -0.01066405, 0.00912966, 0.03, 0.52358537,0.026163, 0.042, 0.00769231, 0.00753237, 0.52319376],
                            [0.02506229, -0.01686662, 0.06327203, 0.407, 0.10931707,0.1492, 0.407, 0.04461538, 0.02506229, 0.13188551],
                            [0.05540528, -0.05540528, 0.05916798, 0.36, 5.93456098,0.18526, 0.36, 0.14474359, 0.03879351, 5.91696978],
                            [0.05114493, -0.04906722, 0.03169166, 0.444, 1.59946341,0.089803, 0.444, 0.051, 0.05114493, 1.39984371],
                            [1.87455177e-04, -8.82323678e-05, 3.60629601e-02, 0.00000000e+00, 0.00000000e+00, 7.63680000e-02, 0.00000000e+00, 0.00000000e+00, 1.87455177e-04,-1.21670239e-05],
                            [0.00573919, -0.00573919, 0.00418135, 0.04, 0.0444878,0.013435, 0.04, 0.00984615, 0.00463094, 0.03577326],
                            [0.02921642, -0.02921642, 0.06455675, 0.145, 1.28114634,0.15486, 0.145, 0.0314359, 0.00907076, 1.30595106],
                            [0.08101569, -0.05496349, 0.03591274, 0.557, 5.79265854,0.1294, 0.557, 0.10274359, 0.08101569, 5.7521422 ],
                            [5.37317466e-07, -2.56175141e-07, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 5.37317466e-07, 1.03127202e-08],
                            [0., 0., 0., 0., 0., 0., 0., 0., 0., 0.],
                            [7.79437718e-04, -7.79437718e-04, 1.24131975e-02, 0.00000000e+00, 0.00000000e+00, 3.04060000e-02, 0.00000000e+00, 0.00000000e+00, 3.70210952e-04, 3.27917545e-05],
                            [7.19856850e-05, -7.19856850e-05, 2.88676449e-04, 1.00000000e-03, 2.43902439e-05, 7.07110000e-04, 2.00000000e-03, 1.02564103e-04, 4.31188798e-05, 2.60691650e-05],
                            [0.00866364, -0.00866364, 0.01214903, 0.03, 0.46446341,0.033941, 0.048, 0.00933333, 0.00584264, 0.46299034],
                            [0.03683296, -0.03683296, 0.07256353, 0.2, 0.89,0.18385, 0.353, 0.0225641, 0.02517257, 0.88537216],
                            [0.03602508, -0.03342342, 0.05528397, 0.172, 0.12058537,0.14991, 0.173, 0.05969231, 0.03602508, 0.10686173],
                            [0.02253416, -0.02253416, 0.01985341, 0.112, 5.89978049,0.084146, 0.12, 0.04453846, 0.01958278, 5.88525247],
                            [0.00622806, -0.00622806, 0.00926554, 0.015, 0.01317073,0.026163, 0.036, 0.00538462, 0.00320484, 0.00829287],
                            [0.01379371, -0.01379371, 0.00563317, 0.02, 1.16509756,0.020506, 0.043, 0.01051282, 0.0134218, 1.16338281],
                            [0.01159407, -0.01159407, 0.01052999, 0.058, 1.19278049,0.028991, 0.058, 0.00697436, 0.0092909, 1.19462633],
                            [0.01666124, -0.01508531, 0.01591668, 0.15, 4.25739024,0.057276, 0.15, 0.02835897, 0.01666124, 4.23504838],
                            [0.02791508, -0.02791508, 0.00861704, 0.059, 2.72943902,0.031113, 0.059, 0.01371795, 0.02720056, 2.66137841],
                            [0.02677122, -0.02677122, 0.00984899, 0.043, 4.52929268,0.051619, 0.074, 0.02512821, 0.02513023, 4.52779804],
                            [0.00820856, -0.00820856, 0.04182701, 0.229, 0.85280488,0.12374, 0.229, 0.02958974, 0.00626605, 0.8738938 ],
                            [0.00660789, -0.00533542, 0.00713786, 0.049, 0.06365854,0.027577, 0.049, 0.01835897, 0.00660789, 0.05446593],
                            [0.00553774, -0.00553774, 0.00491419, 0.026, 3.13085366,0.014849, 0.026, 0.00651282, 0.0033944, 3.12413464],
                            [7.01018256e-03, -7.01018256e-03, 2.29380976e-02, 1.26000000e-01, 6.56448780e+00, 4.45480000e-02, 1.26000000e-01, 1.09743590e-02, 5.16783812e-03, 6.55718858e+00],
                            [0.01903315, -0.01903315, 0.02305625, 0.102, 4.03021951,0.073539, 0.127, 0.04748718, 0.01826975, 4.03932544],
                            [0.010121, -0.00916936, 0.01151868, 0.072, 0.54807317,0.037477, 0.072, 0.02423077, 0.010121, 0.56664269],
                            [0.02477958, -0.02477958, 0.00811683, 0.072, 2.91585366,0.033941, 0.106, 0.01771795, 0.02394629, 2.93431172],
                            [0.01211938, -0.01211938, 0.01493102, 0.059, 1.22153659,0.038891, 0.059, 0.0164359, 0.00614063, 1.21761274],
                            [2.29049895e-02, -2.29049895e-02, 6.47077661e-03, 3.14000000e-01, 1.10041220e+01, 1.23740000e-01, 3.27000000e-01, 1.18076923e-01, 2.12905216e-02, 1.09333661e+01],
                            [0.01060994, -0.01060994, 0.01007483, 0.034, 3.8605122,0.024042, 0.034, 0.01323077, 0.00897269, 3.85004794],
                            [0.00627491, -0.00627491, 0.00093988, 0.022, 0.77860976,0.0056569, 0.022, 0.00546154, 0.0034974, 0.77367652],
                            [0.05803024, -0.0551299, 0.10802924, 0.289,15.55358537, 0.20435, 0.289, 0.05861538, 0.05803024, 15.56158018],
                            [0.01871651, -0.01186601, 0.0663762, 0.171,11.80826829, 0.13011, 0.171, 0.02697436, 0.01871651, 11.8298579 ],
                            [0.05996643, -0.05996643, 0.01671762, 0.244, 0.87841463,0.042426, 0.244, 0.02264103, 0.03965289, 0.87488872],
                            [4.98711743e-02, -4.98711743e-02, 3.65041596e-03, 5.28000000e-01, 1.48976098e+01, 1.90920000e-01, 5.28000000e-01, 1.90358974e-01, 3.84779140e-02, 1.48525928e+01],
                            [0.01136134, -0.01136134, 0.00951859, 0.069, 3.72480488,0.043841, 0.132, 0.02179487, 0.00932468, 3.66845001],
                            [0.03194417, -0.03194417, 0.00795281, 0.026, 1.34785366,0.028284, 0.028, 0.00864103, 0.03178111, 1.33144448],
                            [2.64572119e-02, -2.21160394e-02, 7.74479696e-04, 7.50000000e-02, 4.26239024e+00, 2.82840000e-02, 7.50000000e-02, 2.74358974e-02, 2.64572119e-02, 4.25156996e+00],
                            [0.02750577, -0.0258611, 0.04753049, 0.27, 9.51297561,0.092631, 0.27, 0.06833333, 0.02750577, 9.52726146],
                            [0.00866709, -0.00541625, 0.06576809, 0.186, 0.70678049,0.15415, 0.186, 0.01346154, 0.00866709, 0.70474759],
                            [0.03066446, -0.02713743, 0.02467478, 0.232, 4.49546341,0.097581, 0.35, 0.08861538, 0.03066446, 4.52324303],
                            [0.03264028, -0.03264028, 0.06438005, 0.102, 0.34746341,0.15415, 0.2, 0.01420513, 0.02531656, 0.31306559],
                            [0.10027745, -0.10027745, 0.24027297, 2.921,19.25053659, 0.87257, 2.921, 0.42974359, 0.095849, 19.15442972],
                            [0.00700114, -0.00699188, 0.00745196, 0.051, 0.07097561,0.028991, 0.056, 0.01948718, 0.00700114, 0.07004534],
                            [0.02156914, -0.02156914, 0.00709096, 0.037, 2.22063415,0.028284, 0.055, 0.01587179, 0.01496926, 2.2271266 ],
                            [0.04052342, -0.04052342, 0.01554335, 0.088, 4.43229268,0.060104, 0.088, 0.02589744, 0.02803208, 4.44932817],
                            [0.03391535, -0.03391535, 0.03559771, 0.18, 0.57178049,0.13576, 0.236, 0.08284615, 0.0298453, 0.61786751],
                            [0.01311557, -0.01311557, 0.01651455, 0.096, 5.56714634,0.034648, 0.098, 0.02623077, 0.01097477, 5.57445898],
                            [0.02984218, -0.02984218, 0.04015245, 0.156, 0.49970732,0.11384, 0.162, 0.05010256, 0.02111739, 0.50149962],
                            [0.00525823, -0.00525823, 0.0044271, 0.017, 1.22795122,0.014849, 0.024, 0.00615385, 0.00431849, 1.2262395 ],
                            [0.01034227, -0.0064927, 0.00905154, 0.054, 3.7145122,0.022627, 0.059, 0.01389744, 0.01034227, 3.71109431],
                            [0.01480054, -0.01480054, 0.01288003, 0.1, 4.24504878,0.04879, 0.1, 0.03451282, 0.01429233, 4.26818139],
                            [0.02487982, -0.02487982, 0.04157097, 0.238, 9.47917073,0.084146, 0.238, 0.06812821, 0.02178416, 9.4506215 ],
                            [0.0519567, -0.0519567, 0.03125581, 0.244, 3.01041463,0.10394, 0.244, 0.04551282, 0.03842769, 3.00134531],
                            [0.11461199, -0.11461199, 0.18484777, 1.671,15.49843902, 0.63498, 1.671, 0.38930769, 0.10481226, 15.48006191],
                            [0.07299834, -0.06491022, 0.18753433, 0.99,21.14187805, 0.35002, 0.99, 0.19264103, 0.07299834, 21.11450374],
                            [0.00637064, -0.00336012, 0.05694093, 0.131, 1.2014878,0.13647, 0.175, 0.01325641, 0.00637064, 1.18082784],
                            [3.20491472e-02, -3.20491472e-02, 5.58137707e-03, 3.42000000e-01, 1.17296585e+01, 1.27280000e-01, 3.52000000e-01, 8.53589744e-02, 2.53251481e-02, 1.17137294e+01],
                            [0.02781479, -0.02778159, 0.02039015, 0.06, 4.47456098,0.060811, 0.06, 0.02194872, 0.02781479, 4.48594928],
                            [2.46837687e-02, -2.46837687e-02, 5.41311894e-03, 5.08000000e-01, 1.48384390e+01, 1.92330000e-01, 5.44000000e-01, 1.59897436e-01, 2.11520468e-02, 1.48894277e+01],
                            [0.00868918, -0.00474268, 0.05575972, 0.069, 0.05356098,0.13576, 0.069, 0.00579487, 0.00868918, 0.02191644],
                            [0.04526781, -0.04408048, 0.07762039, 0.253,13.55292683, 0.16405, 0.461, 0.10697436, 0.04526781, 13.52730947],
                            [0.05879119, -0.05879119, 0.13295615, 0.455,18.92917073, 0.30759, 0.83, 0.13789744, 0.04592195, 18.93763555],
                            [0.02189384, -0.01216995, 0.05066665, 0.246, 9.98273171,0.098995, 0.28, 0.06220513, 0.02189384, 9.99516767],
                            [0.02539376, -0.02539376, 0.01551926, 0.07, 4.18487805,0.048083, 0.07, 0.01479487, 0.01839075, 4.18516621],
                            [0.03555225, -0.02731574, 0.02814509, 0.204, 3.88414634,0.099702, 0.235, 0.08258974, 0.03555225, 3.93872613],
                            [0.0582598, -0.04036336, 0.04616096, 0.319,11.37339024, 0.11455, 0.337, 0.06123077, 0.0582598, 11.34920676],
                            [0.01981077, -0.01981077, 0.03330417, 0.106, 8.0285122,0.062933, 0.188, 0.03092308, 0.01455206, 8.03116754],
                            [0.05959746, -0.04430587, 0, 0.99,21.09304878, 0.35002, 0.99, 0.43153846, 0.05959746, 21.13322117],
                            [3.21965997e-02, -3.21965997e-02, 4.40061045e-03, 2.82000000e-01, 1.03585610e+01, 1.06070000e-01, 2.82000000e-01, 1.04769231e-01, 3.08413155e-02, 1.03278009e+01],
                            [1.61063213e-02, -1.61063213e-02, 5.67556300e-02, 3.14000000e-01, 1.12777805e+01, 1.14550000e-01, 3.14000000e-01, 4.11025641e-02, 1.10897848e-02, 1.12605687e+01],
                            [0.02189384, -0.01179437, 0.05216424, 0.151,10.04721951, 0.098995, 0.151, 0.04297436, 0.02189384, 10.08818858],
                            [0.00238657, -0.00238657, 0.0034182, 0.014, 0.77714634,0.011314, 0.016, 0.00351282, 0.00232057, 0.77743005],
                            [0.06433761, -0.06083287, 0.14957506, 0.946,20.62917073, 0.33446, 0.946, 0.37597436, 0.06433761, 20.61320173],
                            [1.39575245e-04, -6.63125529e-05, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 1.39575245e-04, 8.37811140e-07],
                            [2.78771985e-10, -2.78771985e-10, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 0.00000000e+00, 2.70507545e-10, 2.56354866e-11],
                            [1.17302985e-02, -7.71146815e-03, 2.53259440e-02, 1.64000000e-01, 7.73348780e+00, 6.50540000e-02, 1.74000000e-01, 4.20000000e-02, 1.17302985e-02, 7.74824436e+00],
                            [0.01495479, -0.01495479, 0.05647272, 0.306,11.12082927, 0.11455, 0.306, 0.03379487, 0.01350048, 11.13703625],
                            [0.06224454, -0.06224454, 0.10018563, 0.528,14.96365854, 0.19799, 0.528, 0.08512821, 0.04174687, 14.9855656 ],
                            [1.50784174e-02, -1.50784174e-02, 5.75469303e-02, 3.14000000e-01, 1.12202439e+01, 1.11020000e-01, 3.14000000e-01, 8.52051282e-02, 8.68754010e-03, 1.12353267e+01],
                            [0.00507143, -0.00507143, 0.00379475, 0.017, 0.02482927,0.0098995, 0.023, 0.00358974, 0.00308139, 0.023151 ],
                            [0.02900558, -0.02900558, 0.09336675, 0.528,14.83170732, 0.18668, 0.528, 0.08441026, 0.0267984, 14.78693574],
                            [0.01743156, -0.01634197, 0.06142791, 0.354,11.68380488, 0.13011, 0.355, 0.09658974, 0.01743156, 11.6854788 ],
                            [0.06157537, -0.04061912, 0.09114882, 0.434, 8.39795122,0.31396, 0.878, 0.17061538, 0.06157537, 8.3171658 ],
                            [0.01157586, -0.00787553, 0.00781547, 0.103, 3.28846341,0.034648, 0.103, 0.02582051, 0.01157586, 3.29875571],
                            [7.33132616e-03, -6.81397261e-03, 3.00898494e-02, 1.88000000e-01, 7.75687805e+00, 6.64680000e-02, 1.88000000e-01, 4.87179487e-02, 7.33132616e-03, 7.75743580e+00],
                            [0.0325806, -0.02489858, 0.06056313, 0.347, 1.96856098,0.17678, 0.347, 0.09082051, 0.0325806, 1.93503196],
                            [0.02057271, -0.02057271, 0.03637653, 0.168, 7.91641463,0.070004, 0.168, 0.03194872, 0.0175232, 7.87590288],
                            [0.02488661, -0.02177646, 0.01564241, 0.205, 0.40578049,0.087681, 0.205, 0.05994872, 0.02488661, 0.41578688],
                            [0.07366755, -0.07366755, 0.13654436, 0.99,20.81831707, 0.35002, 0.99, 0.11958974, 0.04818856, 20.83153348],
                            [0.02708256, -0.02708256, 0.08111137, 0.224,13.16290244, 0.15839, 0.42, 0.09841026, 0.02505871, 13.18839945],
                            [0.00958152, -0.00958152, 0.02217553, 0.12, 6.09365854,0.045255, 0.12, 0.02551282, 0.00690841, 6.10985475],
                            [0.02958837, -0.01655602, 0.0402901, 0.206, 4.10741463,0.1393, 0.276, 0.05653846, 0.02958837, 4.02902721],
                            [0.07592541, -0.07592541, 0.19758112, 1.738,16.99126829, 0.50275, 1.738, 0.51812821, 0.06254254, 16.89231623],
                            [0.03172265, -0.0218731, 0.0643494, 0.611, 6.06187805,0.21567, 0.611, 0.11628205, 0.03172265, 6.03346158],
                            [0.00524933, -0.00425262, 0.00617516, 0.022, 0.88902439,0.015556, 0.022, 0.00415385, 0.00524933, 0.88950276],
                            [2.94223481e-02, -1.75922046e-02, 1.93346063e-03, 3.24000000e-01, 1.13452439e+01, 1.14550000e-01, 3.24000000e-01, 1.01076923e-01, 2.94223481e-02, 1.14096319e+01],
                            [0.04812244, -0.03838389, 0.06461018, 0.461,13.97795122, 0.1789, 0.461, 0.13925641, 0.04812244, 13.95949649],
                            [0.03621849, -0.03433152, 0.03043904, 0.165, 1.58258537,0.08556, 0.184, 0.03061538, 0.03621849, 1.53392771],
                            [0.02373659, -0.02277035, 0.05867974, 0.434,13.38863415, 0.16405, 0.434, 0.07953846, 0.02373659, 13.34239253],
                            [0.02111005, -0.02111005, 0.07987431, 0.224,13.04126829, 0.15839, 0.224, 0.04605128, 0.02028668, 13.05917311],
                            [0.03671861, -0.03671861, 0.01580834, 0.064, 3.6292439,0.060811, 0.084, 0.03615385, 0.03589838, 3.59330461],
                            [0.03538591, -0.03538591, 0.05734932, 0.175,11.07687805, 0.11738, 0.175, 0.04389744, 0.02528186, 11.06401646],
                            [0.01139907, -0.01139907, 0.0404773, 0.256, 9.43263415,0.09051, 0.256, 0.04312821, 0.01139243, 9.44721736],
                            [2.21139985e-02, -2.21139985e-02, 7.74479696e-04, 1.13000000e-01, 4.05441463e+00, 2.68700000e-02, 1.13000000e-01, 3.17948718e-02, 1.83677482e-02, 4.02531967e+00],
                            [0.03887186, -0.02634328, 0.0509491, 0.189,11.66063415, 0.13364, 0.189, 0.07851282, 0.03887186, 11.63597796],
                            [4.18064577e-02, -4.18064577e-02, 7.74589241e-03, 3.82000000e-01, 1.81040000e+01, 1.41420000e-02, 7.64000000e-01, 7.98974359e-02, 2.37616498e-02, 1.81240583e+01],
                            [0.0610875, -0.05055225, 0.17753526, 1.452,16.057, 0.49144, 1.452, 0.30223077, 0.0610875, 15.96984146],
                            [0.04722622, -0.02572493, 0.06540382, 0.494,14.07109756, 0.17466, 0.494, 0.16689744, 0.04722622, 14.02996765],
                            [0.00210856, -0.00193237, 0.01084809, 0.048, 0.28902439,0.028284, 0.053, 0.00574359, 0.00210856, 0.28286954],
                            [0.01251112, -0.01153534, 0.03329564, 0.094, 7.72314634,0.066468, 0.178, 0.04507692, 0.01251112, 7.71739425],
                            [0.07891577, -0.05286848, 0.10496868, 0.385,17.73741463, 0.2588, 0.751, 0.13687179, 0.07891577, 17.7611277 ],
                            [2.57691246e-03, -1.65339484e-03, 1.37839166e-03, 1.00000000e-02, 1.53980488e+00, 9.19240000e-03, 1.00000000e-02, 4.35897436e-03, 2.57691246e-03, 1.52310545e+00],
                            [1.65446780e-02, -1.65446780e-02, 5.91533656e-02, 1.30000000e-02, 1.14603171e+01, 1.14550000e-01, 1.30000000e-02, 1.33333333e-03, 1.26380521e-02, 1.14674074e+01],
                            [4.11400031e-02, -3.79141831e-02, 6.84964646e-03, 5.12000000e-01, 1.48432683e+01, 1.97990000e-01, 5.28000000e-01, 1.59692308e-01, 4.11400031e-02, 1.47993898e+01],
                            [0.02155868, -0.01887666, 0.07962665, 0.238,13.24478049, 0.15344, 0.238, 0.04825641, 0.02155868, 13.22587027],
                            [0.00599973, -0.00599973, 0.00697133, 0.034, 0.63663415,0.019799, 0.034, 0.00594872, 0.00480282, 0.61555439],
                            [0.04266586, -0.04266586, 0.0106888, 0.126, 3.67914634,0.041719, 0.132, 0.02676923, 0.03469959, 3.67052968],
                            [0.00042144, -0.00042144, 0.00109543, 0.004, 0.11331707,0.0021213, 0.006, 0.0014359, 0.00027545, 0.11329803],
                            [0.02663118, -0.02639755, 0.01336405, 0.07, 4.36412195,0.054447, 0.079, 0.025, 0.02663118, 4.359373 ],
                            [0.00730426, -0.00730426, 0.00435895, 0.072, 0.61409756,0.012728, 0.096, 0.00930769, 0.00603133, 0.60116405],
                            [0.01877031, -0.00838677, 0.02480503, 0.068, 2.56031707,0.077075, 0.068, 0.02294872, 0.01877031, 2.4790144 ],
                            [0.01901328, -0.01785507, 0.0129973, 0.155, 2.71221951,0.047376, 0.155, 0.03697436, 0.01901328, 2.68282596],
                            [0.00190969, -0.00179795, 0.00162791, 0.01, 0.06846341,0.0056569, 0.01, 0.00307692, 0.00190969, 0.06774886],
                            [0.02343739, -0.02343739, 0.06043373, 0.17,10.8854878, 0.11172, 0.17, 0.07266667, 0.01923799, 10.87016693],
                            [1.59897956e-02, -1.15451745e-02, 1.01170444e-02, 3.20000000e-01, 1.16454634e+01, 1.37890000e-01, 3.47000000e-01, 8.94615385e-02, 1.59897956e-02, 1.15869935e+01],
                            [0.01774385, -0.01578744, 0.03231943, 0.085, 2.70382927,0.12233, 0.133, 0.03161538, 0.01774385, 2.68325119],
                            [0.02835417, -0.02835417, 0.0188188, 0.111, 3.12563415,0.054447, 0.111, 0.02897436, 0.01213407, 3.17337638],
                            [2.35933619e-04, -1.84870042e-04, 2.60312573e-11, 4.00000000e-03, 8.50975610e-02, 1.41420000e-03, 4.00000000e-03, 8.20512821e-04, 2.35933619e-04, 8.50491997e-02],
                            [0.00684639, -0.0053288, 0.01423898, 0.121, 0.82373171,0.041012, 0.121, 0.01853846, 0.00684639, 0.82149219],
                            [0.00656202, -0.00298516, 0.00525832, 0.04, 0.32917073,0.013435, 0.04, 0.00435897, 0.00656202, 0.32469572],
                            [0.03061193, -0.01830022, 0.02278918, 0.083, 3.852,0.07566, 0.083, 0.02366667, 0.03061193, 3.82389036],
                            [0.03450864, -0.03206468, 0.07235446, 0.354, 4.39680488,0.20506, 0.354, 0.09258974, 0.03450864, 4.35973108],
                            [1.33757941e-03, -1.33757941e-03, 1.81681257e-02, 1.60000000e-02, 6.81234146e+00, 4.73760000e-02, 1.60000000e-02, 5.74358974e-03, 7.88662159e-04, 6.81237822e+00],
                            [0.05408959, -0.05408959, 0.07739115, 0.232,13.12702439, 0.16263, 0.46, 0.12551282, 0.02715891, 13.09196866],
                            [0.01414166, -0.01414166, 0, 0.214, 9.05543902,0, 0.214, 0.01707692, 0.01284695, 9.04209973],
                            [0.00262455, -0.00262455, 0.00092195, 0.021, 0.13329268,0.0035355, 0.021, 0.00338462, 0.00176235, 0.13419267],
                            [0.00766504, -0.00766504, 0.02270081, 0.065, 0.15463415,0.067882, 0.065, 0.01617949, 0.00760269, 0.15422134],
                            [0.01601326, -0.01601326, 0.00953142, 0.044, 4.34658537,0.029698, 0.082, 0.02784615, 0.0152717, 4.34824852],
                            [1.36058722e-04, -1.36058722e-04, 9.39849132e-04, 2.00000000e-03, 1.47902439e-01, 2.12130000e-03, 4.00000000e-03, 3.84615385e-04, 1.20873351e-04, 1.48269290e-01],
                            [0.01692731, -0.01692731, 0.00934515, 0.17, 2.76797561,0.04879, 0.17, 0.03212821, 0.00804809, 2.84223399],
                            [0.00021792, -0.00016886, 0.00077459, 0.002, 0.16504878,0.0014142, 0.004, 0.00020513, 0.00021792, 0.16495962],
                            [0.0025624, -0.0017206, 0.00242901, 0.013, 0.22082927,0.0084853, 0.015, 0.00476923, 0.0025624, 0.21293887],
                            [0.00825234, -0.00825234, 0.00915876, 0.068, 4.30339024,0.028284, 0.073, 0.02058974, 0.00762434, 4.32659455],
                            [0.03004306, -0.02691667, 0.04972124, 0.152,10.16087805, 0.10324, 0.158, 0.07189744, 0.03004306, 10.1129346 ],
                            [0.02155975, -0.02155975, 0.04474389, 0.129, 8.92531707,0.084146, 0.236, 0.07838462, 0.02002646, 8.88551601],
                            [1.14268607e-02, -9.73097720e-03, 7.75027419e-04, 6.40000000e-02, 3.79953659e+00, 2.40420000e-02, 6.40000000e-02, 1.87692308e-02, 1.14268607e-02, 3.80236594e+00],
                            [0.02707888, -0.02707888, 0.01880556, 0.076, 4.34395122,0.053033, 0.076, 0.02764103, 0.02459437, 4.34208839],
                            [0.01727205, -0.01727205, 0.01056655, 0.066, 1.30939024,0.03182, 0.066, 0.0165641, 0.01633014, 1.2967273 ],
                            [0.00877545, -0.00877545, 0.01277828, 0.028, 0.0914878,0.032527, 0.037, 0.00364103, 0.00682708, 0.08295897],
                            [0.00040731, -0.00036169, 0.00082663, 0.006, 0.13621951,0.0021213, 0.006, 0.00174359, 0.00040731, 0.13689296],
                            [0.00145672, -0.00145672, 0.00184393, 0.006, 0.05692683,0.0056569, 0.008, 0.00189744, 0.00103611, 0.05532006],
                            [0.00539013, -0.00539013, 0.01083511, 0.025, 0.60539024,0.02687, 0.052, 0.00476923, 0.00387435, 0.60237517],
                            [0.00576748, -0.00576748, 0.01658765, 0.082, 2.201,0.045962, 0.082, 0.01251282, 0.00554347, 2.2021353 ],
                            [0.00038011, -0.00035004, 0.0011832, 0.006, 0.07460976,0.0028284, 0.006, 0.00097436, 0.00038011, 0.07430141],
                            [0.00232859, -0.00232859, 0.00442728, 0.011, 0.04782927,0.012021, 0.012, 0.00353846, 0.00200706, 0.04406202],
                            [0.03710778, -0.03710778, 0.00465324, 0.076, 4.443,0.03182, 0.086, 0.02310256, 0.03407041, 4.40858239],
                            [1.13293306e-02, -1.13293306e-02, 3.98170020e-02, 2.08000000e-01, 8.79678049e+00, 8.06100000e-02, 2.08000000e-01, 4.19487179e-02, 8.02841512e-03, 8.78487800e+00],
                            [0.00241135, -0.00186458, 0.0089488, 0.059, 1.06817073,0.02192, 0.059, 0.00302564, 0.00241135, 1.06816635],
                            [0.01979542, -0.0079298, 0.02172778, 0.128, 2.58236585,0.062225, 0.128, 0.02112821, 0.01979542, 2.5705713 ],
                            [0.02311331, -0.02311331, 0.01137534, 0.062, 4.10326829,0.048083, 0.076, 0.02253846, 0.02280474, 4.08634796],
                            [0.00020308, -0.00020308, 0.00116188, 0.006, 0.08992683,0.0021213, 0.006, 0.00030769, 0.0001032, 0.09000535],
                            [0.02430024, -0.01198259, 0.05125066, 0.284,10.61465854, 0.10041, 0.284, 0.04492308, 0.02430024, 10.68180335],
                            [0.00456388, -0.00456388, 0.00310646, 0.037, 0.09085366,0.0098995, 0.037, 0.00694872, 0.00379939, 0.09607742],
                            [0.0258999, -0.02057017, 0.03178882, 0.123, 5.97997561,0.086974, 0.123, 0.02774359, 0.0258999, 5.91008859],
                            [0.02995729, -0.02995729, 0.02248653, 0.065, 3.89078049,0.067175, 0.072, 0.02235897, 0.02419343, 3.84855841],
                            [0.03168889, -0.01073342, 0.02388286, 0.091, 4.27865854,0.08061, 0.091, 0.02253846, 0.03168889, 4.27815091],
                            [0.00558419, -0.00558419, 0.00877511, 0.022, 1.34473171,0.030406, 0.026, 0.00953846, 0.00182773, 1.37010789],
                            [0.0002998, -0.0002998, 0.00057734, 0.004, 0.14253659,0.0014142, 0.004, 0.00097436, 0.00028829, 0.14237889],
                            [0.00219221, -0.00094528, 0.01264908, 0.057, 0.64868293,0.027577, 0.057, 0.00864103, 0.00219221, 0.63404688],
                            [0.00025949, -0.00025949, 0.00057734, 0.004, 0.07617073,0.0014142, 0.004, 0.00179487, 0.00015403, 0.07597272],
                            [3.54995192e-04, -3.54995192e-04, 5.16299928e-02, 1.00000000e-02, 2.85024390e-01, 1.28690000e-01, 1.00000000e-02, 2.41025641e-03, 2.75611646e-04, 2.85969148e-01],
                            [0.01468492, -0.01456906, 0.01003006, 0.072, 4.19787805,0.030406, 0.072, 0.02764103, 0.01468492, 4.21230508],
                            [0.02925993, -0.02564169, 0.0243944, 0.069, 3.981,0.081317, 0.069, 0.01961538, 0.02925993, 3.88386281],
                            [0.00033805, -0.00023428, 0.00154918, 0.005, 0.06904878,0.0035355, 0.005, 0.00117949, 0.00033805, 0.06819891],
                            [0.02167887, -0.01716068, 0.03000235, 0.167, 7.68443902,0.068589, 0.176, 0.05779487, 0.02167887, 7.62067289],
                            [0.00860053, -0.00687122, 0.00984125, 0.028, 1.9362439,0.032527, 0.028, 0.01158974, 0.00860053, 1.85928806],
                            [1.62829923e-02, -1.27782612e-02, 5.03488484e-03, 2.93000000e-01, 1.10292927e+01, 9.19240000e-03, 2.93000000e-01, 4.98717949e-02, 1.62829923e-02, 1.10624738e+01],
                            [5.11402880e-02, -5.11402880e-02, 0.00000000e+00, 4.40000000e-02, 2.07410488e+01, 1.55560000e-02, 4.40000000e-02, 1.24102564e-02, 2.17468552e-02, 2.07288498e+01],
                            [0.02048881, -0.01983837, 0.01785776, 0.068, 3.90980488,0.045255, 0.068, 0.02412821, 0.02048881, 3.92262256],
                            [0.00066236, -0.00066236, 0.00180276, 0.008, 0.15141463,0.0049497, 0.008, 0.00161538, 0.00045688, 0.15286961],
                            [0.00060837, -0.00053966, 0.00036514, 0.006, 0.15390244,0.0021213, 0.006, 0.00184615, 0.00060837, 0.1543969 ],
                            [0.00261392, -0.0025949, 0.00945165, 0.046, 1.07970732,0.026163, 0.046, 0.00725641, 0.00261392, 1.0819483 ],
                            [2.21525541e-02, -2.09299114e-02, 4.55533610e-03, 1.75000000e-01, 7.59114634e+00, 6.50540000e-02, 1.75000000e-01, 4.90256410e-02, 2.21525541e-02, 7.58764892e+00],
                            [7.56803574e-03, -5.22375992e-03, 1.39641061e-03, 3.40000000e-02, 1.45278049e+00, 8.48530000e-03, 3.90000000e-02, 8.23076923e-03, 7.56803574e-03, 1.46401853e+00],
                            [0.00042417, -0.00042417, 0.00109543, 0.003, 0.16736585,0.0021213, 0.006, 0.00123077, 0.00031221, 0.16745186],
                            [3.55599268e-04, -3.33958979e-04, 2.60312573e-11, 4.00000000e-03, 1.18829268e-01, 1.41420000e-03, 4.00000000e-03, 1.17948718e-03, 3.55599268e-04, 1.18657803e-01],
                            [8.49163246e-03, -7.36645573e-03, 1.26472005e-03, 4.00000000e-02, 4.19334146e+00, 2.82840000e-02, 8.00000000e-02, 2.64615385e-02, 8.49163246e-03, 4.21478989e+00],
                            [0.04132153, -0.04132153, 0.08233984, 0.491,13.82968293, 0.1789, 0.491, 0.09758974, 0.02614823, 13.93101321],
                            [0.00138337, -0.00104703, 0.00203719, 0.007, 0.09217073,0.006364, 0.007, 0.00258974, 0.00138337, 0.07952256],
                            [0.02013966, -0.02013966, 0.06794402, 0.35,11.46958537, 0.12728, 0.35, 0.09010256, 0.01988961, 11.45768721],
                            [0.00205373, -0.00205373, 0.0030984, 0.015, 0.06936585,0.0084853, 0.017, 0.00394872, 0.00152146, 0.06837444],
                            [0.01040894, -0.01040894, 0.00890233, 0.068, 0.26421951,0.025456, 0.068, 0.00902564, 0.00546137, 0.2587529 ],
                            [8.19684883e-03, -8.19684883e-03, 2.09202670e-02, 6.30000000e-02, 6.51109756e+00, 4.45480000e-02, 6.30000000e-02, 1.66153846e-02, 5.80672162e-03, 6.50965345e+00],
                            [0.00023369, -0.00023369, 0.00246646, 0.007, 0.16326829,0.006364, 0.007, 0.00079487, 0.00019522, 0.1627248 ],
                            [0.00490015, -0.00322078, 0.00861112, 0.016, 1.29187805,0.025456, 0.016, 0.00689744, 0.00490015, 1.27916244],
                            [0.0134886, -0.00766548, 0.0189313, 0.109, 6.21553659,0.041012, 0.109, 0.03533333, 0.0134886, 6.22416714],
                            [0.02813757, -0.01747513, 0.0254467, 0.127, 6.28670732,0.084146, 0.127, 0.03102564, 0.02813757, 6.35501978],
                            [0.02044022, -0.02044022, 0.03488198, 0.176, 8.10602439,0.065761, 0.176, 0.04466667, 0.01777938, 8.12759058],
                            [0.01947497, -0.01946271, 0.05739752, 0.284,10.64578049, 0.10889, 0.296, 0.04564103, 0.01947497, 10.64447668],
                            [0.01922614, -0.01922614, 0.02849813, 0.097, 4.29178049,0.08061, 0.097, 0.02720513, 0.01556252, 4.30987745],
                            [0.00027514, -0.00027514, 0.00073029, 0.004, 0.07358537,0.0014142, 0.004, 0.00153846, 0.0002546, 0.0735262]])

    num_support_vec = [61, 97, 89]

    return {'dual_coef': dual_coef,
            'support_vec': support_vec,
            'intercept': intercept,
            'gamma': gamma,
            'num_support_vec': num_support_vec}


if __name__ == '__main__':

    binary_test_data = np.array([
        [0., 0., 0., 0., 0., 0., 0., 0.14212846, 0.02368808, 0.0580237, 0.06960633, 0.04921911, 0.],
        [0.21073171, 0.794, 0.922, 0.14076923, 0.20974742, -0.32312654, 0.32312654, 0.88741901, 0.17300546, 0.3544437, 0.39891235, 0.5271785, 0.0076014],
        [0.04058537, 0.009, 0.008, 0.00225641, 0.03362015, -0.00420592, 0.00420592, 0.00565685, 0.00235702, 0.00193218, 0.00581951, 0.00861878, 0.00671751],
        [0.07887805, 0.0035, 0.007, 0.00179487, 0.07598638, -0.00096018, 0.00113304, 0.00494975, 0.00235702, 0.00177012, 0.00098742, 0.00128452, 0.00335876],
        [0.10126829, 0.0015, 0.004, 0.00174359, 0.09954269, -0.00034342, 0.00034342, 0.00212132, 0.00153206, 0.00028868, 0.00075829, 0.00064872, 0.00212132]
    ])
    binary_predictions = predict_binary_classifier(binary_test_data)

    import pickle
    f2 = open('/Users/sarataylor/Dev/eda-explorer-public/SVMBinary.p', 'rb')
    s2 = f2.read()
    clf = pickle.loads(s2)

    assert(len([1 for i in range(5) if binary_predictions[i] != clf.predict(binary_test_data)[i]]) == 0)

    # Test multiclass
    test_data = np.array([[3.11141105e-02, -3.11136868e-02, 2.42079822e-02, 7.49220000e-02, 1.15335178e+01, 8.37681119e-02, 7.49220000e-02, 1.03606795e-02, 3.11141105e-02, 1.15205823e+01],
                          [0., 0., 0., 0., 0., 0., 0., 0., 0., 0.],
                          [0.10255266, -0.10255266, 0.03827904, 0.328471, 6.61645195, 0.26352986, 0.406695, 0.02494941, 0.05696297, 7.64941098],
                          [0.1095642, -0.1095642, 0.08589464, 0.113983, 11.49373772, 0.26352986, 0.113983, 0.01375942, 0.03753318, 11.51816541],
                          [0.15404637, -0.08878016, 0.1020834, 0.768917, 11.40673696, 0.28606288, 0.768917, 0.08697605, 0.15404637, 11.46339086]])

    multi_predictions = predict_multiclass_classifier(test_data)

    f2 = open('/Users/sarataylor/Dev/eda-explorer-public/SVMMulticlass.p', 'rb')
    s2 = f2.read()
    clf = pickle.loads(s2)

    assert (len([1 for i in range(5) if multi_predictions[i] != clf.predict(test_data)[i]]) == 0)



