import os
import subprocess
import sys
from importlib import reload
from textwrap import dedent
from unittest.mock import MagicMock, patch

import pytest
from typer.testing import CliRunner

from bec_lib.utils.plugin_manager._constants import ANSWER_KEYS
from bec_lib.utils.plugin_manager._util import (
    _goto_dir,
    existing_data,
    git_stage_files,
    make_commit,
)

# Too complicated for import mechanics - tests "without" BW present should run first, then with
pytestmark = pytest.mark.random_order(disabled=True)


def mocked_app(module_patch):
    with patch.dict(sys.modules, dict(module_patch)):
        from bec_lib.utils.plugin_manager import main

        reload(main)
        yield main._app, module_patch[0][1]

        loaded_package_modules = [
            key
            for key, value in sys.modules.items()
            if "bec_lib.utils.plugin_manager.main" in str(value) or "bec_widgets" in str(value)
        ]
        for key in loaded_package_modules:
            del sys.modules[key]


@pytest.fixture(scope="module")
def app_none():
    # hide bec_widgets when running these tests
    yield from mocked_app([("bec_widgets", None)])


@pytest.fixture(scope="module")
def app_with_bw():
    # hide bec_widgets when running these tests
    mock_module = MagicMock()
    yield from mocked_app(
        [("bec_widgets.utils.bec_plugin_manager.edit_ui", mock_module), ("bec_widgets", None)]
    )


@pytest.fixture(scope="module")
def create_app():
    # hide bec_widgets when running these tests
    with patch.dict(sys.modules, {"bec_widgets": None}):
        from bec_lib.utils.plugin_manager import create

        reload(create)
        yield create._app


@pytest.fixture
def runner():
    return CliRunner()


def test_plugin_manager_doesnt_add_missing_command(app_none):
    app, _ = app_none
    assert len(app.registered_commands) == 0


def test_plugin_manager_create_subapp_needs_extra_command(runner, app_none):
    app, _ = app_none
    result = runner.invoke(app, ["create"])
    assert result.exit_code == 2
    assert "Missing command." in result.output


def test_plugin_manager_create_subapp_wo_bw(runner, create_app):
    result = runner.invoke(create_app, ["scan", "test"])
    assert result.exit_code == 1
    assert type(result.exception) is NotImplementedError

    result = runner.invoke(create_app, ["device", "test"])
    assert result.exit_code == 1
    assert type(result.exception) is NotImplementedError


def test_plugin_manager_adds_found_command(runner, app_with_bw):
    app, mock_edit_ui = app_with_bw
    assert len(app.registered_commands) == 1
    result = runner.invoke(app, ["edit-ui", "widget_to_edit"])
    assert result.exit_code == 0
    mock_edit_ui.open_and_watch_ui_editor.assert_called_with("widget_to_edit")


def test_existing_data(tmp_path):
    with open(tmp_path / ".copier-answers.yml", "w") as f:
        f.write(
            dedent(
                """
                # Do not edit this file!
                # It is needed to track the repo template version, and editing may break things.
                # This file will be overwritten by copier on template updates.

                _commit: v1.0.5
                _src_path: https://github.com/bec-project/plugin_copier_template.git
                make_commit: true
                project_name: bec_testing_plugin
                widget_plugins_input:
                -   name: example_widget_plugin
                    use_ui: true
                """
            )
        )
    result = existing_data(tmp_path, [ANSWER_KEYS.WIDGETS])
    assert result == {"widget_plugins_input": [{"name": "example_widget_plugin", "use_ui": True}]}


def test_goto_dir(tmp_path):
    os.makedirs(tmp_path / "test_dir")
    original_path = os.getcwd()
    with _goto_dir(tmp_path / "test_dir"):
        assert os.getcwd() == str(tmp_path / "test_dir")
    assert os.getcwd() == original_path


@pytest.fixture
def git_repo(tmp_path):
    with _goto_dir(tmp_path):
        subprocess.run(["git", "init", "-b", "main"])
        subprocess.run(["git", "config", "user.email", "test"])
        subprocess.run(["git", "config", "user.name", "test"])
        with open(tmp_path / "test.txt", "w") as f:
            f.write("test\n")
        with open(tmp_path / "test2.txt", "w") as f:
            f.write("test\n")
        yield tmp_path


def test_git_stage_files(git_repo):

    git_stage_files(git_repo, ["test.txt"])

    result = subprocess.run(["git", "diff", "--name-only", "--cached"], stdout=subprocess.PIPE)
    assert result.stdout.decode() == "test.txt\n"

    git_stage_files(git_repo)
    result = subprocess.run(["git", "diff", "--name-only", "--cached"], stdout=subprocess.PIPE)
    assert result.stdout.decode() == "test.txt\ntest2.txt\n"


def test_make_commit(git_repo):
    git_stage_files(git_repo)
    make_commit(git_repo, "test commit")
    result = subprocess.run(["git", "log"], stdout=subprocess.PIPE)
    assert "test commit" in result.stdout.decode()
