"""
Maps all the MQTT topics to either attributes or metrics.
"""

from typing import List
from .constants import MetricKind, MetricNature, MetricType, ValueType, RangeType
from .data_classes import TopicDescriptor
from ._victron_enums import (
    AcActiveInputSource,
    ChargeSchedule,
    DESSMode,
    DESSReactiveStrategy,
    DESSStrategy,
    DigitalInputInputState,
    DigitalInputState,
    DigitalInputType,
    ESSMode,
    ErrorCode,
    FluidType,
    GeneratorRunningByConditionCode,
    InverterMode,
    GenericOnOff,
    ChargerMode,
    EvChargerMode,
    MppOperationMode,
    PhoenixInverterMode,
    State,
    TemperatureStatus,
    TemperatureType,
    DESSErrorCode,
    DESSRestrictions,
    GenericAlarmEnum,
    ESSState,
    ESSModeHub4,
    ActiveInputEnum,
)

# Good sources for topics is:
# https://github.com/victronenergy/venus/wiki/dbus
# https://view.officeapps.live.com/op/view.aspx?src=https%3A%2F%2Fwww.victronenergy.com%2Fupload%2Fdocuments%2FCCGX-Modbus-TCP-register-list-3.60.xlsx
# mqtt-explorer

topics: List[TopicDescriptor] = [
    # generic device attributes
    TopicDescriptor(
        topic="N/{installation_id}/vebus/{device_id}/Devices/0/SerialNumber",
        message_type=MetricKind.ATTRIBUTE,
        short_id="serial_number",
        value_type=ValueType.STRING,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/{device_type}/{device_id}/CustomName",
        message_type=MetricKind.ATTRIBUTE,
        short_id="custom_name",
        value_type=ValueType.STRING,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/{device_type}/{device_id}/FirmwareVersion",
        message_type=MetricKind.ATTRIBUTE,
        short_id="firmware_version",
        value_type=ValueType.STRING,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/{device_type}/{device_id}/Manufacturer",
        message_type=MetricKind.ATTRIBUTE,
        short_id="manufacturer",
        value_type=ValueType.STRING,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/{device_type}/{device_id}/ProductId",
        message_type=MetricKind.ATTRIBUTE,
        short_id="victron_productid",
        value_type=ValueType.INT,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/{device_type}/{device_id}/ProductName",
        message_type=MetricKind.ATTRIBUTE,
        short_id="model",
        value_type=ValueType.STRING,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/{device_type}/{device_id}/Serial",
        message_type=MetricKind.ATTRIBUTE,
        short_id="serial_number",
        value_type=ValueType.STRING,
    ),
    # all other topics (sorted alphabetically)
    # ACLoad topics
    TopicDescriptor(
        topic="N/{installation_id}/acload/{device_id}/Ac/Current",
        message_type=MetricKind.SENSOR,
        short_id="acload_current",
        name="Load current",
        metric_type=MetricType.CURRENT,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/acload/{device_id}/Ac/Energy/Forward",
        message_type=MetricKind.SENSOR,
        short_id="acload_energy_forward",
        name="Load consumption",
        metric_type=MetricType.ENERGY,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/acload/{device_id}/Ac/Frequency",
        message_type=MetricKind.SENSOR,
        short_id="acload_frequency",
        name="Load frequency",
        metric_type=MetricType.FREQUENCY,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/acload/{device_id}/Ac/Power",
        message_type=MetricKind.SENSOR,
        short_id="acload_power",
        name="Load power",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/acload/{device_id}/Ac/Voltage",
        message_type=MetricKind.SENSOR,
        short_id="acload_voltage",
        name="Load voltage",
        metric_type=MetricType.VOLTAGE,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/acload/{device_id}/Ac/{phase}/Current",
        message_type=MetricKind.SENSOR,
        short_id="acload_current_{phase}",
        name="Load current on {phase}",
        metric_type=MetricType.CURRENT,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/acload/{device_id}/Ac/{phase}/Energy/Forward",
        message_type=MetricKind.SENSOR,
        short_id="acload_energy_forward_{phase}",
        name="Load consumption on {phase}",
        metric_type=MetricType.ENERGY,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/acload/{device_id}/Ac/{phase}/Power",
        message_type=MetricKind.SENSOR,
        short_id="acload_power_{phase}",
        name="Load power on {phase}",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/acload/{device_id}/Ac/{phase}/Voltage",
        message_type=MetricKind.SENSOR,
        short_id="acload_voltage_{phase}",
        name="Load voltage on {phase}",
        metric_type=MetricType.VOLTAGE,
    ),
    # Alternator topics
    TopicDescriptor(
        topic="N/{installation_id}/alternator/{device_id}/Dc/0/Current",
        message_type=MetricKind.SENSOR,
        short_id="alternator_dc_current",
        name="DC Output Current",
        metric_type=MetricType.CURRENT,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/alternator/{device_id}/Dc/0/Power",
        message_type=MetricKind.SENSOR,
        short_id="alternator_dc_power",
        name="DC Output Power",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/alternator/{device_id}/Dc/0/Voltage",
        message_type=MetricKind.SENSOR,
        short_id="alternator_dc_voltage",
        name="DC Output Voltage",
        metric_type=MetricType.VOLTAGE,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/alternator/{device_id}/Dc/In/I",
        message_type=MetricKind.SENSOR,
        short_id="alternator_input_current",
        name="Input Current",
        metric_type=MetricType.CURRENT,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/alternator/{device_id}/Dc/In/P",
        message_type=MetricKind.SENSOR,
        short_id="alternator_input_power",
        name="Input Power",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/alternator/{device_id}/Dc/In/V",
        message_type=MetricKind.SENSOR,
        short_id="alternator_input_voltage",
        name="Input Voltage",
        metric_type=MetricType.VOLTAGE,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/alternator/{device_id}/Mode",
        message_type=MetricKind.SWITCH,
        short_id="alternator_mode",
        name="Alternator Mode",
        value_type=ValueType.ENUM,
        enum=ChargerMode,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/alternator/{device_id}/State",
        message_type=MetricKind.SENSOR,
        short_id="alternator_state",
        name="Alternator State",
        value_type=ValueType.ENUM,
        enum=State,
    ),
    # Battery topics
    TopicDescriptor(
        topic="N/{installation_id}/battery/{device_id}/Alarms/CellImbalance",
        message_type=MetricKind.SENSOR,
        short_id="battery_cell_imbalance",
        name="Battery cell imbalance",
        value_type=ValueType.ENUM,
        enum=GenericAlarmEnum,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/battery/{device_id}/Alarms/HighChargeCurrent",
        message_type=MetricKind.SENSOR,
        short_id="battery_high_charge_current",
        name="Battery high charge current",
        value_type=ValueType.ENUM,
        enum=GenericAlarmEnum,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/battery/{device_id}/Alarms/HighChargeTemperature",
        message_type=MetricKind.SENSOR,
        short_id="battery_high_charge_temperature",
        name="Battery high charge temperature",
        value_type=ValueType.ENUM,
        enum=GenericAlarmEnum,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/battery/{device_id}/Alarms/HighDischargeCurrent",
        message_type=MetricKind.SENSOR,
        short_id="battery_high_discharge_current",
        name="Battery high discharge current",
        value_type=ValueType.ENUM,
        enum=GenericAlarmEnum,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/battery/{device_id}/Alarms/InternalFailure",
        message_type=MetricKind.SENSOR,
        short_id="battery_internal_failure",
        name="Battery internal failure",
        value_type=ValueType.ENUM,
        enum=GenericAlarmEnum,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/battery/{device_id}/Alarms/LowCellVoltage",
        message_type=MetricKind.SENSOR,
        short_id="battery_low_cell_voltage",
        name="Battery low cell voltage",
        value_type=ValueType.ENUM,
        enum=GenericAlarmEnum,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/battery/{device_id}/Alarms/LowChargeTemperature",
        message_type=MetricKind.SENSOR,
        short_id="battery_low_charge_temperature",
        name="Battery low charge temperature",
        value_type=ValueType.ENUM,
        enum=GenericAlarmEnum,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/battery/{device_id}/Capacity",
        message_type=MetricKind.SENSOR,
        short_id="battery_capacity",
        name="Battery capacity",
        metric_type=MetricType.ELECTRIC_STORAGE_CAPACITY,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/battery/{device_id}/ConsumedAmphours",
        message_type=MetricKind.SENSOR,
        short_id="battery_consumed_amphours",
        name="Battery consumed amp-hours",
        metric_type=MetricType.ELECTRIC_STORAGE_CAPACITY,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/battery/{device_id}/Dc/0/Current",
        message_type=MetricKind.SENSOR,
        short_id="battery_current",
        name="DC Bus current",
        metric_type=MetricType.CURRENT,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/battery/{device_id}/Dc/0/Power",
        message_type=MetricKind.SENSOR,
        short_id="battery_power",
        name="Battery power",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/battery/{device_id}/Dc/0/Temperature",
        message_type=MetricKind.SENSOR,
        short_id="battery_temperature",
        name="Battery temperature",
        metric_type=MetricType.TEMPERATURE,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/battery/{device_id}/Dc/0/Voltage",
        message_type=MetricKind.SENSOR,
        short_id="battery_voltage",
        name="DC Bus voltage",
        metric_type=MetricType.VOLTAGE,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/battery/{device_id}/Dc/1/Voltage",
        message_type=MetricKind.SENSOR,
        short_id="auxiliary_battery_voltage",
        name="Auxiliary battery voltage",
        metric_type=MetricType.VOLTAGE,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/battery/{device_id}/History/AutomaticSyncs",
        message_type=MetricKind.SENSOR,
        short_id="battery_automatic_syncs",
        name="Automatic syncs",
        unit_of_measurement="count",
        metric_type=MetricType.NONE,
        metric_nature=MetricNature.CUMULATIVE,
        value_type=ValueType.INT,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/battery/{device_id}/History/ChargedEnergy",
        message_type=MetricKind.SENSOR,
        short_id="battery_charged_energy",
        name="Charged energy",
        metric_type=MetricType.ENERGY,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/battery/{device_id}/History/DischargedEnergy",
        message_type=MetricKind.SENSOR,
        short_id="battery_discharged_energy",
        name="Discharged energy",
        metric_type=MetricType.ENERGY,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/battery/{device_id}/Info/ChargeMode",
        message_type=MetricKind.SENSOR,
        short_id="charge_mode",
        name="Battery charge mode",
        value_type=ValueType.STRING,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/battery/{device_id}/Info/MaxChargeCurrent",
        message_type=MetricKind.SENSOR,
        short_id="battery_max_charge_current",
        name="Battery maximum charge current allowed",
        metric_type=MetricType.CURRENT,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/battery/{device_id}/Info/MaxChargeVoltage",
        message_type=MetricKind.SENSOR,
        short_id="battery_max_charge_voltage",
        name="Battery maximum charging voltage allowed",
        metric_type=MetricType.VOLTAGE,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/battery/{device_id}/Info/MaxDischargeCurrent",
        message_type=MetricKind.SENSOR,
        short_id="battery_max_discharge_current",
        name="Battery maximum discharge current allowed",
        metric_type=MetricType.CURRENT,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/battery/{device_id}/InstalledCapacity",
        message_type=MetricKind.SENSOR,
        short_id="battery_installed_capacity",
        name="Battery installed capacity",
        metric_type=MetricType.ELECTRIC_STORAGE_CAPACITY,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/battery/{device_id}/Soc",
        message_type=MetricKind.SENSOR,
        short_id="battery_soc",
        name="Battery charge",
        metric_type=MetricType.ELECTRIC_STORAGE_PERCENTAGE,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/battery/{device_id}/Soh",
        message_type=MetricKind.SENSOR,
        short_id="battery_soh",
        name="Battery State of Health",
        unit_of_measurement="%",
        metric_type=MetricType.PERCENTAGE,
        value_type=ValueType.FLOAT,
        precision=1,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/battery/{device_id}/System/MaxCellTemperature",
        message_type=MetricKind.SENSOR,
        short_id="battery_max_cell_temperature",
        name="Battery maximum cell temperature",
        metric_type=MetricType.TEMPERATURE,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/battery/{device_id}/System/MaxCellVoltage",
        message_type=MetricKind.SENSOR,
        short_id="battery_max_cell_voltage",
        name="Battery maximum cell voltage",
        metric_type=MetricType.VOLTAGE,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/battery/{device_id}/System/MaxVoltageCellId",
        message_type=MetricKind.SENSOR,
        short_id="battery_max_voltage_cell_id",
        name="Battery maximum voltage cell ID",
        value_type=ValueType.STRING,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/battery/{device_id}/System/MinCellTemperature",
        message_type=MetricKind.SENSOR,
        short_id="battery_min_cell_temperature",
        name="Battery minimum cell temperature",
        metric_type=MetricType.TEMPERATURE,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/battery/{device_id}/System/MinCellVoltage",
        message_type=MetricKind.SENSOR,
        short_id="battery_min_cell_voltage",
        name="Battery minimum cell voltage",
        metric_type=MetricType.VOLTAGE,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/battery/{device_id}/System/MinVoltageCellId",
        message_type=MetricKind.SENSOR,
        short_id="battery_min_voltage_cell_id",
        name="Battery minimum voltage cell ID",
        value_type=ValueType.STRING,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/battery/{device_id}/System/NrOfModulesBlockingCharge",
        message_type=MetricKind.SENSOR,
        short_id="battery_nr_modules_blocking_charge",
        name="Battery number of modules blocking charge",
        value_type=ValueType.INT,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/battery/{device_id}/System/NrOfModulesBlockingDischarge",
        message_type=MetricKind.SENSOR,
        short_id="battery_nr_modules_blocking_discharge",
        name="Battery number of modules blocking discharge",
        value_type=ValueType.INT,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/battery/{device_id}/System/NrOfModulesOffline",
        message_type=MetricKind.SENSOR,
        short_id="battery_nr_modules_offline",
        name="Battery number of modules offline",
        value_type=ValueType.INT,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/battery/{device_id}/System/NrOfModulesOnline",
        message_type=MetricKind.SENSOR,
        short_id="battery_nr_modules_online",
        name="Battery number of modules online",
        value_type=ValueType.INT,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/battery/{device_id}/TimeToGo",
        message_type=MetricKind.SENSOR,
        short_id="battery_time_to_go",
        name="Battery time to go",
        unit_of_measurement="s",
        metric_type=MetricType.TIME,
        metric_nature=MetricNature.INSTANTANEOUS,
        value_type=ValueType.INT,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/battery/{device_id}/Voltages/Cell{cell_id(1-16)}",
        message_type=MetricKind.SENSOR,
        short_id="battery_cell_{cell_id}_voltage",
        name="Battery cell {cell_id} voltage",
        metric_type=MetricType.VOLTAGE,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/battery/{device_id}/Voltages/Diff",
        message_type=MetricKind.SENSOR,
        short_id="battery_cell_voltage_deviation",
        name="Battery cell voltage deviation",
        metric_type=MetricType.VOLTAGE,
    ),
    # DC Load topics
    TopicDescriptor(
        topic="N/{installation_id}/dcload/{device_id}/Dc/0/Current",
        message_type=MetricKind.SENSOR,
        short_id="dcload_current",
        name="DC load current",
        metric_type=MetricType.CURRENT,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/dcload/{device_id}/Dc/0/Power",
        message_type=MetricKind.SENSOR,
        short_id="dcload_power",
        name="DC load power",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/dcload/{device_id}/Dc/0/Voltage",
        message_type=MetricKind.SENSOR,
        short_id="dcload_voltage",
        name="DC load voltage",
        metric_type=MetricType.VOLTAGE,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/dcsystem/{device_id}/Dc/0/Current",
        message_type=MetricKind.SENSOR,
        short_id="dcsystem_current",
        name="DC System Current",
        metric_type=MetricType.CURRENT,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/dcsystem/{device_id}/Dc/0/Power",
        message_type=MetricKind.SENSOR,
        short_id="dcsystem_power",
        name="DC System Power",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/dcsystem/{device_id}/Dc/0/Voltage",
        message_type=MetricKind.SENSOR,
        short_id="dcsystem_voltage",
        name="DC System Voltage",
        metric_type=MetricType.VOLTAGE,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/dcsystem/{device_id}/Dc/1/Voltage",
        message_type=MetricKind.SENSOR,
        short_id="dcsystem_aux_voltage",
        name="DC System Auxiliary Voltage",
        metric_type=MetricType.VOLTAGE,
    ),
    # Digital Input topics
    TopicDescriptor(
        topic="N/{installation_id}/digitalinput/{device_id}/Alarm",
        message_type=MetricKind.SENSOR,
        short_id="digitalinput_alarm",
        name="Digital input alarm",
        value_type=ValueType.ENUM,
        enum=GenericAlarmEnum,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/digitalinput/{device_id}/InputState",
        message_type=MetricKind.SENSOR,
        short_id="digitalinput_input_state_raw",
        name="Digital input raw state",
        value_type=ValueType.ENUM,
        enum=DigitalInputInputState,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/digitalinput/{device_id}/Settings/InvertTranslation",
        message_type=MetricKind.SWITCH,
        short_id="digitalinput_settings_invert_translation",
        name="Digital input settings invert translation",
        value_type=ValueType.ENUM,
        enum=GenericOnOff,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/digitalinput/{device_id}/State",
        message_type=MetricKind.SENSOR,
        short_id="digitalinput_state",
        name="Digital input state",
        value_type=ValueType.ENUM,
        enum=DigitalInputState,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/digitalinput/{device_id}/Type",
        message_type=MetricKind.SENSOR,
        short_id="digitalinput_type",
        name="Digital input type",
        value_type=ValueType.ENUM,
        enum=DigitalInputType,
    ),
    # EV Charger topics
    TopicDescriptor(
        topic="N/{installation_id}/evcharger/{device_id}/Ac/{phase}/Power",
        message_type=MetricKind.SENSOR,
        short_id="evcharger_power_{phase}",
        name="EV charger power {phase}",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/evcharger/{device_id}/Connected",
        message_type=MetricKind.BINARY_SENSOR,
        short_id="evcharger_connected",
        name="EV charger connected",
        value_type=ValueType.ENUM,
        enum=GenericOnOff,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/evcharger/{device_id}/Current",
        message_type=MetricKind.SENSOR,
        short_id="evcharger_current",
        name="EV charger current",
        metric_type=MetricType.CURRENT,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/evcharger/{device_id}/Mode",
        message_type=MetricKind.SELECT,
        short_id="evcharger_mode",
        name="EV charger mode",
        value_type=ValueType.ENUM,
        enum=EvChargerMode,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/evcharger/{device_id}/SetCurrent",
        message_type=MetricKind.NUMBER,
        short_id="evcharger_set_current",
        name="EV charger set current",
        metric_type=MetricType.CURRENT,
        value_type=ValueType.INT,
        min=0,
        max=32,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/evcharger/{device_id}/StartStop",
        message_type=MetricKind.SWITCH,
        short_id="evcharger_charge",
        name="EV charger charge",
        value_type=ValueType.ENUM,
        enum=GenericOnOff,
    ),
    # Generator topics
    TopicDescriptor(
        topic="N/{installation_id}/generator/{device_id}/AccumulatedRuntime",
        message_type=MetricKind.SENSOR,
        short_id="generator_total_runtime",
        name="Generator total runtime",
        unit_of_measurement="h",
        metric_type=MetricType.TIME,
        metric_nature=MetricNature.CUMULATIVE,
        value_type=ValueType.INT_SECONDS_TO_HOURS,
        precision=1,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/generator/{device_id}/AutoStartEnabled",
        message_type=MetricKind.SWITCH,
        short_id="generator_autorun",
        name="Generator auto start enabled",
        value_type=ValueType.ENUM,
        enum=GenericOnOff,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/generator/{device_id}/ManualStart",
        message_type=MetricKind.SWITCH,
        short_id="generator_manual_start",
        name="Generator manual start",
        value_type=ValueType.ENUM,
        enum=GenericOnOff,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/generator/{device_id}/RunningByConditionCode",
        message_type=MetricKind.SENSOR,
        short_id="generator_run_state",
        name="Generator run state",
        metric_type=MetricType.NONE,
        value_type=ValueType.ENUM,
        enum=GeneratorRunningByConditionCode,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/generator/{device_id}/ServiceCounter",
        message_type=MetricKind.SENSOR,
        short_id="generator_service_counter",
        name="Generator service counter",
        unit_of_measurement="h",
        metric_type=MetricType.TIME,
        metric_nature=MetricNature.CUMULATIVE,
        value_type=ValueType.INT_SECONDS_TO_HOURS,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/generator/{device_id}/TodayRuntime",
        message_type=MetricKind.SENSOR,
        short_id="generator_today_runtime",
        name="Generator today runtime",
        unit_of_measurement="h",
        metric_type=MetricType.TIME,
        metric_nature=MetricNature.INSTANTANEOUS,
        value_type=ValueType.INT_SECONDS_TO_HOURS,
        precision=1,
    ),
    # GPS topics
    TopicDescriptor(
        topic="N/{installation_id}/gps/{device_id}/Altitude",
        message_type=MetricKind.SENSOR,
        short_id="gps_altitude",
        name="GPS Altitude",
        metric_nature=MetricNature.INSTANTANEOUS,
        value_type=ValueType.FLOAT,
        precision=2,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/gps/{device_id}/Connected",
        message_type=MetricKind.BINARY_SENSOR,
        short_id="gps_connected",
        name="GPS Connected",
        value_type=ValueType.ENUM,
        enum=GenericOnOff,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/gps/{device_id}/Course",
        message_type=MetricKind.SENSOR,
        short_id="gps_course",
        name="GPS Course",
        metric_nature=MetricNature.INSTANTANEOUS,
        metric_type=MetricType.HEADING,
        value_type=ValueType.FLOAT,
        precision=2,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/gps/{device_id}/Fix",
        message_type=MetricKind.BINARY_SENSOR,
        short_id="gps_fix",
        name="GPS Fix",
        value_type=ValueType.ENUM,
        enum=GenericOnOff,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/gps/{device_id}/NrOfSatellites",
        message_type=MetricKind.SENSOR,
        short_id="gps_nrofsatellites",
        name="GPS Nr Of Satellites",
        metric_nature=MetricNature.INSTANTANEOUS,
        value_type=ValueType.INT,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/gps/{device_id}/Position/Latitude",
        message_type=MetricKind.SENSOR,
        short_id="gps_latitude",
        name="GPS Latitude",
        metric_nature=MetricNature.INSTANTANEOUS,
        metric_type=MetricType.LOCATION,
        value_type=ValueType.FLOAT,
        precision=None,  # Use full precision of GPS device
    ),
    TopicDescriptor(
        topic="N/{installation_id}/gps/{device_id}/Position/Longitude",
        message_type=MetricKind.SENSOR,
        short_id="gps_longitude",
        name="GPS Longitude",
        metric_nature=MetricNature.INSTANTANEOUS,
        metric_type=MetricType.LOCATION,
        value_type=ValueType.FLOAT,
        precision=None,  # Use full precision of GPS device
    ),
    TopicDescriptor(
        topic="N/{installation_id}/gps/{device_id}/Speed",
        message_type=MetricKind.SENSOR,
        short_id="gps_speed",
        name="GPS Speed",
        metric_nature=MetricNature.INSTANTANEOUS,
        unit_of_measurement="m/s",
        metric_type=MetricType.SPEED,
        value_type=ValueType.FLOAT,
        precision=2,
    ),
    # Grid topics
    TopicDescriptor(
        topic="N/{installation_id}/grid/{device_id}/Ac/Current",
        message_type=MetricKind.SENSOR,
        short_id="grid_current",
        name="Grid current",
        metric_type=MetricType.CURRENT,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/grid/{device_id}/Ac/Energy/Forward",
        message_type=MetricKind.SENSOR,
        short_id="grid_energy_forward",
        name="Grid consumption",
        metric_type=MetricType.ENERGY,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/grid/{device_id}/Ac/Energy/Reverse",
        message_type=MetricKind.SENSOR,
        short_id="grid_energy_reverse",
        name="Grid feed-in",
        metric_type=MetricType.ENERGY,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/grid/{device_id}/Ac/Frequency",
        message_type=MetricKind.SENSOR,
        short_id="grid_frequency",
        name="Grid frequency",
        metric_type=MetricType.FREQUENCY,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/grid/{device_id}/Ac/N/Current",
        message_type=MetricKind.SENSOR,
        short_id="grid_current_n",
        name="Grid current on N",
        metric_type=MetricType.CURRENT,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/grid/{device_id}/Ac/PENVoltage",
        message_type=MetricKind.SENSOR,
        short_id="grid_voltage_pen",
        name="Grid voltage on PEN",
        metric_type=MetricType.VOLTAGE,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/grid/{device_id}/Ac/Power",
        message_type=MetricKind.SENSOR,
        short_id="grid_power",
        name="Grid power",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/grid/{device_id}/Ac/Voltage",
        message_type=MetricKind.SENSOR,
        short_id="grid_voltage",
        name="Grid voltage",
        metric_type=MetricType.VOLTAGE,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/grid/{device_id}/Ac/{phase}/Current",
        message_type=MetricKind.SENSOR,
        short_id="grid_current_{phase}",
        name="Grid current on {phase}",
        metric_type=MetricType.CURRENT,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/grid/{device_id}/Ac/{phase}/Energy/Forward",
        message_type=MetricKind.SENSOR,
        short_id="grid_energy_forward_{phase}",
        name="Grid consumption on {phase}",
        metric_type=MetricType.ENERGY,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/grid/{device_id}/Ac/{phase}/Energy/Reverse",
        message_type=MetricKind.SENSOR,
        short_id="grid_energy_reverse_{phase}",
        name="Grid feed-in on {phase}",
        metric_type=MetricType.ENERGY,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/grid/{device_id}/Ac/{phase}/Power",
        message_type=MetricKind.SENSOR,
        short_id="grid_power_{phase}",
        name="Grid power on {phase}",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/grid/{device_id}/Ac/{phase}/Voltage",
        message_type=MetricKind.SENSOR,
        short_id="grid_voltage_{phase}",
        name="Grid voltage on {phase}",
        metric_type=MetricType.VOLTAGE,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/grid/{device_id}/Ac/{phase}/VoltageLineToLine",
        message_type=MetricKind.SENSOR,
        short_id="grid_voltage_{phase}_{next_phase}",
        name="Grid voltage {phase} to {next_phase}",
        metric_type=MetricType.VOLTAGE,
    ),
    # GX system heartbeat
    TopicDescriptor(
        topic="N/{installation_id}/heartbeat",
        message_type=MetricKind.SENSOR,
        short_id="system_heartbeat",
        name="GX system heartbeat",
        value_type=ValueType.INT,
    ),
    # Heatpump topics
    TopicDescriptor(
        topic="N/{installation_id}/heatpump/{device_id}/Ac/Current",
        message_type=MetricKind.SENSOR,
        short_id="heatpump_current",
        name="heatpump current",
        metric_type=MetricType.CURRENT,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/heatpump/{device_id}/Ac/Energy/Forward",
        message_type=MetricKind.SENSOR,
        short_id="heatpump_energy_forward",
        name="heatpump consumption",
        metric_type=MetricType.ENERGY,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/heatpump/{device_id}/Ac/Frequency",
        message_type=MetricKind.SENSOR,
        short_id="heatpump_frequency",
        name="heatpump frequency",
        metric_type=MetricType.FREQUENCY,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/heatpump/{device_id}/Ac/Power",
        message_type=MetricKind.SENSOR,
        short_id="heatpump_power",
        name="heatpump power",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/heatpump/{device_id}/Ac/Voltage",
        message_type=MetricKind.SENSOR,
        short_id="heatpump_voltage",
        name="heatpump voltage",
        metric_type=MetricType.VOLTAGE,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/heatpump/{device_id}/Ac/{phase}/Current",
        message_type=MetricKind.SENSOR,
        short_id="heatpump_current_{phase}",
        name="heatpump current on {phase}",
        metric_type=MetricType.CURRENT,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/heatpump/{device_id}/Ac/{phase}/Energy/Forward",
        message_type=MetricKind.SENSOR,
        short_id="heatpump_energy_forward_{phase}",
        name="heatpump consumption on {phase}",
        metric_type=MetricType.ENERGY,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/heatpump/{device_id}/Ac/{phase}/Power",
        message_type=MetricKind.SENSOR,
        short_id="heatpump_power_{phase}",
        name="heatpump power on {phase}",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/heatpump/{device_id}/Ac/{phase}/Voltage",
        message_type=MetricKind.SENSOR,
        short_id="heatpump_voltage_{phase}",
        name="heatpump voltage on {phase}",
        metric_type=MetricType.VOLTAGE,
    ),
    # Inverter topics (Like Phoenix)
    TopicDescriptor(
        topic="N/{installation_id}/inverter/{device_id}/Ac/Out/{phase}/I",
        message_type=MetricKind.SENSOR,
        short_id="inverter_output_current_{phase}",
        name="Inverter output current {phase}",
        metric_type=MetricType.CURRENT,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/inverter/{device_id}/Ac/Out/{phase}/P",
        message_type=MetricKind.SENSOR,
        short_id="inverter_output_power_{phase}",
        name="Inverter output power {phase}",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/inverter/{device_id}/Ac/Out/{phase}/S",
        message_type=MetricKind.SENSOR,
        short_id="inverter_output_apparent_power_{phase}",
        name="Inverter output apparent power {phase}",
        metric_type=MetricType.APPARENT_POWER,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/inverter/{device_id}/Ac/Out/{phase}/V",
        message_type=MetricKind.SENSOR,
        short_id="inverter_output_voltage_{phase}",
        name="Inverter output voltage {phase}",
        metric_type=MetricType.VOLTAGE,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/inverter/{device_id}/Alarms/HighTemperature",
        message_type=MetricKind.BINARY_SENSOR,
        short_id="inverter_alarm_high_temperature",
        name="Inverter alarm HighTemperature",
        value_type=ValueType.ENUM,
        enum=GenericOnOff,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/inverter/{device_id}/Alarms/HighVoltage",
        message_type=MetricKind.BINARY_SENSOR,
        short_id="inverter_alarm_high_voltage",
        name="Inverter alarm HighVoltage",
        value_type=ValueType.ENUM,
        enum=GenericOnOff,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/inverter/{device_id}/Alarms/HighVoltageAcOut",
        message_type=MetricKind.BINARY_SENSOR,
        short_id="inverter_alarm_high_voltage_ac_out",
        name="Inverter alarm HighVoltageAcOut",
        value_type=ValueType.ENUM,
        enum=GenericOnOff,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/inverter/{device_id}/Alarms/LowTemperature",
        message_type=MetricKind.BINARY_SENSOR,
        short_id="inverter_alarm_low_temperature",
        name="Inverter alarm LowTemperature",
        value_type=ValueType.ENUM,
        enum=GenericOnOff,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/inverter/{device_id}/Alarms/LowVoltage",
        message_type=MetricKind.BINARY_SENSOR,
        short_id="inverter_alarm_low_voltage",
        name="Inverter alarm LowVoltage",
        value_type=ValueType.ENUM,
        enum=GenericOnOff,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/inverter/{device_id}/Alarms/LowVoltageAcOut",
        message_type=MetricKind.BINARY_SENSOR,
        short_id="inverter_alarm_low_voltage_ac_out",
        name="Inverter alarm LowVoltageAcOut",
        value_type=ValueType.ENUM,
        enum=GenericOnOff,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/inverter/{device_id}/Alarms/Overload",
        message_type=MetricKind.BINARY_SENSOR,
        short_id="inverter_alarm_overload",
        name="Inverter alarm Overload",
        value_type=ValueType.ENUM,
        enum=GenericOnOff,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/inverter/{device_id}/Alarms/Ripple",
        message_type=MetricKind.BINARY_SENSOR,
        short_id="inverter_alarm_ripple",
        name="Inverter alarm Ripple",
        value_type=ValueType.ENUM,
        enum=GenericOnOff,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/inverter/{device_id}/Mode",
        message_type=MetricKind.SELECT,
        short_id="inverter_mode",
        name="Inverter mode",
        value_type=ValueType.ENUM,
        enum=PhoenixInverterMode,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/inverter/{device_id}/State",
        message_type=MetricKind.SENSOR,
        short_id="inverter_state",
        name="Inverter state",
        value_type=ValueType.ENUM,
        enum=State,
    ),
    # Multi RS Solar topics
    TopicDescriptor(
        topic="N/{installation_id}/multi/{device_id}/Ac/ActiveIn/ActiveInput",
        message_type=MetricKind.SENSOR,
        short_id="multi_active_input",
        name="Active AC Input",
        value_type=ValueType.ENUM,
        enum=ActiveInputEnum,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/multi/{device_id}/Ac/In/1/{phase}/I",
        message_type=MetricKind.SENSOR,
        short_id="multirssolar_acin_current_{phase}",
        name="Current {phase}",
        metric_type=MetricType.CURRENT,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/multi/{device_id}/Ac/In/1/{phase}/P",
        message_type=MetricKind.SENSOR,
        short_id="multirssolar_acin_power_{phase}",
        name="Power on {phase}",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/multi/{device_id}/Ac/In/1/{phase}/V",
        message_type=MetricKind.SENSOR,
        short_id="multirssolar_acin_voltage_{phase}",
        name="Voltage on {phase}",
        metric_type=MetricType.VOLTAGE,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/multi/{device_id}/Ac/NumberOfPhases",
        message_type=MetricKind.SENSOR,
        short_id="multirssolar_phases",
        name="Phases",
        metric_nature=MetricNature.INSTANTANEOUS,
        value_type=ValueType.INT,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/multi/{device_id}/Ac/Out/{output}/{phase}/I",
        message_type=MetricKind.SENSOR,
        short_id="multirssolar_acout_{output}_current_{phase}",
        name="AC Out {output} Current on {phase}",
        metric_type=MetricType.CURRENT,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/multi/{device_id}/Ac/Out/{output}/{phase}/P",
        message_type=MetricKind.SENSOR,
        short_id="multirssolar_acout_{output}_power_{phase}",
        name="AC Out {output} Power on {phase}",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/multi/{device_id}/Ac/Out/{output}/{phase}/V",
        message_type=MetricKind.SENSOR,
        short_id="multirssolar_acout_{output}_voltage_{phase}",
        name="AC Out {output} Voltage on {phase}",
        metric_type=MetricType.VOLTAGE,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/multi/{device_id}/Energy/AcIn1ToAcOut",
        message_type=MetricKind.SENSOR,
        short_id="multi_acin1_to_acout",
        name="ACin1 to ACout",
        metric_type=MetricType.ENERGY,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/multi/{device_id}/Energy/AcIn1ToInverter",
        message_type=MetricKind.SENSOR,
        short_id="multi_acin1_to_inverter",
        name="ACin1 to Inverter",
        metric_type=MetricType.ENERGY,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/multi/{device_id}/Energy/AcOutToAcIn1",
        message_type=MetricKind.SENSOR,
        short_id="multi_acout_to_acin1",
        name="ACout to ACin1",
        metric_type=MetricType.ENERGY,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/multi/{device_id}/Energy/InverterToAcIn1",
        message_type=MetricKind.SENSOR,
        short_id="multi_inverter_to_acin1",
        name="Inverter to ACin1",
        metric_type=MetricType.ENERGY,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/multi/{device_id}/Energy/InverterToAcOut",
        message_type=MetricKind.SENSOR,
        short_id="multi_inverter_to_acout",
        name="Inverter to ACout",
        metric_type=MetricType.ENERGY,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/multi/{device_id}/Energy/OutToInverter",
        message_type=MetricKind.SENSOR,
        short_id="multi_acout_to_inverter",
        name="ACout to Inverter",
        metric_type=MetricType.ENERGY,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/multi/{device_id}/Energy/SolarToAcIn1",
        message_type=MetricKind.SENSOR,
        short_id="multi_solar_to_acin1",
        name="Solar to ACin1",
        metric_type=MetricType.ENERGY,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/multi/{device_id}/Energy/SolarToAcOut",
        message_type=MetricKind.SENSOR,
        short_id="multi_solar_to_acout",
        name="Solar to ACout",
        metric_type=MetricType.ENERGY,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/multi/{device_id}/Energy/SolarToBattery",
        message_type=MetricKind.SENSOR,
        short_id="multi_solar_to_battery",
        name="Solar to Battery",
        metric_type=MetricType.ENERGY,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/multi/{device_id}/Ess/AcPowerSetpoint",
        message_type=MetricKind.SENSOR,
        short_id="multi_ess_ac_power_setpoint",
        name="ESS AC Power Setpoint",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/multi/{device_id}/Ess/DisableCharge",
        message_type=MetricKind.SWITCH,
        short_id="multi_disable_charge",
        name="ESS Disable Charge",
        value_type=ValueType.ENUM,
        enum=GenericOnOff,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/multi/{device_id}/Ess/DisableFeedIn",
        message_type=MetricKind.SWITCH,
        short_id="multi_disable_feed_in",
        name="ESS Disable Feed In",
        value_type=ValueType.ENUM,
        enum=GenericOnOff,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/multi/{device_id}/Ess/InverterPowerSetpoint",
        message_type=MetricKind.SENSOR,
        short_id="multirssolar_inverter_power_setpoint",
        name="Inverter Power Setpoint",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/multi/{device_id}/History/Daily/0/Pv/{mppt_id}/Yield",
        message_type=MetricKind.SENSOR,
        short_id="multi_mppt_{mppt_id}_yield_today",
        name="MPPT {mppt_id} Yield today",
        metric_type=MetricType.ENERGY,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/multi/{device_id}/History/Daily/1/Pv/{mppt_id}/Yield",
        message_type=MetricKind.SENSOR,
        short_id="multi_mppt_{mppt_id}_yield_yesterday",
        name="MPPT {mppt_id} Yield yesterday",
        metric_type=MetricType.ENERGY,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/multi/{device_id}/Pv/{mpptnumber}/MppOperationMode",
        message_type=MetricKind.SENSOR,
        short_id="multirssolar_mppt_{mpptnumber}_state",
        name="MPPT {mpptnumber} state",
        value_type=ValueType.ENUM,
        enum=MppOperationMode,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/multi/{device_id}/Pv/{mpptnumber}/P",
        message_type=MetricKind.SENSOR,
        short_id="multirssolar_mppt_{mpptnumber}_power",
        name="MPPT {mpptnumber} Power",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/multi/{device_id}/Pv/{mpptnumber}/V",
        message_type=MetricKind.SENSOR,
        short_id="multirssolar_mppt_{mpptnumber}_voltage",
        name="MPPT {mpptnumber} PV Voltage",
        metric_type=MetricType.VOLTAGE,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/multi/{device_id}/Relay/0/State",
        message_type=MetricKind.SWITCH,
        short_id="multi_relay0_state",
        name="Relay on Multi RS State",
        value_type=ValueType.ENUM,
        enum=GenericOnOff,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/multi/{device_id}/Settings/Ess/MinimumSocLimit",
        message_type=MetricKind.NUMBER,
        short_id="multi_ess_min_soc_limit",
        name="ESS Minimum SoC Limit",
        metric_type=MetricType.ELECTRIC_STORAGE_PERCENTAGE,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/multi/{device_id}/Settings/Ess/Mode",
        message_type=MetricKind.SENSOR,
        short_id="multirssolar_ess_mode",
        name="ESS Mode",
        value_type=ValueType.ENUM,
        enum=ESSMode,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/multi/{device_id}/State",
        message_type=MetricKind.SENSOR,
        short_id="multirssolar_state",
        name="State",
        value_type=ValueType.ENUM,
        enum=State,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/multi/{device_id}/Yield/Power",
        message_type=MetricKind.SENSOR,
        short_id="multirssolar_pv_power_total",
        name="PV Power Total",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/multi/{device_id}/Yield/User",
        message_type=MetricKind.SENSOR,
        short_id="multirssolar_total_pv_yield",
        name="Total PV Yield",
        metric_type=MetricType.ENERGY,
    ),
    # Platform topics
    TopicDescriptor(
        topic="N/{installation_id}/platform/{device_id}/Device/Reboot",
        message_type=MetricKind.BUTTON,
        short_id="platform_device_reboot",
        name="Platform device reboot",
        value_type=ValueType.ENUM,
        enum=GenericOnOff,
        experimental=True,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/platform/{device_id}/Firmware/Installed/Version",
        message_type=MetricKind.SENSOR,
        value_type=ValueType.STRING,
        short_id="platform_venus_firmware_installed_version",
        name="Firmware Venus Installed Version",
    ),
    TopicDescriptor(
        topic="N/{installation_id}/platform/{device_id}/Firmware/Online/AvailableVersion",
        message_type=MetricKind.SENSOR,
        value_type=ValueType.STRING,
        short_id="platform_venus_firmware_available_version",
        name="Firmware Venus Available Version",
    ),
    # PV Inverter topics
    TopicDescriptor(
        topic="N/{installation_id}/pvinverter/{device_id}/Ac/Energy/Forward",
        message_type=MetricKind.SENSOR,
        short_id="pvinverter_yield_total",
        name="Total Yield",
        metric_type=MetricType.ENERGY,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/pvinverter/{device_id}/Ac/Power",
        message_type=MetricKind.SENSOR,
        short_id="pvinverter_power_total",
        name="Power Total",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/pvinverter/{device_id}/Ac/{phase}/Current",
        message_type=MetricKind.SENSOR,
        short_id="pvinverter_current_{phase}",
        name="Current {phase}",
        metric_type=MetricType.CURRENT,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/pvinverter/{device_id}/Ac/{phase}/Energy/Forward",
        message_type=MetricKind.SENSOR,
        short_id="pvinverter_yield_{phase}",
        name="Yield {phase}",
        metric_type=MetricType.ENERGY,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/pvinverter/{device_id}/Ac/{phase}/Power",
        message_type=MetricKind.SENSOR,
        short_id="pvinverter_power_{phase}",
        name="Power {phase}",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/pvinverter/{device_id}/Ac/{phase}/Voltage",
        message_type=MetricKind.SENSOR,
        short_id="pvinverter_voltage_{phase}",
        name="Voltage {phase}",
        metric_type=MetricType.VOLTAGE,
    ),
    # Settings topics
    TopicDescriptor(
        topic="N/{installation_id}/settings/{device_id}/Settings/CGwacs/AcExportLimit",
        message_type=MetricKind.NUMBER,
        short_id="system_ac_export_limit",
        name="AC Export Limit",
        metric_type=MetricType.CURRENT,
        value_type=ValueType.INT,
        min=-1,  # -1 means not set
        max=100,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/settings/{device_id}/Settings/CGwacs/AcInputLimit",
        message_type=MetricKind.NUMBER,
        short_id="system_ac_input_limit",
        name="AC Input Limit",
        metric_type=MetricType.CURRENT,
        value_type=ValueType.INT,
        min=-1,  # -1 means not set
        max=100,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/settings/{device_id}/Settings/CGwacs/AcPowerSetPoint",
        message_type=MetricKind.NUMBER,
        short_id="system_ac_power_set_point",
        name="AC Power Setpoint",
        metric_type=MetricType.POWER,
        value_type=ValueType.INT,
        min_max_range=RangeType.DYNAMIC,
        min=-10000,  # Dynamic range, depends on device model,
        max=10000,  # Dynamic range, depends on device model
    ),
    TopicDescriptor(
        topic="N/{installation_id}/settings/{device_id}/Settings/CGwacs/BatteryLife/MinimumSocLimit",
        message_type=MetricKind.NUMBER,
        short_id="system_ess_min_soc_limit",
        name="ESS min SOC limit",
        metric_type=MetricType.ELECTRIC_STORAGE_PERCENTAGE,
        value_type=ValueType.INT,
        min_max_range=RangeType.DYNAMIC,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/settings/{device_id}/Settings/CGwacs/BatteryLife/Schedule/Charge/{slot}/Day",
        message_type=MetricKind.SELECT,
        short_id="system_ess_schedule_charge_{slot}_days",
        name="ESS BatteryLife Schedule charge {slot} days",
        value_type=ValueType.ENUM,
        enum=ChargeSchedule,
    ),
    TopicDescriptor(
        topic="$$func/system/schedule_charge_enabled:schedule_charge_enabled_set",
        depends_on=["system_ess_schedule_charge_{slot}_days"],
        message_type=MetricKind.SWITCH,
        short_id="system_ess_schedule_charge_{slot}_enabled",
        name="ESS BatteryLife Schedule charge {slot} enabled",
        value_type=ValueType.ENUM,
        enum=GenericOnOff
    ),
    TopicDescriptor(
        topic="N/{installation_id}/settings/{device_id}/Settings/CGwacs/BatteryLife/Schedule/Charge/{slot}/Duration",
        message_type=MetricKind.NUMBER,
        short_id="system_ess_schedule_charge_{slot}_duration",
        name="ESS BatteryLife Schedule charge {slot} Duration",
        metric_type=MetricType.TIME,
        unit_of_measurement = "min",
        value_type = ValueType.INT_SECONDS_TO_MINUTES,
        experimental=True
    ),
    TopicDescriptor(
        topic="N/{installation_id}/settings/{device_id}/Settings/CGwacs/BatteryLife/Schedule/Charge/{slot}/Soc",
        message_type=MetricKind.NUMBER,
        short_id="system_ess_schedule_charge_{slot}_soc",
        name="ESS BatteryLife Schedule charge {slot} SOC",
        metric_type=MetricType.ELECTRIC_STORAGE_PERCENTAGE,
        value_type=ValueType.INT,
        experimental=True
    ),
    TopicDescriptor(
        topic="N/{installation_id}/settings/{device_id}/Settings/CGwacs/BatteryLife/Schedule/Charge/{slot}/Start",
        message_type=MetricKind.TIME,
        short_id="system_ess_schedule_charge_{slot}_start",
        name="ESS BatteryLife Schedule charge {slot} Start",
        metric_type=MetricType.TIME,
        unit_of_measurement="min",
        value_type=ValueType.INT_SECONDS_TO_MINUTES,
        experimental=True
    ),
    TopicDescriptor(
        topic="N/{installation_id}/settings/{device_id}/Settings/CGwacs/BatteryLife/State",
        message_type=MetricKind.SELECT,
        short_id="system_ess_batterylife_state",
        name="ESS BatteryLife State",
        value_type=ValueType.ENUM,
        enum=ESSState,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/settings/{device_id}/Settings/CGwacs/Hub4Mode",
        message_type=MetricKind.SELECT,
        short_id="system_ess_mode",
        name="ESS Mode (Hub4)",
        value_type=ValueType.ENUM,
        enum=ESSModeHub4,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/settings/{device_id}/Settings/CGwacs/MaxDischargePower",
        message_type=MetricKind.NUMBER,
        short_id="system_ess_max_inverter_power_limit",
        name="ESS max inverter power limit",
        metric_type=MetricType.POWER,
        value_type=ValueType.INT,
        min_max_range=RangeType.DYNAMIC,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/settings/{device_id}/Settings/CGwacs/MaxFeedInPower",
        message_type=MetricKind.NUMBER,
        short_id="system_ess_max_feed_in_power",
        name="ESS max feed-in power",
        metric_type=MetricType.POWER,
        value_type=ValueType.INT,
        min_max_range=RangeType.DYNAMIC,
        min=-1,
        max=1000000,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/settings/{device_id}/Settings/CGwacs/OvervoltageFeedIn",
        message_type=MetricKind.SWITCH,
        short_id="system_settings_overvoltage_feedin",
        name="PV DC Overvoltage FeedIn",
        value_type=ValueType.ENUM,
        enum=GenericOnOff,
    ),
    # Dynamic ESS settings topics
    TopicDescriptor(
        topic="N/{installation_id}/settings/{device_id}/Settings/DynamicEss/Mode",
        message_type=MetricKind.SELECT,
        short_id="system_settings_dess_mode",
        name="DESS Mode",
        value_type=ValueType.ENUM,
        enum=DESSMode,
    ),
    # Generator settings topics
    TopicDescriptor(
        topic="N/{installation_id}/settings/{device_id}/Settings/Generator{gen_id(0-1)}/BatteryVoltage/Enabled",
        message_type=MetricKind.SWITCH,
        short_id="generator_{gen_id}_start_on_voltage_enabled",
        name="Generator start on voltage enabled",
        value_type=ValueType.ENUM,
        enum=GenericOnOff,
        depends_on=["generator_{gen_id}_generator_autorun"],
    ),
    TopicDescriptor(
        topic="N/{installation_id}/settings/{device_id}/Settings/Generator{gen_id(0-1)}/BatteryVoltage/QuietHoursStartValue",
        message_type=MetricKind.NUMBER,
        short_id="generator_{gen_id}_qh_start_on_voltage",
        name="Generator QH start on voltage",
        metric_type=MetricType.VOLTAGE,
        min_max_range=RangeType.DYNAMIC,
        depends_on=["generator_{gen_id}_generator_autorun"],
    ),
    TopicDescriptor(
        topic="N/{installation_id}/settings/{device_id}/Settings/Generator{gen_id(0-1)}/BatteryVoltage/QuietHoursStopValue",
        message_type=MetricKind.NUMBER,
        short_id="generator_{gen_id}_qh_stop_on_voltage",
        name="Generator QH stop on voltage",
        metric_type=MetricType.VOLTAGE,
        min_max_range=RangeType.DYNAMIC,
        depends_on=["generator_{gen_id}_generator_autorun"],
    ),
    TopicDescriptor(
        topic="N/{installation_id}/settings/{device_id}/Settings/Generator{gen_id(0-1)}/BatteryVoltage/StartTimer",
        message_type=MetricKind.NUMBER,
        short_id="generator_{gen_id}_start_on_voltage_timer",
        name="Generator start on voltage timer",
        unit_of_measurement="s",
        value_type=ValueType.INT,
        min_max_range=RangeType.DYNAMIC,
        depends_on=["generator_{gen_id}_generator_autorun"],
    ),
    TopicDescriptor(
        topic="N/{installation_id}/settings/{device_id}/Settings/Generator{gen_id(0-1)}/BatteryVoltage/StartValue",
        message_type=MetricKind.NUMBER,
        short_id="generator_{gen_id}_start_on_voltage",
        name="Generator start on voltage",
        metric_type=MetricType.VOLTAGE,
        min_max_range=RangeType.DYNAMIC,
        depends_on=["generator_{gen_id}_generator_autorun"],
    ),
    TopicDescriptor(
        topic="N/{installation_id}/settings/{device_id}/Settings/Generator{gen_id(0-1)}/BatteryVoltage/StopTimer",
        message_type=MetricKind.NUMBER,
        short_id="generator_{gen_id}_stop_on_voltage_timer",
        name="Generator stop on voltage timer",
        unit_of_measurement="s",
        value_type=ValueType.INT,
        min_max_range=RangeType.DYNAMIC,
        depends_on=["generator_{gen_id}_generator_autorun"],
    ),
    TopicDescriptor(
        topic="N/{installation_id}/settings/{device_id}/Settings/Generator{gen_id(0-1)}/BatteryVoltage/StopValue",
        message_type=MetricKind.NUMBER,
        short_id="generator_{gen_id}_stop_on_voltage",
        name="Generator stop on voltage",
        metric_type=MetricType.VOLTAGE,
        min_max_range=RangeType.DYNAMIC,
        depends_on=["generator_{gen_id}_generator_autorun"],
    ),
    TopicDescriptor(
        topic="N/{installation_id}/settings/{device_id}/Settings/Generator{gen_id(0-1)}/CoolDownTime",
        message_type=MetricKind.NUMBER,
        short_id="generator_{gen_id}_cool_down_timer",
        name="Generator cool down timer",
        unit_of_measurement="s",
        value_type=ValueType.INT,
        min=0,
        max=600,
        depends_on=["generator_{gen_id}_generator_autorun"],
    ),
    TopicDescriptor(
        topic="N/{installation_id}/settings/{device_id}/Settings/Generator{gen_id(0-1)}/GeneratorStopTime",
        message_type=MetricKind.NUMBER,
        short_id="generator_{gen_id}_shut_down_timer",
        name="Generator shut down timer",
        unit_of_measurement="s",
        value_type=ValueType.INT,
        min=0,
        max=600,
        depends_on=["generator_{gen_id}_generator_autorun"],
    ),
    TopicDescriptor(
        topic="N/{installation_id}/settings/{device_id}/Settings/Generator{gen_id(0-1)}/InverterHighTemp/Enabled",
        message_type=MetricKind.SWITCH,
        short_id="generator_{gen_id}_start_on_temp_enabled",
        name="Generator start on high temp enabled",
        value_type=ValueType.ENUM,
        enum=GenericOnOff,
        depends_on=["generator_{gen_id}_generator_autorun"],
    ),
    TopicDescriptor(
        topic="N/{installation_id}/settings/{device_id}/Settings/Generator{gen_id(0-1)}/InverterHighTemp/StartTimer",
        message_type=MetricKind.NUMBER,
        short_id="generator_{gen_id}_start_on_temp_timer",
        name="Generator start on temp timer",
        unit_of_measurement="s",
        value_type=ValueType.INT,
        min_max_range=RangeType.DYNAMIC,
        depends_on=["generator_{gen_id}_generator_autorun"],
    ),
    TopicDescriptor(
        topic="N/{installation_id}/settings/{device_id}/Settings/Generator{gen_id(0-1)}/InverterHighTemp/StopTimer",
        message_type=MetricKind.NUMBER,
        short_id="generator_{gen_id}_stop_on_temp_timer",
        name="Generator stop on temp timer",
        unit_of_measurement="s",
        value_type=ValueType.INT,
        min_max_range=RangeType.DYNAMIC,
        depends_on=["generator_{gen_id}_generator_autorun"],
    ),
    TopicDescriptor(
        topic="N/{installation_id}/settings/{device_id}/Settings/Generator{gen_id(0-1)}/QuietHours/Enabled",
        message_type=MetricKind.SWITCH,
        short_id="generator_{gen_id}_quiet_hours_enabled",
        name="Generator quiet hours enabled",
        value_type=ValueType.ENUM,
        enum=GenericOnOff,
        depends_on=["generator_{gen_id}_generator_autorun"],
    ),
    TopicDescriptor(
        topic="N/{installation_id}/settings/{device_id}/Settings/Generator{gen_id(0-1)}/ServiceInterval",
        message_type=MetricKind.NUMBER,
        short_id="generator_{gen_id}_service_interval",
        name="Generator service interval",
        unit_of_measurement="h",
        value_type=ValueType.INT_SECONDS_TO_HOURS,
        min=0,
        max=500,
        depends_on=["generator_{gen_id}_generator_autorun"],
    ),
    TopicDescriptor(
        topic="N/{installation_id}/settings/{device_id}/Settings/Generator{gen_id(0-1)}/Soc/Enabled",
        message_type=MetricKind.SWITCH,
        short_id="generator_{gen_id}_start_on_soc_enabled",
        name="Generator start on SOC enabled",
        value_type=ValueType.ENUM,
        enum=GenericOnOff,
        depends_on=["generator_{gen_id}_generator_autorun"],
    ),
    TopicDescriptor(
        topic="N/{installation_id}/settings/{device_id}/Settings/Generator{gen_id(0-1)}/Soc/QuietHoursStartValue",
        message_type=MetricKind.NUMBER,
        short_id="generator_{gen_id}_qh_start_on_soc",
        name="Generator QH start on SOC",
        value_type=ValueType.INT,
        min_max_range=RangeType.DYNAMIC,
        depends_on=["generator_{gen_id}_generator_autorun"],
    ),
    TopicDescriptor(
        topic="N/{installation_id}/settings/{device_id}/Settings/Generator{gen_id(0-1)}/Soc/QuietHoursStopValue",
        message_type=MetricKind.NUMBER,
        short_id="generator_{gen_id}_qh_stop_on_soc",
        name="Generator QH stop on SOC",
        value_type=ValueType.INT,
        min_max_range=RangeType.DYNAMIC,
        depends_on=["generator_{gen_id}_generator_autorun"],
    ),
    TopicDescriptor(
        topic="N/{installation_id}/settings/{device_id}/Settings/Generator{gen_id(0-1)}/Soc/StartTimer",
        message_type=MetricKind.NUMBER,
        short_id="generator_{gen_id}_start_on_soc_timer",
        name="Generator start on SOC timer",
        unit_of_measurement="s",
        value_type=ValueType.INT,
        min=0,
        max=10000,
        depends_on=["generator_{gen_id}_generator_autorun"],
    ),
    TopicDescriptor(
        topic="N/{installation_id}/settings/{device_id}/Settings/Generator{gen_id(0-1)}/Soc/StartValue",
        message_type=MetricKind.NUMBER,
        short_id="generator_{gen_id}_start_on_soc",
        name="Generator start on SOC",
        value_type=ValueType.INT,
        min_max_range=RangeType.DYNAMIC,
        depends_on=["generator_{gen_id}_generator_autorun"],
    ),
    TopicDescriptor(
        topic="N/{installation_id}/settings/{device_id}/Settings/Generator{gen_id(0-1)}/Soc/StopTimer",
        message_type=MetricKind.NUMBER,
        short_id="generator_{gen_id}_stop_on_soc_timer",
        name="Generator stop on SOC timer",
        unit_of_measurement="s",
        value_type=ValueType.INT,
        min=0,
        max=10000,
        depends_on=["generator_{gen_id}_generator_autorun"],
    ),
    TopicDescriptor(
        topic="N/{installation_id}/settings/{device_id}/Settings/Generator{gen_id(0-1)}/Soc/StopValue",
        message_type=MetricKind.NUMBER,
        short_id="generator_{gen_id}_stop_on_soc",
        name="Generator stop on SOC",
        value_type=ValueType.INT,
        min_max_range=RangeType.DYNAMIC,
        depends_on=["generator_{gen_id}_generator_autorun"],
    ),
    TopicDescriptor(
        topic="N/{installation_id}/settings/{device_id}/Settings/Generator{gen_id(0-1)}/WarmUpTime",
        message_type=MetricKind.NUMBER,
        short_id="generator_{gen_id}_warm_up_timer",
        name="Generator warm up timer",
        unit_of_measurement="s",
        value_type=ValueType.INT,
        min=0,
        max=1800,
        depends_on=["generator_{gen_id}_generator_autorun"],
    ),
    # Relay Custom Name topics
    TopicDescriptor(
        topic="N/{installation_id}/settings/{device_id}/Settings/Relay/{relay}/CustomName",
        message_type=MetricKind.SENSOR,
        short_id="system_relay_{relay}_custom_name",
        name="Relay {relay} Custom Name",
        value_type=ValueType.STRING,
    ),
    # System Setup topics
    TopicDescriptor(
        topic="N/{installation_id}/settings/{device_id}/Settings/SystemSetup/MaxChargeCurrent",
        message_type=MetricKind.NUMBER,
        short_id="system_ess_max_charge_current",
        name="ESS max charge current",
        metric_type=MetricType.CURRENT,
        value_type=ValueType.INT,
        min_max_range=RangeType.DYNAMIC,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/settings/{device_id}/Settings/SystemSetup/MaxChargeVoltage",
        message_type=MetricKind.NUMBER,
        short_id="system_ess_max_charge_voltage",
        name="ESS max charge voltage",
        metric_type=MetricType.VOLTAGE,
        value_type=ValueType.INT,
        min_max_range=RangeType.DYNAMIC,
        step=0.1
    ),
    TopicDescriptor(
        topic="N/{installation_id}/settings/{device_id}/Settings/TransferSwitch/GeneratorCurrentLimit",
        message_type=MetricKind.NUMBER,
        short_id="transfer_switch_generator_current_limit",
        name="Generator AC current limit",
        metric_type=MetricType.CURRENT,
        value_type=ValueType.FLOAT,
        precision=1,
        min=0,
        max=60,
    ),
    # Solar Charger topics
    TopicDescriptor(
        topic="N/{installation_id}/solarcharger/{device_id}/Dc/0/Current",
        message_type=MetricKind.SENSOR,
        short_id="solarcharger_dc_current",
        name="DC (Batt) Bus current",
        metric_type=MetricType.CURRENT,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/solarcharger/{device_id}/Dc/0/Voltage",
        message_type=MetricKind.SENSOR,
        short_id="solarcharger_dc_voltage",
        name="DC (Batt) Bus voltage",
        metric_type=MetricType.VOLTAGE,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/solarcharger/{device_id}/ErrorCode",
        message_type=MetricKind.SENSOR,
        short_id="solarcharger_error_code",
        name="Solar Charger Error Code",
        value_type=ValueType.ENUM,
        enum=ErrorCode,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/solarcharger/{device_id}/History/Daily/0/MaxBatteryVoltage",
        message_type=MetricKind.SENSOR,
        short_id="solarcharger_max_battery_voltage_today",
        name="Solar Charger Max Battery Voltage Today",
        metric_type=MetricType.VOLTAGE,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/solarcharger/{device_id}/History/Daily/0/MaxPower",
        message_type=MetricKind.SENSOR,
        short_id="solarcharger_max_power_today",
        name="Max Power Today",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/solarcharger/{device_id}/History/Daily/0/MinBatteryVoltage",
        message_type=MetricKind.SENSOR,
        short_id="solarcharger_min_battery_voltage_today",
        name="Solar Charger Min Battery Voltage Today",
        metric_type=MetricType.VOLTAGE,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/solarcharger/{device_id}/History/Daily/0/TimeInAbsorption",
        message_type=MetricKind.SENSOR,
        short_id="solarcharger_time_in_absorption_today",
        name="Solar Charger Time in Absorption Today",
        unit_of_measurement="s",
        metric_nature=MetricNature.INSTANTANEOUS,
        value_type=ValueType.INT,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/solarcharger/{device_id}/History/Daily/0/TimeInBulk",
        message_type=MetricKind.SENSOR,
        short_id="solarcharger_time_in_bulk_today",
        name="Solar Charger Time in Bulk Today",
        unit_of_measurement="s",
        metric_nature=MetricNature.INSTANTANEOUS,
        value_type=ValueType.INT,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/solarcharger/{device_id}/History/Daily/0/TimeInFloat",
        message_type=MetricKind.SENSOR,
        short_id="solarcharger_time_in_float_today",
        name="Solar Charger Time in Float Today",
        unit_of_measurement="s",
        metric_nature=MetricNature.INSTANTANEOUS,
        value_type=ValueType.INT,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/solarcharger/{device_id}/History/Daily/0/Yield",
        message_type=MetricKind.SENSOR,
        short_id="solarcharger_yield_today",
        name="Yield Today",
        metric_type=MetricType.ENERGY,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/solarcharger/{device_id}/History/Daily/1/MaxPower",
        message_type=MetricKind.SENSOR,
        short_id="solarcharger_max_power_yesterday",
        name="Max Power Yesterday",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/solarcharger/{device_id}/History/Daily/1/Yield",
        message_type=MetricKind.SENSOR,
        short_id="solarcharger_yield_yesterday",
        name="Yield Yesterday",
        metric_type=MetricType.ENERGY,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/solarcharger/{device_id}/Load/I",
        message_type=MetricKind.SENSOR,
        short_id="solarcharger_load_current",
        name="Load Bus current",
        metric_type=MetricType.CURRENT,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/solarcharger/{device_id}/Load/State",
        message_type=MetricKind.BINARY_SENSOR,
        short_id="solarcharger_load_state",
        name="Solar Charger Load State",
        value_type=ValueType.ENUM,
        enum=GenericOnOff,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/solarcharger/{device_id}/Mode",
        message_type=MetricKind.SWITCH,
        short_id="solarcharger_mode",
        name="Solar Charger Mode",
        value_type=ValueType.ENUM,
        enum=ChargerMode,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/solarcharger/{device_id}/MppOperationMode",
        message_type=MetricKind.SENSOR,
        short_id="solarcharger_mppt_operation_mode",
        name="Solar Charger MPPT Operation Mode",
        value_type=ValueType.ENUM,
        enum=MppOperationMode,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/solarcharger/{device_id}/Pv/V",
        message_type=MetricKind.SENSOR,
        short_id="solarcharger_voltage",
        name="PV Bus Voltage",
        metric_type=MetricType.VOLTAGE,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/solarcharger/{device_id}/Pv/{tracker}/MppOperationMode",
        message_type=MetricKind.SENSOR,
        short_id="solarcharger_tracker_{tracker}_operation_mode",
        name="PV Tracker {tracker:solarcharger_tracker_{tracker}_name} Operation Mode",
        value_type=ValueType.ENUM,
        enum=MppOperationMode,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/solarcharger/{device_id}/Pv/{tracker}/Name",
        message_type=MetricKind.SENSOR,
        short_id="solarcharger_tracker_{tracker}_name",
        name="PV Tracker {tracker} Name",
        value_type=ValueType.STRING,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/solarcharger/{device_id}/Pv/{tracker}/P",
        message_type=MetricKind.SENSOR,
        short_id="solarcharger_tracker_{tracker}_power",
        name="PV Tracker {tracker:solarcharger_tracker_{tracker}_name} Power",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/solarcharger/{device_id}/Pv/{tracker}/V",
        message_type=MetricKind.SENSOR,
        short_id="solarcharger_tracker_{tracker}_voltage",
        name="PV Tracker {tracker:solarcharger_tracker_{tracker}_name} Voltage",
        metric_type=MetricType.VOLTAGE,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/solarcharger/{device_id}/Relay/0/State",
        message_type=MetricKind.SWITCH,
        short_id="solarcharger_relay_state",
        name="Solar Charger Relay State",
        value_type=ValueType.ENUM,
        enum=GenericOnOff,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/solarcharger/{device_id}/State",
        message_type=MetricKind.SENSOR,
        short_id="solarcharger_state",
        name="Solar Charger State",
        value_type=ValueType.ENUM,
        enum=State,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/solarcharger/{device_id}/Yield/Power",
        message_type=MetricKind.SENSOR,
        short_id="solarcharger_yield_power",
        name="PV Yield Power",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/solarcharger/{device_id}/Yield/User",
        message_type=MetricKind.SENSOR,
        short_id="solarcharger_yield_total",
        name="Total Yield",
        metric_type=MetricType.ENERGY,
    ),
    # Switch topics
    TopicDescriptor(
        topic="N/{installation_id}/switch/{device_id}/SwitchableOutput/output_{output(1-4)}/Dimming",
        message_type=MetricKind.NUMBER,
        short_id="switch_{output}_dimming",
        name="Switch {output:switch_{output}_custom_name} Dimming",
        value_type=ValueType.INT,
        min=0,
        max=100,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/switch/{device_id}/SwitchableOutput/output_{output(1-4)}/Settings/CustomName",
        message_type=MetricKind.SENSOR,
        short_id="switch_{output}_custom_name",
        name="Switch {output} Custom Name",
        value_type=ValueType.STRING,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/switch/{device_id}/SwitchableOutput/output_{output(1-4)}/State",
        message_type=MetricKind.SWITCH,
        short_id="switch_{output}_state",
        name="Switch {output:switch_{output}_custom_name} State",
        value_type=ValueType.ENUM,
        enum=GenericOnOff,
    ),
    # System topics
    TopicDescriptor(
        topic="N/{installation_id}/system/{device_id}/Ac/ActiveIn/Source",
        message_type=MetricKind.SENSOR,
        short_id="system_ac_active_input_source",
        name="AC Active Input Source",
        value_type=ValueType.ENUM,
        enum=AcActiveInputSource,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/system/{device_id}/Ac/Consumption/NumberOfPhases",
        message_type=MetricKind.SENSOR,
        short_id="system_consumption_phases",
        name="Consumption phases",
        metric_nature=MetricNature.INSTANTANEOUS,
        value_type=ValueType.INT,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/system/{device_id}/Ac/Consumption/{phase}/Current",
        message_type=MetricKind.SENSOR,
        short_id="system_consumption_current_{phase}",
        name="Consumption Current {phase}",
        metric_type=MetricType.CURRENT,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/system/{device_id}/Ac/Consumption/{phase}/Power",
        message_type=MetricKind.SENSOR,
        short_id="system_consumption_power_{phase}",
        name="Consumption Power {phase}",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/system/{device_id}/Ac/ConsumptionOnInput/{phase}/Power",
        message_type=MetricKind.SENSOR,
        short_id="system_ac_loads_{phase}",
        name="AC loads on {phase}",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/system/{device_id}/Ac/ConsumptionOnOutput/NumberOfPhases",
        message_type=MetricKind.SENSOR,
        short_id="system_consumption_on_output_phases",
        name="Consumption On Output phases",
        metric_nature=MetricNature.INSTANTANEOUS,
        value_type=ValueType.INT,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/system/{device_id}/Ac/ConsumptionOnOutput/{phase}/Power",
        message_type=MetricKind.SENSOR,
        short_id="system_critical_loads_{phase}",
        name="Critical loads on {phase}",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/system/{device_id}/Ac/Genset/{phase}/Power",
        message_type=MetricKind.SENSOR,
        short_id="system_generator_load_{phase}",
        name="Genset Load {phase}",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/system/{device_id}/Ac/Grid/NumberOfPhases",
        message_type=MetricKind.SENSOR,
        short_id="system_grid_phases",  # system attribute
        name="Grid phases",
        metric_nature=MetricNature.INSTANTANEOUS,
        value_type=ValueType.INT_DEFAULT_0,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/system/{device_id}/Ac/Grid/{phase}/Current",
        message_type=MetricKind.SENSOR,
        short_id="system_grid_current_{phase}",
        name="Grid Current {phase}",
        metric_type=MetricType.CURRENT,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/system/{device_id}/Ac/Grid/{phase}/Power",
        message_type=MetricKind.SENSOR,
        short_id="system_grid_power_{phase}",
        name="Grid Power {phase}",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/system/{device_id}/Ac/PvOnOutput/NumberOfPhases",
        message_type=MetricKind.SENSOR,
        short_id="system_pv_on_output_phases",
        name="PV on Output phases",
        metric_nature=MetricNature.INSTANTANEOUS,
        value_type=ValueType.INT,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/system/{device_id}/Ac/PvOnOutput/{phase}/Current",
        message_type=MetricKind.SENSOR,
        short_id="system_pv_on_output_current_{phase}",
        name="PV on Output Current {phase}",
        metric_type=MetricType.CURRENT,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/system/{device_id}/Ac/PvOnOutput/{phase}/Power",
        message_type=MetricKind.SENSOR,
        short_id="system_pv_on_output_power_{phase}",
        name="PV on Output Power {phase}",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/system/{device_id}/Control/ActiveSocLimit",
        message_type=MetricKind.SENSOR,
        short_id="system_control_active_soc_limit",
        name="Active SOC Limit",
        metric_type=MetricType.ELECTRIC_STORAGE_PERCENTAGE,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/system/{device_id}/Control/ScheduledSoc",
        message_type=MetricKind.SENSOR,
        short_id="system_control_scheduled_soc",
        name="Scheduled SOC",
        metric_type=MetricType.ELECTRIC_STORAGE_PERCENTAGE,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/system/{device_id}/Dc/Battery/Current",
        message_type=MetricKind.SENSOR,
        short_id="system_dc_battery_current",
        name="DC Battery Current",
        metric_type=MetricType.CURRENT,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/system/{device_id}/Dc/Battery/Power",
        message_type=MetricKind.SENSOR,
        short_id="system_dc_battery_power",
        name="DC Battery Power",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="$$func/system/system_dc_battery_charge_power",
        depends_on=["system_dc_battery_power"],
        message_type=MetricKind.SENSOR,
        short_id="system_dc_battery_charge_energy",
        name="DC Battery Charge Energy",
        metric_type=MetricType.ENERGY,
        precision=3,
        experimental=True,
    ),
    TopicDescriptor(
        topic="$$func/system/system_dc_battery_discharge_power",
        depends_on=["system_dc_battery_power"],
        message_type=MetricKind.SENSOR,
        short_id="system_dc_battery_discharge_energy",
        name="DC Battery Discharge Energy",
        metric_type=MetricType.ENERGY,
        precision=3,
        experimental=True,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/system/{device_id}/Dc/Battery/Soc",
        message_type=MetricKind.SENSOR,
        short_id="system_dc_battery_soc",
        name="DC Battery Charge",
        metric_type=MetricType.ELECTRIC_STORAGE_PERCENTAGE,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/system/{device_id}/Dc/Battery/Voltage",
        message_type=MetricKind.SENSOR,
        short_id="system_dc_battery_voltage",
        name="DC Battery Voltage",
        metric_type=MetricType.VOLTAGE,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/system/{device_id}/Dc/Pv/Current",
        message_type=MetricKind.SENSOR,
        short_id="system_dc_pv_current",
        name="PV Current",
        metric_type=MetricType.CURRENT,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/system/{device_id}/Dc/Pv/Power",
        message_type=MetricKind.SENSOR,
        short_id="system_dc_pv_power",
        name="PV Power",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="$$func/system/left_riemann_sum",
        depends_on=["system_dc_pv_power"],
        message_type=MetricKind.SENSOR,
        short_id="system_dc_pv_energy",
        name="PV Energy",
        metric_type=MetricType.ENERGY,
        precision=3,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/system/{device_id}/Dc/System/Power",
        message_type=MetricKind.SENSOR,
        short_id="system_dc_consumption",
        name="DC Consumption",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/system/{device_id}/DynamicEss/Active",
        message_type=MetricKind.BINARY_SENSOR,
        short_id="system_dynamicess_active",
        name="Dynamic ESS Active",
        value_type=ValueType.ENUM,
        enum=GenericOnOff,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/system/{device_id}/DynamicEss/AllowGridFeedIn",
        message_type=MetricKind.BINARY_SENSOR,
        short_id="system_dynamicess_allow_gridfeedin",
        name="Dynamic ESS Allow Grid Feed In",
        value_type=ValueType.ENUM,
        enum=GenericOnOff,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/system/{device_id}/DynamicEss/Available",
        message_type=MetricKind.BINARY_SENSOR,
        short_id="system_dynamicess_available",
        name="Dynamic ESS Available",
        value_type=ValueType.ENUM,
        enum=GenericOnOff,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/system/{device_id}/DynamicEss/AvailableOverhead",
        message_type=MetricKind.SENSOR,
        short_id="system_dynamicess_available_overhead",
        name="Dynamic ESS Available Overhead",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/system/{device_id}/DynamicEss/ErrorCode",
        message_type=MetricKind.SENSOR,
        short_id="system_dynamicess_error",
        name="Dynamic ESS Error",
        value_type=ValueType.ENUM,
        enum=DESSErrorCode,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/system/{device_id}/DynamicEss/LastScheduledEnd",
        message_type=MetricKind.SENSOR,
        short_id="system_dynamicess_last_scheduled_end",
        name="Dynamic ESS Last Scheduled End",
        value_type=ValueType.EPOCH,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/system/{device_id}/DynamicEss/LastScheduledStart",
        message_type=MetricKind.SENSOR,
        short_id="system_dynamicess_last_scheduled_start",
        name="Dynamic ESS Last Scheduled Start",
        value_type=ValueType.EPOCH,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/system/{device_id}/DynamicEss/MinimumSoc",
        message_type=MetricKind.SENSOR,
        short_id="system_dynamicess_minimum_soc",
        name="Dynamic ESS Minimum SOC",
        unit_of_measurement="%",
        metric_type=MetricType.ELECTRIC_STORAGE_PERCENTAGE,
        precision=0,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/system/{device_id}/DynamicEss/NumberOfSchedules",
        message_type=MetricKind.SENSOR,
        short_id="system_dynamicess_schedule_count",
        name="Dynamic ESS Number Of Schedules",
        metric_nature=MetricNature.INSTANTANEOUS,
        value_type=ValueType.INT,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/system/{device_id}/DynamicEss/ReactiveStrategy",
        message_type=MetricKind.SENSOR,
        short_id="system_dynamicess_reactive_strategy",
        name="Dynamic ESS Reactive Strategy",
        value_type=ValueType.ENUM,
        enum=DESSReactiveStrategy,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/system/{device_id}/DynamicEss/Restrictions",
        message_type=MetricKind.SENSOR,
        short_id="system_dynamicess_restrictions",
        name="Dynamic ESS Restrictions",
        value_type=ValueType.ENUM,
        enum=DESSRestrictions,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/system/{device_id}/DynamicEss/Strategy",
        message_type=MetricKind.SENSOR,
        short_id="system_dynamicess_strategy",
        name="Dynamic ESS Strategy",
        value_type=ValueType.ENUM,
        enum=DESSStrategy,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/system/{device_id}/DynamicEss/TargetSoc",
        message_type=MetricKind.SENSOR,
        short_id="system_dynamicess_target_soc",
        name="Dynamic ESS Target SOC",
        metric_type=MetricType.ELECTRIC_STORAGE_PERCENTAGE,
        precision=0,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/system/{device_id}/PV/Current",
        message_type=MetricKind.SENSOR,
        short_id="solarcharger_current",
        name="PV Bus current",
        metric_type=MetricType.CURRENT,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/system/{device_id}/Relay/{relay}/State",
        message_type=MetricKind.SWITCH,
        short_id="system_relay_{relay}",
        name="Relay {relay} state",
        value_type=ValueType.ENUM,
        enum=GenericOnOff,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/system/{device_id}/SystemState/State",
        message_type=MetricKind.SENSOR,
        short_id="system_state",
        name="System state",
        value_type=ValueType.ENUM,
        enum=State,
    ),
    # Tank topics
    TopicDescriptor(
        topic="N/{installation_id}/tank/{device_id}/BatteryVoltage",
        message_type=MetricKind.SENSOR,
        short_id="tank_battery_voltage",
        name="Tank sensor battery voltage",
        metric_type=MetricType.VOLTAGE,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/tank/{device_id}/FluidType",
        message_type=MetricKind.SENSOR,
        short_id="tank_fluid_type",
        name="Fluid Type",
        value_type=ValueType.ENUM,
        enum=FluidType,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/tank/{device_id}/Level",
        message_type=MetricKind.SENSOR,
        short_id="tank_level",
        name="Level",
        unit_of_measurement="%",
        metric_type=MetricType.PERCENTAGE,
        metric_nature=MetricNature.INSTANTANEOUS,
        value_type=ValueType.INT,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/tank/{device_id}/Remaining",
        message_type=MetricKind.SENSOR,
        short_id="tank_remaining",
        name="Remaining",
        unit_of_measurement="m³",
        metric_type=MetricType.LIQUID_VOLUME,
        metric_nature=MetricNature.CUMULATIVE,
        value_type=ValueType.FLOAT,
        precision=2,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/tank/{device_id}/Temperature",
        message_type=MetricKind.SENSOR,
        short_id="tank_temperature",
        name="Temperature",
        metric_type=MetricType.TEMPERATURE,
    ),
    # Temperature sensor topics
    TopicDescriptor(
        topic="N/{installation_id}/temperature/{device_id}/BatteryVoltage",
        message_type=MetricKind.SENSOR,
        short_id="temperature_battery_voltage",
        name="Temperature sensor battery voltage",
        metric_type=MetricType.VOLTAGE,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/temperature/{device_id}/Humidity",
        message_type=MetricKind.SENSOR,
        short_id="temperature_humidity",
        name="Humidity",
        unit_of_measurement="%",
        metric_type=MetricType.PERCENTAGE,
        metric_nature=MetricNature.INSTANTANEOUS,
        value_type=ValueType.FLOAT,
        precision=1,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/temperature/{device_id}/Offset",
        message_type=MetricKind.NUMBER,
        short_id="temperature_offset",
        name="Temperature offset",
        metric_type=MetricType.TEMPERATURE,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/temperature/{device_id}/Scale",
        message_type=MetricKind.NUMBER,
        short_id="temperature_scale",
        name="Temperature scale factor",
        unit_of_measurement=None,
        metric_type=MetricType.NONE,
        metric_nature=MetricNature.INSTANTANEOUS,
        value_type=ValueType.FLOAT,
        precision=2,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/temperature/{device_id}/Status",
        message_type=MetricKind.SENSOR,
        short_id="temperature_status",
        name="Temperature sensor status",
        value_type=ValueType.ENUM,
        enum=TemperatureStatus,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/temperature/{device_id}/Temperature",
        message_type=MetricKind.SENSOR,
        short_id="temperature_temperature",
        name="Temperature",
        metric_type=MetricType.TEMPERATURE,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/temperature/{device_id}/TemperatureType",
        message_type=MetricKind.SENSOR,
        short_id="temperature_type",
        name="Temperature sensor type",
        value_type=ValueType.ENUM,
        enum=TemperatureType,
    ),
    # VEBus topics
    TopicDescriptor(
        topic="N/{installation_id}/vebus/{device_id}/Ac/ActiveIn/ActiveInput",
        message_type=MetricKind.SENSOR,
        short_id="vebus_inverter_active_input",
        name="Inverter active AC input",
        value_type=ValueType.ENUM,
        enum=AcActiveInputSource,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/vebus/{device_id}/Ac/ActiveIn/CurrentLimit",
        message_type=MetricKind.NUMBER,
        short_id="vebus_inverter_current_limit",
        name="Inverter current limit",
        metric_type=MetricType.CURRENT,
        value_type=ValueType.FLOAT,
        min_max_range=RangeType.DYNAMIC,
        min=0,
        max=16,
        precision=1,
        is_adjustable_suffix="CurrentLimitIsAdjustable",
    ),
    TopicDescriptor(
        topic="N/{installation_id}/vebus/{device_id}/Ac/ActiveIn/{phase}/F",
        message_type=MetricKind.SENSOR,
        short_id="vebus_inverter_input_frequency_{phase}",
        name="Inverter input frequency {phase}",
        metric_type=MetricType.FREQUENCY,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/vebus/{device_id}/Ac/ActiveIn/{phase}/P",
        message_type=MetricKind.SENSOR,
        short_id="vebus_inverter_input_power_{phase}",
        name="Inverter input power {phase}",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/vebus/{device_id}/Ac/ActiveIn/{phase}/S",
        message_type=MetricKind.SENSOR,
        short_id="vebus_inverter_input_apparent_power_{phase}",
        name="Inverter input apparent power {phase}",
        metric_type=MetricType.APPARENT_POWER,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/vebus/{device_id}/Ac/ActiveIn/{phase}/V",
        message_type=MetricKind.SENSOR,
        short_id="vebus_inverter_input_voltage_{phase}",
        name="Inverter input voltage {phase}",
        metric_type=MetricType.VOLTAGE,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/vebus/{device_id}/Ac/Control/IgnoreAcIn1",
        message_type=MetricKind.SWITCH,
        short_id="vebus_inverter_ignoreacin1_onoff_control",
        name="Control IgnoreAcIn1",
        value_type=ValueType.ENUM,
        enum=GenericOnOff,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/vebus/{device_id}/Ac/Out/{phase}/F",
        message_type=MetricKind.SENSOR,
        short_id="vebus_inverter_output_frequency_{phase}",
        name="Inverter output frequency {phase}",
        metric_type=MetricType.FREQUENCY,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/vebus/{device_id}/Ac/Out/{phase}/I",
        message_type=MetricKind.SENSOR,
        short_id="vebus_inverter_output_current_{phase}",
        name="Inverter output current {phase}",
        metric_type=MetricType.CURRENT,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/vebus/{device_id}/Ac/Out/{phase}/P",
        message_type=MetricKind.SENSOR,
        short_id="vebus_inverter_output_power_{phase}",
        name="Inverter output power {phase}",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/vebus/{device_id}/Ac/Out/{phase}/S",
        message_type=MetricKind.SENSOR,
        short_id="vebus_inverter_output_apparent_power_{phase}",
        name="Inverter output apparent power {phase}",
        metric_type=MetricType.APPARENT_POWER,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/vebus/{device_id}/Ac/Out/{phase}/V",
        message_type=MetricKind.SENSOR,
        short_id="vebus_inverter_output_voltage_{phase}",
        name="Inverter output voltage {phase}",
        metric_type=MetricType.VOLTAGE,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/vebus/{device_id}/Ac/State/IgnoreAcIn1",
        message_type=MetricKind.SENSOR,
        short_id="vebus_inverter_ignoreacin1_state",
        name="State of IgnoreAcIn1",
        value_type=ValueType.ENUM,
        enum=GenericOnOff,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/vebus/{device_id}/Alarms/GridLost",
        message_type=MetricKind.SENSOR,
        short_id="vebus_inverter_alarm_grid_lost",
        name="Loss of grid alarm",
        value_type=ValueType.ENUM,
        enum=GenericAlarmEnum,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/vebus/{device_id}/Alarms/HighDcCurrent",
        message_type=MetricKind.SENSOR,
        short_id="vebus_inverter_alarm_high_dc_current",
        name="Inverter high DC current alarm",
        value_type=ValueType.ENUM,
        enum=GenericAlarmEnum,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/vebus/{device_id}/Alarms/HighDcVoltage",
        message_type=MetricKind.SENSOR,
        short_id="vebus_inverter_alarm_high_dc_voltage",
        name="Inverter high DC voltage alarm",
        value_type=ValueType.ENUM,
        enum=GenericAlarmEnum,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/vebus/{device_id}/Alarms/HighTemperature",
        message_type=MetricKind.SENSOR,
        short_id="vebus_inverter_alarm_high_temperature",
        name="Inverter high temperature alarm",
        value_type=ValueType.ENUM,
        enum=GenericAlarmEnum,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/vebus/{device_id}/Alarms/LowBattery",
        message_type=MetricKind.SENSOR,
        short_id="vebus_inverter_alarm_low_battery",
        name="Inverter low battery alarm",
        value_type=ValueType.ENUM,
        enum=GenericAlarmEnum,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/vebus/{device_id}/Alarms/Overload",
        message_type=MetricKind.SENSOR,
        short_id="vebus_inverter_alarm_overload",
        name="Inverter overload alarm",
        value_type=ValueType.ENUM,
        enum=GenericAlarmEnum,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/vebus/{device_id}/Alarms/PhaseRotation",
        message_type=MetricKind.SENSOR,
        short_id="vebus_inverter_alarm_phase_rotation",
        name="Inverter phase rotation alarm",
        value_type=ValueType.ENUM,
        enum=GenericAlarmEnum,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/vebus/{device_id}/Alarms/Ripple",
        message_type=MetricKind.SENSOR,
        short_id="vebus_inverter_alarm_ripple",
        name="Inverter ripple alarm",
        value_type=ValueType.ENUM,
        enum=GenericAlarmEnum,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/vebus/{device_id}/Alarms/TemperatureSensor",
        message_type=MetricKind.SENSOR,
        short_id="vebus_inverter_alarm_temperature_sensor",
        name="Inverter temperature sensor alarm",
        value_type=ValueType.ENUM,
        enum=GenericAlarmEnum,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/vebus/{device_id}/Alarms/VoltageSensor",
        message_type=MetricKind.SENSOR,
        short_id="vebus_inverter_alarm_voltage_sensor",
        name="Inverter voltage sensor alarm",
        value_type=ValueType.ENUM,
        enum=GenericAlarmEnum,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/vebus/{device_id}/Connected",
        message_type=MetricKind.BINARY_SENSOR,
        short_id="vebus_inverter_connected",
        name="Inverter connected",
        value_type=ValueType.ENUM,
        enum=GenericOnOff,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/vebus/{device_id}/Dc/0/Current",
        message_type=MetricKind.SENSOR,
        short_id="vebus_inverter_dc_current",
        name="Inverter DC current",
        metric_type=MetricType.CURRENT,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/vebus/{device_id}/Dc/0/Power",
        message_type=MetricKind.SENSOR,
        short_id="vebus_inverter_dc_power",
        name="Inverter DC power",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/vebus/{device_id}/Dc/0/Temperature",
        message_type=MetricKind.SENSOR,
        short_id="vebus_inverter_dc_temperature",
        name="Inverter DC temperature",
        metric_type=MetricType.TEMPERATURE,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/vebus/{device_id}/Devices/{device_number}/Ac/In/P",
        message_type=MetricKind.SENSOR,
        short_id="vebus_device_{device_number}_input_power_l1",
        name="VE.Bus device {device_number} line 1 input power",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/vebus/{device_id}/Devices/{device_number}/Ac/In/{phase}/P",
        message_type=MetricKind.SENSOR,
        short_id="vebus_device_{device_number}_input_power_{phase}",
        name="VE.Bus device {device_number} line {phase} input power",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/vebus/{device_id}/Devices/{device_number}/Ac/Inverter/P",
        message_type=MetricKind.SENSOR,
        short_id="vebus_device_{device_number}_inverted_power",
        name="VE.Bus device {device_number} inverted power",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/vebus/{device_id}/Devices/{device_number}/Ac/Out/P",
        message_type=MetricKind.SENSOR,
        short_id="vebus_device_{device_number}_output_power_l1",
        name="VE.Bus device {device_number} line 1 output power",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/vebus/{device_id}/Devices/{device_number}/Ac/Out/{phase}/P",
        message_type=MetricKind.SENSOR,
        short_id="vebus_device_{device_number}_output_power_{phase}",
        name="VE.Bus device {device_number} line {phase} output power",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/vebus/{device_id}/Energy/AcIn1ToAcOut",
        message_type=MetricKind.SENSOR,
        short_id="vebus_energy_ac_in1_to_ac_out",
        name="Energy from AcIn1 to AcOut",
        metric_type=MetricType.ENERGY,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/vebus/{device_id}/Energy/AcIn1ToInverter",
        message_type=MetricKind.SENSOR,
        short_id="vebus_energy_ac_in1_to_inverter",
        name="Energy from AcIn1 to inverter",
        metric_type=MetricType.ENERGY,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/vebus/{device_id}/Energy/AcIn2ToAcOut",
        message_type=MetricKind.SENSOR,
        short_id="vebus_energy_ac_in2_to_ac_out",
        name="Energy from AcIn2 to AcOut",
        metric_type=MetricType.ENERGY,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/vebus/{device_id}/Energy/AcIn2ToInverter",
        message_type=MetricKind.SENSOR,
        short_id="vebus_energy_ac_in2_to_inverter",
        name="Energy from AcIn2 to inverter",
        metric_type=MetricType.ENERGY,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/vebus/{device_id}/Energy/AcOutToAcIn1",
        message_type=MetricKind.SENSOR,
        short_id="vebus_energy_ac_out_to_ac_in1",
        name="Energy from AcOut to AcIn1",
        metric_type=MetricType.ENERGY,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/vebus/{device_id}/Energy/AcOutToAcIn2",
        message_type=MetricKind.SENSOR,
        short_id="vebus_energy_ac_out_to_ac_in2",
        name="Energy from AcOut to AcIn2",
        metric_type=MetricType.ENERGY,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/vebus/{device_id}/Energy/InverterToAcIn1",
        message_type=MetricKind.SENSOR,
        short_id="vebus_energy_inverter_to_ac_in1",
        name="Energy from Inverter to AcIn1",
        metric_type=MetricType.ENERGY,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/vebus/{device_id}/Energy/InverterToAcIn2",
        message_type=MetricKind.SENSOR,
        short_id="vebus_energy_inverter_to_ac_in2",
        name="Energy from Inverter to AcIn2",
        metric_type=MetricType.ENERGY,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/vebus/{device_id}/Energy/InverterToAcOut",
        message_type=MetricKind.SENSOR,
        short_id="vebus_energy_inverter_to_ac_out",
        name="Energy from Inverter to AcOut",
        metric_type=MetricType.ENERGY,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/vebus/{device_id}/Energy/OutToInverter",
        message_type=MetricKind.SENSOR,
        short_id="vebus_energy_out_to_inverter",
        name="Energy from Out to inverter",
        metric_type=MetricType.ENERGY,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/vebus/{device_id}/Mode",
        message_type=MetricKind.SELECT,
        short_id="vebus_inverter_mode",
        name="Inverter mode",
        value_type=ValueType.ENUM,
        enum=InverterMode,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/vebus/{device_id}/Settings/Alarm/System/GridLost",
        message_type=MetricKind.SWITCH,
        short_id="vebus_inverter_setting_alarm_grid_lost",
        name="Loss of grid alarm setting",
        value_type=ValueType.ENUM,
        enum=GenericOnOff,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/vebus/{device_id}/Settings/AssistCurrentBoostFactor",
        message_type=MetricKind.NUMBER,
        short_id="multiplus_assist_current_boost_factor",
        name="Assist Current Boost Factor",
        value_type=ValueType.FLOAT,
        min=0.25,
        max=3.5,
        step=0.125,
        precision=3,
    ),
    TopicDescriptor(
        topic="N/{installation_id}/vebus/{device_id}/State",
        message_type=MetricKind.SENSOR,
        short_id="vebus_inverter_state",
        name="Inverter state",
        value_type=ValueType.ENUM,
        enum=State,
    ),
    # All service topics comes last
    TopicDescriptor(
        topic="W/{installation_id}/generator/{device_id}/ServiceCounterReset",
        message_type=MetricKind.SERVICE,
        short_id="generator_service_counter_reset",
        name="Generator service counter reset",
        value_type=ValueType.INT,
    ),
]
