from typing import Callable, Literal, Optional, TypeAlias

from django.http import HttpRequest, HttpResponse
from django_htmx.http import push_url

from dfv.htmx import swap_oob
from dfv.view_stack import get_view_fn_call_stack_from_request_or_raise


RESPONSE_HANDLER: TypeAlias = Callable[[HttpResponse], Optional[HttpResponse]]


def get_response_handlers_from_request(request: HttpRequest) -> list[RESPONSE_HANDLER]:
    handlers = getattr(request, "__dfv_view_response_handlers", [])
    setattr(request, "__dfv_view_response_handlers", handlers)
    return handlers


def add_response_handler(request: HttpRequest, handler: RESPONSE_HANDLER):
    get_view_fn_call_stack_from_request_or_raise(request)
    handlers = get_response_handlers_from_request(request)
    handlers.append(handler)


def process_response(request: HttpRequest, response: HttpResponse) -> HttpResponse:
    handlers = get_response_handlers_from_request(request)
    for handler in handlers:
        result = handler(response)
        response = result if result is not None else response
    return response


def hook_push_url(request: HttpRequest, url: str | Literal[False]):
    add_response_handler(request, lambda response: push_url(response, url))


def hook_swap_oob(
    request: HttpRequest,
    additional: HttpResponse | list[HttpResponse],
    hx_swap_oob_method="outerHTML",
) -> None:
    add_response_handler(request, lambda r: swap_oob(r, additional, hx_swap_oob_method))
