# pylint: disable=unknown-option-value
# pylint: disable=too-many-positional-arguments, line-too-long
"""
Module for perturbing datasets for calibration purposes.

This module provides functions to apply various types of perturbations to
dataset columns, including categorical and numerical perturbations.

Functions
---------
categorical_perturbation(column, num_permutations=5)
    Provide categorical perturbation to datasets.

gaussian_perturbation(column, severity)
    Apply Gaussian noise as numerical perturbation to a column in a DataFrame.

uniform_perturbation(column, severity)
    Apply uniform noise as numerical perturbation to a column in a DataFrame.

perturb_dataset(X_cal, y_cal, categorical_features=None, noise_type='uniform', scale_factor=5, severity=0.5)
    Perturb the dataset for the calibration process.
"""

# Import Libraries
# import configparser
from typing import Optional

import numpy as np

# # Create a ConfigParser object
# config = configparser.ConfigParser()

# # Read the config.ini file
# # Please set the path properly before usage
# config.read('./configurations/config.ini')

# LOW = config.getfloat('perturbation_variables', 'LOW')
# HIGH = config.getfloat('perturbation_variables', 'HIGH')
# STEP = config.getfloat('perturbation_variables', 'STEP')
# Now Let's write functions to provide perturbations to each column specific to each data type


# BASIC PERTURBATIONS FOR THE VERSION I: Provides Gaussian Noise by protecting
# the standard deviation and mean properties of the current column.
def categorical_perturbation(column, num_permutations=5, rng: Optional[np.random.Generator] = None):
    """
    Provide categorical perturbation to datasets.

    Parameters
    ----------
        X_Cal (pandas.DataFrame): Input DataFrame.

    Returns
    -------
        tuple: A tuple containing lists of perturbation column names, types,
                severities, and perturbed datasets.
    """
    column = column.copy()  # Make a copy to avoid modifying the original array
    # Use provided RNG if available; otherwise, create a local generator
    local_rng = rng if rng is not None else np.random.default_rng()
    # Generate at least one permutation; repeat attempts up to num_permutations times
    # to reduce the chance of returning an identical array for short arrays.
    attempts = max(1, int(num_permutations))
    column_perturbed = column
    for _ in range(attempts):
        candidate = local_rng.permutation(column)
        if not np.array_equal(candidate, column):
            column_perturbed = candidate
            break
    else:
        # As a deterministic fallback for tiny arrays or degenerate RNG states,
        # force a minimal change by swapping two positions when possible.
        if column.size > 1 and len(np.unique(column)) > 1:
            i, j = 0, 1
            column_perturbed = column.copy()
            column_perturbed[i], column_perturbed[j] = column_perturbed[j], column_perturbed[i]
        else:
            column_perturbed = column.copy()
    return column_perturbed


# Assuming you have a function to generate alternative instances for numerical columns
def gaussian_perturbation(column, severity, rng: Optional[np.random.Generator] = None):
    """
    Apply Gaussian noise as numerical perturbation to a column in a DataFrame.

    Parameters
    ----------
        data (pandas.DataFrame): Input DataFrame.
        column_name (str): Name of the column to perturb.
        severity (float): Severity of the perturbation.

    Returns
    -------
        pandas.DataFrame: DataFrame with perturbed column.
    """
    # Get the column to perturb
    column = column.copy()

    # Calculate mean and standard deviation of the original column
    original_mean = column.mean()
    original_std = column.std()

    # Generate perturbation based on severity
    local_rng = rng if rng is not None else np.random.default_rng()
    perturbation = local_rng.normal(loc=0, scale=original_std * severity, size=len(column))

    # Apply perturbation while preserving mean and standard deviation
    return original_mean + perturbation


def uniform_perturbation(column, severity, rng: Optional[np.random.Generator] = None):
    """
    Apply uniform noise as numerical perturbation to a column in a DataFrame.

    Parameters
    ----------
        data (pandas.DataFrame): Input DataFrame.
        column_name (str): Name of the column to perturb.
        severity (float): Severity of the perturbation.

    Returns
    -------
        pandas.DataFrame: DataFrame with perturbed column.
    """
    # Get the column to perturb
    column = column.copy()

    # Calculate mean and standard deviation of the original column
    # original_mean = column.mean()
    original_range = column.max() - column.min()

    # Generate perturbation based on severity
    local_rng = rng if rng is not None else np.random.default_rng()
    perturbation = local_rng.uniform(
        low=-original_range * severity, high=original_range * severity, size=len(column)
    )

    # Apply perturbation while preserving mean
    return column + perturbation


# pylint: disable=invalid-name, too-many-arguments
def perturb_dataset(
    X_cal,
    y_cal,
    categorical_features=None,
    noise_type="uniform",
    scale_factor=5,
    severity=0.5,
    *,
    seed: Optional[int] = None,
    rng: Optional[np.random.Generator] = None,
):
    """
    Perturb the dataset for the calibration process.

    Parameters
    ----------
    X_cal : numpy.ndarray
        Input feature matrix.
    y_cal : numpy.ndarray
        Input target vector.
    categorical_features : list, optional
        List of indices for categorical features.
    noise_type : str, optional
        Type of noise to apply ('uniform' or 'gaussian').
    scale_factor : int, optional
        Factor by which to scale the dataset.
    severity : float, optional
        Severity of the perturbation.

    Returns
    -------
    tuple
        Tuple containing perturbed feature matrix, scaled feature matrix,
        scaled target vector, and scale factor.
    """
    perturbed_X_cal = np.tile(X_cal.copy(), (scale_factor, 1))
    scaled_X_cal = perturbed_X_cal.copy()
    scaled_y_cal = np.tile(y_cal.copy(), scale_factor)
    if noise_type not in [
        "uniform",
        "gaussian",
    ]:
        raise ValueError("Noise type must be either 'uniform' or 'gaussian'.")

    # Create a single RNG to be used throughout perturbations to ensure reproducibility
    local_rng = (
        rng
        if rng is not None
        else (np.random.default_rng(seed) if seed is not None else np.random.default_rng())
    )

    for f in range(scaled_X_cal.shape[1]):
        if f in categorical_features:
            perturbed_X_cal[:, f] = categorical_perturbation(perturbed_X_cal[:, f], rng=local_rng)
        elif noise_type == "uniform":
            # Apply numerical alternative perturbation to the selected column -- uniform
            perturbed_X_cal[:, f] = uniform_perturbation(
                perturbed_X_cal[:, f], severity, rng=local_rng
            )
        elif noise_type == "gaussian":
            # Apply numerical alternative perturbation to the selected column -- gaussian
            perturbed_X_cal[:, f] = gaussian_perturbation(
                perturbed_X_cal[:, f], severity, rng=local_rng
            )
    return perturbed_X_cal, scaled_X_cal, scaled_y_cal, scale_factor
