/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
import { aws_ec2 as ec2, aws_secretsmanager as secretsmanager } from 'aws-cdk-lib';
import { CfnMicrosoftAD } from 'aws-cdk-lib/aws-directoryservice';
import { ISecret } from 'aws-cdk-lib/aws-secretsmanager';
import { Construct } from 'constructs';
import * as skylight from '../index';
/**
 * The properties for the AwsManagedMicrosoftAd class.
 */
export interface IAwsManagedMicrosoftAdProps {
    /**
     * The domain name for the Active Directory Domain.
     *
     * @default - 'domain.aws'.
     */
    domainName?: string;
    /**
     * The edition to use for the Active Directory Domain.
     * Allowed values: Enterprise | Standard
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-directoryservice-microsoftad.html#cfn-directoryservice-microsoftad-edition
     * @default - 'Standard'.
     */
    edition?: string;
    /**
     * The secrets manager secret to use must be in format:
     * '{Domain: <domain.name>, UserID: 'Admin', Password: '<password>'}'
     * @default - 'Randomly generated and stored in Secret Manager'.
     */
    secret?: secretsmanager.ISecret;
    /**
     * The secret name to save the Domain Admin object
     * @default - '<domain.name>-secret'.
     */
    secretName?: string;
    /**
     * The VPC to use, must have private subnets.
     */
    vpc: ec2.IVpc;
    /**
     * VPC subnet selection, subnets must be private and exactly 2
     */
    vpcSubnets?: ec2.SelectedSubnets;
    /**
     * The configuration store to save the directory parameters (After deployed)
     */
    configurationStore?: IAwsManagedMicrosoftAdParameters;
    /**
     * Create Domain joined machine to be used to run Powershell commands to that directory. (i.e Create Ad Group)
     * @default - 'true'.
     */
    createWorker?: boolean;
}
export declare enum AwsManagedMicrosoftConfigurationStoreType {
    SSM = "AWS Systems Manager Parameter Store"
}
/**
 * The properties of an DomainWindowsNodeProps, requires Active Directory parameter to read the Secret to join the domain
 * Default setting: Domain joined, m5.2xlarge, latest windows, Managed by SSM.
 */
export interface IAwsManagedMicrosoftAdParameters {
    /**
     * The name of the Configuration Store Type to use
     * @default - 'AWS Systems Manager Parameter Store'.
     */
    configurationStoreType?: AwsManagedMicrosoftConfigurationStoreType;
    /**
     * The name of the SSM Object that contains the secret name in Secrets Manager
     * @default - 'domain-secret'.
     */
    secretPointer?: string;
    /**
     * The name of the SSM Object that contains the Directory ID
     * @default - 'directoryID'.
     */
    directoryIDPointer?: string;
    /**
     * The SSM namespace to read/write parameters to
     * @default - 'cdk-skylight'.
     */
    namespace?: string;
}
/**
 * A Ad Authentication represents an integration pattern of Managed AD and Route 53 Resolver in a specific VPC
 *
 * The Construct creates Managed AD with the provided Secret (Secrets Manager) or generates a new Secret.
 * The secret saved to SSM parameter store so others can use it with other Constructs (Such as Windows node or FSx)
 * The provided VPC or the new created VPC will be configured to forward DNS requests to the Managed AD with Route53 Resolvers
 * The construct also creates (optionally) t3.nano machine that is part of the domain that can be used to run admin-tasks (such as createADGroup)
 *
 * The createADGroup() method creates an Active Directory permission group in the domain, using the domain admin user.
 * Please note: When calling createADGroup() API, a Lambda will be created to start the worker machine (Using AWS-SDK),
 * then each command will be scheduled with State Manager, and the instance will be shut down after complete.
 *
 */
export declare class AwsManagedMicrosoftAd extends Construct {
    readonly microsoftAD: CfnMicrosoftAD;
    readonly adParameters: IAwsManagedMicrosoftAdParameters;
    readonly props: IAwsManagedMicrosoftAdProps;
    readonly domainWindowsNode?: skylight.compute.DomainWindowsNode;
    readonly secret: ISecret;
    constructor(scope: Construct, id: string, props: IAwsManagedMicrosoftAdProps);
    createWorker(domainName: string, domainPassword: ISecret): skylight.compute.DomainWindowsNode;
    createADGroup(groupName: string, groupDescription: string): void;
    createServiceAccount(adServiceAccountName: string, servicePrincipalNames: string, principalsAllowedToRetrieveManagedPassword: string): void;
}
/**
 * A Ad Authentication represents an integration pattern of Managed AD and Route 53 Resolver in a specific VPC
 *
 * The Construct creates Managed AD with the provided Secret (Secrets Manager) or generates a new Secret.
 * The secret saved to SSM parameter store so others can use it with other Constructs (Such as Windows node or FSx)
 * The provided VPC or the new created VPC will be configured to forward DNS requests to the Managed AD with Route53 Resolvers
 * The construct also creates (optionally) t3.nano machine that is part of the domain that can be used to run admin-tasks (such as createADGroup)
 *
 * The createADGroup() method creates an Active Directory permission group in the domain, using the domain admin user.
 * Please note: When calling createADGroup() API, a Lambda will be created to start the worker machine (Using AWS-SDK),
 * then each command will be scheduled with State Manager, and the instance will be shut down after complete.
 *
 */
export declare class AwsManagedMicrosoftAdR53 extends AwsManagedMicrosoftAd {
    constructor(scope: Construct, id: string, props: IAwsManagedMicrosoftAdProps);
}
