"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AwsManagedMicrosoftAdR53 = exports.AwsManagedMicrosoftAd = exports.AwsManagedMicrosoftConfigurationStoreType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const skylight = require("../index");
var AwsManagedMicrosoftConfigurationStoreType;
(function (AwsManagedMicrosoftConfigurationStoreType) {
    AwsManagedMicrosoftConfigurationStoreType["SSM"] = "AWS Systems Manager Parameter Store";
})(AwsManagedMicrosoftConfigurationStoreType = exports.AwsManagedMicrosoftConfigurationStoreType || (exports.AwsManagedMicrosoftConfigurationStoreType = {}));
/**
 * A Ad Authentication represents an integration pattern of Managed AD and Route 53 Resolver in a specific VPC
 *
 * The Construct creates Managed AD with the provided Secret (Secrets Manager) or generates a new Secret.
 * The secret saved to SSM parameter store so others can use it with other Constructs (Such as Windows node or FSx)
 * The provided VPC or the new created VPC will be configured to forward DNS requests to the Managed AD with Route53 Resolvers
 * The construct also creates (optionally) t3.nano machine that is part of the domain that can be used to run admin-tasks (such as createADGroup)
 *
 * The createADGroup() method creates an Active Directory permission group in the domain, using the domain admin user.
 * Please note: When calling createADGroup() API, a Lambda will be created to start the worker machine (Using AWS-SDK),
 * then each command will be scheduled with State Manager, and the instance will be shut down after complete.
 *
 */
class AwsManagedMicrosoftAd extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.props.domainName = props.domainName ?? 'domain.aws';
        this.props.edition = props.edition ?? 'Standard';
        this.props.secretName = props.secretName ?? `${props.domainName}-secret`;
        this.props.createWorker = props.createWorker ?? true;
        this.adParameters = props.configurationStore ?? {
            configurationStoreType: AwsManagedMicrosoftConfigurationStoreType.SSM,
        };
        this.adParameters.secretPointer =
            this.adParameters.secretPointer ?? 'domain-secret';
        this.adParameters.directoryIDPointer =
            this.adParameters.directoryIDPointer ?? 'directoryID';
        if (this.adParameters.namespace) {
            this.adParameters.namespace = `${this.adParameters.namespace}/authentication/mad`;
        }
        else {
            this.adParameters.namespace = 'cdk-skylight/authentication/mad';
        }
        this.secret =
            this.props.secret ??
                new aws_cdk_lib_1.aws_secretsmanager.Secret(this, 'Secret', {
                    generateSecretString: {
                        secretStringTemplate: JSON.stringify({
                            Domain: props.domainName,
                            UserID: 'Admin',
                        }),
                        generateStringKey: 'Password',
                        excludePunctuation: true,
                    },
                    secretName: props.secretName,
                });
        new aws_cdk_lib_1.aws_ssm.StringParameter(this, 'mad-secretName-pointer', {
            parameterName: `/${this.adParameters.namespace}/${this.adParameters.secretPointer}`,
            stringValue: this.props.secretName,
        });
        let subnets;
        if (props.vpcSubnets) {
            if (props.vpcSubnets.hasPublic || props.vpcSubnets.subnets.length !== 2) {
                throw new Error('A public subnet or not exactly 2 subnets where passed in, please pass in two private subnets');
            }
            subnets = props.vpcSubnets;
        }
        else {
            subnets =
                props.vpc.selectSubnets({
                    subnetType: aws_cdk_lib_1.aws_ec2.SubnetType.PRIVATE_WITH_NAT,
                }) ??
                    props.vpc.selectSubnets({
                        subnetType: aws_cdk_lib_1.aws_ec2.SubnetType.PRIVATE_ISOLATED,
                    });
        }
        new aws_cdk_lib_1.CfnOutput(this, 'secret-value-hint', {
            value: `aws secretsmanager get-secret-value --secret-id ${this.secret.secretArn} --query SecretString --output text --region ${aws_cdk_lib_1.Stack.of(scope).region}`,
        });
        this.microsoftAD = new aws_cdk_lib_1.aws_directoryservice.CfnMicrosoftAD(this, 'AWS-Managed-Microsoft-AD', {
            password: this.secret
                .secretValueFromJson('Password')
                .unsafeUnwrap.toString(),
            edition: props.edition,
            name: this.props.domainName,
            vpcSettings: {
                subnetIds: [subnets.subnetIds[0], subnets.subnetIds[1]],
                vpcId: props.vpc.vpcId,
            },
        });
        new aws_cdk_lib_1.CfnOutput(this, 'mad-dns-ips', {
            value: `${aws_cdk_lib_1.Fn.join(',', this.microsoftAD.attrDnsIpAddresses)}`,
        });
        new aws_cdk_lib_1.CfnOutput(this, 'mad-dns-name', {
            value: `${this.props.domainName}`,
        });
        new aws_cdk_lib_1.CfnOutput(this, 'mad-directoyID', {
            value: `${this.microsoftAD.ref}`,
        });
        new aws_cdk_lib_1.aws_ssm.StringParameter(this, 'mad-directoryID-pointer', {
            parameterName: `/${this.adParameters.namespace}/${this.adParameters.directoryIDPointer}`,
            stringValue: this.microsoftAD.ref,
        });
        if (this.props.createWorker) {
            this.domainWindowsNode = this.createWorker(this.props.domainName, this.secret);
            this.domainWindowsNode.runPSwithDomainAdmin([
                'Add-WindowsFeature RSAT-AD-PowerShell',
                'Stop-Computer -ComputerName localhost',
            ], 'ad-powershell');
            this.domainWindowsNode.instance.node.addDependency(this.microsoftAD);
        }
        else {
            this.domainWindowsNode = undefined;
        }
    }
    // Creates DomainWindowsNode that will be used to run admin-tasks to this directory
    createWorker(domainName, domainPassword) {
        return new skylight.compute.DomainWindowsNode(this, 'madWorker', {
            domainName: domainName,
            passwordObject: domainPassword,
            vpc: this.props.vpc,
            instanceType: 't3.small',
            usePrivateSubnet: true,
        });
    }
    // The function creates a Lambda to Start the Windows Worker, then creates SSM Document and Desired state in State Manager to schedule this document on the Worker.
    createADGroup(groupName, groupDescription) {
        if (this.domainWindowsNode) {
            this.domainWindowsNode.startInstance();
            this.domainWindowsNode.runPSwithDomainAdmin([
                `New-ADGroup -Name "${groupDescription}" -SamAccountName "${groupName}" -GroupScope DomainLocal`,
                'Stop-Computer -ComputerName localhost',
            ], 'createAdGroup');
        }
        else {
            console.log("Can't create AD group when no Worker is defined");
        }
    }
    // Experimental
    createServiceAccount(adServiceAccountName, servicePrincipalNames, principalsAllowedToRetrieveManagedPassword) {
        if (this.domainWindowsNode) {
            this.domainWindowsNode.runPSwithDomainAdmin([
                `New-ADServiceAccount -Name "${adServiceAccountName}" -DnsHostName "${adServiceAccountName}.${this.props.domainName}" -ServicePrincipalNames "${servicePrincipalNames}" -PrincipalsAllowedToRetrieveManagedPassword "${principalsAllowedToRetrieveManagedPassword}"`,
            ], 'createServiceAccount');
        }
        else {
            console.log("Can't createServiceAccount when no Worker is defined");
        }
    }
}
exports.AwsManagedMicrosoftAd = AwsManagedMicrosoftAd;
_a = JSII_RTTI_SYMBOL_1;
AwsManagedMicrosoftAd[_a] = { fqn: "cdk-skylight.authentication.AwsManagedMicrosoftAd", version: "1.1.492" };
/**
 * A Ad Authentication represents an integration pattern of Managed AD and Route 53 Resolver in a specific VPC
 *
 * The Construct creates Managed AD with the provided Secret (Secrets Manager) or generates a new Secret.
 * The secret saved to SSM parameter store so others can use it with other Constructs (Such as Windows node or FSx)
 * The provided VPC or the new created VPC will be configured to forward DNS requests to the Managed AD with Route53 Resolvers
 * The construct also creates (optionally) t3.nano machine that is part of the domain that can be used to run admin-tasks (such as createADGroup)
 *
 * The createADGroup() method creates an Active Directory permission group in the domain, using the domain admin user.
 * Please note: When calling createADGroup() API, a Lambda will be created to start the worker machine (Using AWS-SDK),
 * then each command will be scheduled with State Manager, and the instance will be shut down after complete.
 *
 */
class AwsManagedMicrosoftAdR53 extends AwsManagedMicrosoftAd {
    constructor(scope, id, props) {
        super(scope, id, props);
        let subnets;
        if (props.vpcSubnets) {
            if (props.vpcSubnets.hasPublic || props.vpcSubnets.subnets.length !== 2) {
                throw new Error('A public subnet or not exactly 2 subnets where passed in, please pass in two private subnets');
            }
            subnets = props.vpcSubnets;
        }
        else {
            subnets =
                props.vpc.selectSubnets({
                    subnetType: aws_cdk_lib_1.aws_ec2.SubnetType.PRIVATE_WITH_NAT,
                }) ??
                    props.vpc.selectSubnets({
                        subnetType: aws_cdk_lib_1.aws_ec2.SubnetType.PRIVATE_ISOLATED,
                    });
        }
        const sg = new aws_cdk_lib_1.aws_ec2.SecurityGroup(this, 'r53-outbound-resolver-SG', {
            vpc: props.vpc,
        });
        sg.addIngressRule(aws_cdk_lib_1.aws_ec2.Peer.ipv4(props.vpc.vpcCidrBlock), aws_cdk_lib_1.aws_ec2.Port.udp(53));
        sg.addIngressRule(aws_cdk_lib_1.aws_ec2.Peer.ipv4(props.vpc.vpcCidrBlock), aws_cdk_lib_1.aws_ec2.Port.tcp(53));
        const outBoundResolver = new aws_cdk_lib_1.aws_route53resolver.CfnResolverEndpoint(this, 'R53-Resolver-Endpoint', {
            direction: 'OUTBOUND',
            ipAddresses: subnets.subnetIds.map((s) => {
                return { subnetId: s };
            }),
            securityGroupIds: [sg.securityGroupId],
        });
        const resolverRules = new aws_cdk_lib_1.aws_route53resolver.CfnResolverRule(this, 'R53-Resolve-Rule', {
            domainName: this.props.domainName,
            resolverEndpointId: outBoundResolver.ref,
            ruleType: 'FORWARD',
            targetIps: [
                { ip: aws_cdk_lib_1.Fn.select(0, this.microsoftAD.attrDnsIpAddresses) },
                { ip: aws_cdk_lib_1.Fn.select(1, this.microsoftAD.attrDnsIpAddresses) },
            ],
        });
        new aws_cdk_lib_1.aws_route53resolver.CfnResolverRuleAssociation(this, 'R53-Resolver-Association', {
            resolverRuleId: resolverRules.attrResolverRuleId,
            vpcId: props.vpc.vpcId,
        });
    }
}
exports.AwsManagedMicrosoftAdR53 = AwsManagedMicrosoftAdR53;
_b = JSII_RTTI_SYMBOL_1;
AwsManagedMicrosoftAdR53[_b] = { fqn: "cdk-skylight.authentication.AwsManagedMicrosoftAdR53", version: "1.1.492" };
//# sourceMappingURL=data:application/json;base64,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