/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
import { aws_autoscaling, aws_ec2, aws_eks, aws_iam } from 'aws-cdk-lib';
import { AutoScalingGroup } from 'aws-cdk-lib/aws-autoscaling';
import { Construct } from 'constructs';
import * as skylight from '../../index';
export interface IRuntimeNodes {
    /**
     * Method to add userData to the nodes
     */
    addUserData(...commands: string[]): void;
    /**
     * Method to configure the Nodes to part of AD Domain
     * Secret: The secrets manager secret to use must be in format:
     * '{Domain: <domain.name>, UserID: 'Admin', Password: '<password>'}' (From cdk-skylight.AwsManagedMicrosoftAdR53 Object)
     */
    addAdDependency?(adParametersStore: skylight.authentication.IAwsManagedMicrosoftAdParameters): void;
    /**
     * Method to configure persistent storage dependency to the hosts by using Global Mapping.
     */
    addStorageDependency(adParametersStore: skylight.authentication.IAwsManagedMicrosoftAdParameters, fsxParametersStore: skylight.storage.IFSxWindowsParameters, folderName: string): void;
    /**
     * Method to add the nodes to specific Cluster
     */
    addEKSDependency?(eksCluster: aws_eks.Cluster): void;
    /**
     * Method to add support for LocalCredFile <Experimental>
     */
    addLocalCredFile?(adParametersStore: skylight.authentication.IAwsManagedMicrosoftAdParameters, ADGroupName: string, AccountName: string): void;
}
export interface IWindowsEKSNodesProps {
    vpc: aws_ec2.IVpc;
    /**
     * The SSM namespace to save parameters to
     * @default - 'cdk-skylight'.
     */
    namespace?: string;
    /**
     * The instance to use
     * @default - 'm5.large'.
     */
    instanceType?: aws_ec2.InstanceType;
}
export declare class WindowsEKSNodes extends Construct implements IRuntimeNodes {
    readonly asg: AutoScalingGroup;
    readonly windowsWorkersRole: aws_iam.Role;
    readonly asgResource: aws_autoscaling.CfnAutoScalingGroup;
    readonly vpc: aws_ec2.IVpc;
    readonly nodesSg: aws_ec2.SecurityGroup;
    constructor(scope: Construct, id: string, props: IWindowsEKSNodesProps);
    addUserData(...commands: string[]): void;
    addAdDependency(adParametersStore: skylight.authentication.IAwsManagedMicrosoftAdParameters): void;
    runPowerShellSSMDocument(name: string, commands: string[]): void;
    gMSAWebHookAutoInstall(eksCluster: aws_eks.Cluster, privateSignerName: string, awsaccountid: string, awsregion: string): void;
    addStorageDependency(adParametersStore: skylight.authentication.IAwsManagedMicrosoftAdParameters, fsxParametersStore: skylight.storage.IFSxWindowsParameters, folderName: string): void;
    addEKSDependency(eksCluster: aws_eks.Cluster): void;
    addLocalCredFile(adParametersStore: skylight.authentication.IAwsManagedMicrosoftAdParameters, ADGroupName: string, AccountName: string): void;
}
