/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
import { aws_iam as iam, aws_ec2 as ec2, aws_secretsmanager } from 'aws-cdk-lib';
import { ISecret } from 'aws-cdk-lib/aws-secretsmanager';
import { Construct } from 'constructs';
/**
 * The properties of an DomainWindowsNodeProps, requires Active Directory parameter to read the Secret to join the domain
 * Default setting: Domain joined, m5.2xlarge, latest windows, Managed by SSM.
 */
export interface IDomainWindowsNodeProps {
    /**
     * IAM Instance role permissions
     * @default - 'AmazonSSMManagedInstanceCore, AmazonSSMDirectoryServiceAccess'.
     */
    iamManagedPoliciesList?: iam.IManagedPolicy[];
    /**
     * The EC2 Instance type to use
     *
     * @default - 'm5.2xlarge'.
     */
    instanceType?: string;
    /**
     * Choose if to launch the instance in Private or in Public subnet
     * Private = Subnet that routes to the internet, but not vice versa.
     * Public = Subnet that routes to the internet and vice versa.
     * @default - Private.
     */
    usePrivateSubnet?: boolean;
    /**
     * The name of the AMI to search in SSM (ec2.LookupNodeImage) supports Regex
     *  @default - 'Windows_Server-2022-English-Full'
     */
    amiName?: string;
    /**
     * Specific UserData to use
     *
     * The UserData may still be mutated after creation.
     *
     *  @default - 'undefined'
     */
    userData?: string;
    domainName?: string;
    passwordObject?: aws_secretsmanager.ISecret;
    /**
     * @default - 'true'
     */
    windowsMachine?: boolean;
    /**
     * The VPC to use
     */
    vpc: ec2.IVpc;
}
/**
 * A Domain Windows Node represents one Windows EC2 instance configured with Active Directory.
 *
 * The DomainWindowsNode can be customized to different instance sizes and additional permissions set just like any other EC2 Instance.
 * You can use this construct to run elevated domain tasks with domain permissions or run your application in a single instance setup.
 *
 * The machine will be joined to the provided Active Directory domain using a custom CloudFormation bootstrap that will wait until the required reboot to join the domain. Then it will register the machine in SSM and pull tasks from the SSM State manager.
 *
 * You can send tasks to that machine using the provided methods: runPsCommands() and runPSwithDomainAdmin()
 *
 */
export declare class DomainWindowsNode extends Construct {
    readonly instance: ec2.Instance;
    readonly nodeRole: iam.Role;
    readonly vpc: ec2.IVpc;
    readonly passwordObject?: ISecret;
    constructor(scope: Construct, id: string, props: IDomainWindowsNodeProps);
    /**
     * Running bash scripts on the Node with SSM Document.
     * i.e: runPsCommands(["echo 'hello world'", "echo 'Second command'"], "myScript")
     */
    runShellCommands(ShellCommands: string[], id: string): void;
    /**
     * Running PowerShell scripts on the Node with SSM Document.
     * i.e: runPsCommands(["Write-host 'Hello world'", "Write-host 'Second command'"], "myScript")
     */
    runPsCommands(psCommands: string[], id: string): void;
    /**
     * Open the security group of the Node Node to specific IP address on port 3389
     * i.e: openRDP("1.1.1.1/32")
     */
    openRDP(ipaddress: string): void;
    /**
     * Running PowerShell scripts on the Node with SSM Document with Domain Admin (Using the Secret used to join the machine to the domain)
     * i.e: runPsCommands(["Write-host 'Hello world'", "Write-host 'Second command'"], "myScript")
     * The provided psCommands will be stored in C:\Scripts and will be run with scheduled task with Domain Admin rights
     */
    runPSwithDomainAdmin(psCommands: string[], id: string): void;
    startInstance(): void;
}
