import os
import logging
import base64
from PySide6 import QtCore, QtWidgets

AICODEPREP_GUI_VERSION = "1.0"


def _prefs_path():
    """Get the path to the preferences file, preferring .aicodeprep-gui, with .auicp as legacy for migration"""
    new_path = os.path.join(os.getcwd(), ".aicodeprep-gui")
    legacy_path = os.path.join(os.getcwd(), ".auicp")
    if os.path.exists(new_path):
        return new_path
    elif os.path.exists(legacy_path):
        return legacy_path
    else:
        return new_path


def _write_prefs_file(checked_relpaths, window_size=None, splitter_state=None, output_format=None, pro_features=None):
    """Write preferences to .aicodeprep-gui file, now supports [format] and [pro] sections."""
    new_path = os.path.join(os.getcwd(), ".aicodeprep-gui")
    try:
        with open(new_path, "w", encoding="utf-8") as f:
            header = (
                f"# .aicodeprep-gui LLM/AI context helper settings file\n"
                f"# This file stores your preferences (checked code files, window size) for this folder.\n"
                f"# Generated by aicodeprep-gui.\n"
                f"# Homepage: https://wuu73.org/aicp\n"
                f"# GitHub: https://github.com/detroittommy879/aicodeprep-gui\n"
                f"# ----------------------------------------------------------\n"
                f"# aicodeprep-gui preferences file version {AICODEPREP_GUI_VERSION}\n"
            )
            f.write(header)
            f.write(f"version={AICODEPREP_GUI_VERSION}\n\n")
            if window_size:
                f.write(
                    f"[window]\nwidth={window_size[0]}\nheight={window_size[1]}\n")
                if splitter_state is not None:
                    splitter_data = base64.b64encode(
                        splitter_state).decode('utf-8')
                    f.write(f"splitter_state={splitter_data}\n")
                f.write("\n")
            if output_format in ("xml", "markdown"):
                f.write(f"[format]\noutput_format={output_format}\n\n")
            if pro_features:
                f.write("[pro]\n")
                for key, value in pro_features.items():
                    f.write(f"{key}={str(value).lower()}\n")
                f.write("\n")
            if checked_relpaths:
                f.write("[files]\n" + "\n".join(checked_relpaths) + "\n")
        logging.info(f"Saved preferences to {new_path}")
    except Exception as e:
        logging.warning(f"Could not write .aicodeprep-gui: {e}")


def _read_prefs_file():
    """Read preferences file with backwards compatibility for legacy .auicp files (migrates to .aicodeprep-gui).
    Returns checked, window_size, splitter_state, output_format, pro_features.
    """
    checked, window_size, splitter_state = set(), None, None
    width_val, height_val = None, None
    output_format = "xml"
    pro_features = {}

    legacy_path = os.path.join(os.getcwd(), ".auicp")
    new_path = os.path.join(os.getcwd(), ".aicodeprep-gui")

    prefs_path = _prefs_path()

    try:
        with open(prefs_path, "r", encoding="utf-8") as f:
            section = None
            for line in f.read().splitlines():
                if line.strip().startswith('[') and line.strip().endswith(']'):
                    section = line.strip()[1:-1]
                    continue
                if not section:
                    continue

                if section == "files":
                    if line.strip():
                        checked.add(line.strip())
                elif section == "window":
                    if line.startswith('width='):
                        try:
                            width_val = int(line.split('=')[1])
                        except (ValueError, IndexError):
                            pass
                    elif line.startswith('height='):
                        try:
                            height_val = int(line.split('=')[1])
                        except (ValueError, IndexError):
                            pass
                    elif line.startswith('splitter_state='):
                        try:
                            splitter_data = line.split('=', 1)[1]
                            splitter_state = base64.b64decode(
                                splitter_data.encode('utf-8'))
                        except Exception as e:
                            logging.warning(
                                f"Failed to decode splitter state: {e}")
                elif section == "format":
                    if line.startswith("output_format="):
                        val = line.split("=", 1)[1].strip().lower()
                        if val in ("xml", "markdown"):
                            output_format = val
                elif section == "pro":
                    if "=" in line:
                        key, value = line.split("=", 1)
                        key = key.strip()
                        value = value.strip().lower()
                        # Convert string boolean to actual boolean
                        if value in ("true", "false"):
                            pro_features[key] = value == "true"
                        else:
                            pro_features[key] = value

            if width_val is not None and height_val is not None:
                window_size = (width_val, height_val)

        if prefs_path == legacy_path and not os.path.exists(new_path):
            logging.info(
                "Migrating preferences from .auicp to .aicodeprep-gui")
            try:
                _write_prefs_file(list(checked), window_size,
                                  splitter_state, output_format)
                logging.info(
                    "Successfully migrated preferences to .aicodeprep-gui")
            except Exception as e:
                logging.error(f"Failed to migrate preferences: {e}")

    except FileNotFoundError:
        file_type = ".auicp" if prefs_path.endswith(
            ".auicp") else ".aicodeprep-gui"
        logging.info(f"{file_type} file not found, will create on save.")
    except Exception as e:
        logging.error(f"Error reading preferences file: {e}")

    return checked, window_size, splitter_state, output_format, pro_features


class PreferencesManager:
    def __init__(self, main_window):
        self.main_window = main_window
        self.checked_files_from_prefs = set()
        self.window_size_from_prefs = None
        self.splitter_state_from_prefs = None
        self.output_format_from_prefs = "xml"
        self.pro_features_from_prefs = {}
        # True only if a prefs file actually exists on disk
        self.prefs_file_exists = False
        # Backward-compat flag; set true only when prefs file exists
        self.prefs_loaded = False

    def load_prefs_if_exists(self):
        # Determine if a prefs file exists before reading so we don't override smart defaults when missing
        prefs_path = _prefs_path()
        self.prefs_file_exists = os.path.exists(prefs_path)

        checked, window_size, splitter_state, output_format, pro_features = _read_prefs_file()
        self.checked_files_from_prefs = checked
        self.window_size_from_prefs = window_size
        self.splitter_state_from_prefs = splitter_state
        self.output_format_from_prefs = output_format
        self.pro_features_from_prefs = pro_features

        # Only mark as loaded when a prefs file actually exists
        self.prefs_loaded = self.prefs_file_exists

    def save_prefs(self):
        checked_relpaths = []
        for rel_path, item in self.main_window.path_to_item.items():
            if item.checkState(0) == QtCore.Qt.Checked:
                file_path_abs = item.data(0, QtCore.Qt.UserRole)
                if file_path_abs and os.path.isfile(file_path_abs):
                    checked_relpaths.append(rel_path)
        size = self.main_window.size()
        splitter_state = self.main_window.splitter.saveState()
        fmt = self.main_window.format_combo.currentData()

        # Collect pro features state
        pro_features = {}
        if hasattr(self.main_window, 'preview_toggle'):
            pro_features['preview_window_enabled'] = self.main_window.preview_toggle.isChecked(
            )
        if hasattr(self.main_window, 'syntax_highlight_toggle'):
            pro_features['syntax_highlight_enabled'] = self.main_window.syntax_highlight_toggle.isChecked()

        _write_prefs_file(checked_relpaths, window_size=(
            size.width(), size.height()), splitter_state=splitter_state, output_format=fmt, pro_features=pro_features)
        self.main_window._save_prompt_options()

    def load_from_prefs_button_clicked(self):
        prefs_path = _prefs_path()
        if os.path.exists(prefs_path):
            self.load_prefs_if_exists()
            self.main_window.tree_widget.blockSignals(True)
            try:
                iterator = QtWidgets.QTreeWidgetItemIterator(
                    self.main_window.tree_widget)
                while iterator.value():
                    item = iterator.value()
                    if item.flags() & QtCore.Qt.ItemIsUserCheckable and os.path.isfile(item.data(0, QtCore.Qt.UserRole)):
                        item.setCheckState(0, QtCore.Qt.Unchecked)
                    iterator += 1

                for rel_path in self.checked_files_from_prefs:
                    if rel_path in self.main_window.path_to_item:
                        item = self.main_window.path_to_item[rel_path]
                        if os.path.isfile(item.data(0, QtCore.Qt.UserRole)):
                            item.setCheckState(0, QtCore.Qt.Checked)
                            parent = item.parent()
                            while parent:
                                if parent.checkState(0) != QtCore.Qt.PartiallyChecked and parent.checkState(0) != QtCore.Qt.Checked:
                                    parent.setCheckState(
                                        0, QtCore.Qt.PartiallyChecked)
                                parent = parent.parent()
            finally:
                self.main_window.tree_widget.blockSignals(False)
            self.main_window._expand_folders_for_paths(
                self.checked_files_from_prefs)
            file_type = ".auicp" if prefs_path.endswith(
                ".auicp") else ".aicodeprep-gui"
            self.main_window.text_label.setText(
                f"Loaded selection from {file_type}")
            self.main_window.update_token_counter()
        else:
            self.main_window.text_label.setText(
                "No preferences file found (.aicodeprep-gui)")
