#!/usr/bin/env python3
"""Test script to verify the MCP server works correctly."""

import asyncio
import json
import os
import sys
from pathlib import Path

# Add the src directory to the path
sys.path.insert(0, str(Path(__file__).parent / "src"))

from parallel_llm_mcp.server import ParallelLLMServer


async def test_server():
    """Test the server functionality."""
    print("Testing Parallel LLM MCP Server...")
    
    # Check if API key is set
    api_key = os.getenv("OPENROUTER_API_KEY")
    if not api_key:
        print("❌ OPENROUTER_API_KEY environment variable not set")
        print("Please set it with:")
        print("$env:OPENROUTER_API_KEY='your-api-key-here'")
        return False
    
    try:
        # Initialize server
        print("✅ Initializing server...")
        server = ParallelLLMServer()
        print("✅ Server initialized successfully")
        
        print("✅ Available models:")
        for i, model in enumerate(server.parallel_models, 1):
            print(f"   {i}. {model}")
        
        print(f"✅ Synthesizer model: {server.synthesizer_model}")
        
        # Test that tools are registered (FastMCP manages this internally)
        print("✅ MCP tools registered")
        
        print("\n🚀 Server is ready to use!")
        print("\nTo run the MCP server:")
        print("python -m parallel_llm_mcp.server")
        print("\nOr use the entry point:")
        print("parallel-llm-mcp")
        
        return True
        
    except Exception as e:
        print(f"❌ Error testing server: {e}")
        return False


if __name__ == "__main__":
    success = asyncio.run(test_server())
    sys.exit(0 if success else 1)