# [your_file].py

# Copyright (C) [2025] Eduardo Antonio Ferrera Rodríguez
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation, either version 3 of the License, or any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY. See the COPYING file for more details.


# pyfrontkit/css.py
from pathlib import Path

class CSSRegistry:
    """
    Simplified CSS selector registry for FrontPy.
    - Registers tags, ids, and classes once.
    - No cascades are generated.
    """

    _tags = set()
    _ids = set()
    _classes = set()
    _css_path = Path("style.css")

    @classmethod
    def _ensure_file(cls):
        if not cls._css_path.exists():
            cls._css_path.write_text("/* CSS file generated by FrontPy */\n\n", encoding="utf-8")

    @classmethod
    def register_block(cls, block):
        """
        Registers a block:
        - tag
        - id if present
        - classes if present
        """
        # Register tag
        if block.tag not in cls._tags:
            cls._tags.add(block.tag)
            cls._write_selector(block.tag)

        # Register ID
        block_id = block.attrs.get("id")
        if block_id and block_id not in cls._ids:
            cls._ids.add(block_id)
            cls._write_selector(f"#{block_id}")

        # Register classes
        classes = block.attrs.get("class")
        if classes:
            for cls_name in str(classes).split():
                if cls_name not in cls._classes:
                    cls._classes.add(cls_name)
                    cls._write_selector(f".{cls_name}")

        # Recursively register children blocks
        for child in getattr(block, "children", []):
            if hasattr(child, "tag"):  # Only blocks with a tag
                cls.register_block(child)

        # Register content tags (ctn_)
        for ctn_item in getattr(block, "content_items", []):
            content_tag = getattr(ctn_item, "tag", None)
            if content_tag and content_tag not in cls._tags:
                cls._tags.add(content_tag)
                cls._write_selector(content_tag)

    @classmethod
    def _write_selector(cls, selector):
        cls._ensure_file()
        with cls._css_path.open("a", encoding="utf-8") as f:
            f.write(f"{selector} {{\n    /* styles here */\n}}\n\n")

    @classmethod
    def generate_css(cls):
        """
        Generates the complete CSS in memory.
        Returns string ready to write to style.css
        """
        lines = ["/* Selectors generated by FrontPy */\n"]
        for tag in sorted(cls._tags):
            lines.append(f"{tag} {{\n    /* styles here */\n}}\n")
        for id_name in sorted(cls._ids):
            lines.append(f"#{id_name} {{\n    /* styles here */\n}}\n")
        for cls_name in sorted(cls._classes):
            lines.append(f".{cls_name} {{\n    /* styles here */\n}}\n")
        return "\n".join(lines)
