
# Copyright (C) [2025] Eduardo Antonio Ferrera Rodríguez
# 
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY. See the COPYING file for more details.


# pyfrontkit/html_doc.py
# pyfrontkit/html_doc.py

# pyfrontkit/html_doc.py

from .dom import DOM
from .css import CSSRegistry
from .block import Block

class HtmlDoc:
    def __init__(self, title="Untitled Document", path=".", **head_attrs):
        self.title = title
        self.path = path.strip()
        if not self.path.endswith("/"):
            self.path += "/"

        self.html_file = self.path + "index.html"
        self.css_file = self.path + "style.css"

        # Optional head attributes
        self.head_attrs = head_attrs

        # Base HTML and CSS templates
        self._html_base = self._generate_base_html()
        self._css_base = self._generate_base_css()

    # ------------------------------
    # Head generation
    # ------------------------------
    def _generate_head(self):
        lines = ["<meta charset='UTF-8'>"]
        lines.append(f"<title>{self.title}</title>")
        lines.append("<link rel='stylesheet' href='style.css'>")

        metas = self.head_attrs.get("meta", [])
        for m in metas:
            attrs = " ".join(f'{k}="{v}"' for k, v in m.items())
            lines.append(f"<meta {attrs}>")

        for s in self.head_attrs.get("script", []):
            lines.append(f"<script>{s}</script>")

        for st in self.head_attrs.get("style", []):
            lines.append(f"<style>{st}</style>")

        return "\n    ".join(lines)

    def _generate_base_html(self):
        head_content = self._generate_head()
        return f"""<!DOCTYPE html>
<html lang="en">
<head>
    {head_content}
</head>
<body>
{{body_content}}
</body>
</html>
"""

    def _generate_base_css(self):
        return """/* Stylesheet generated by PyFrontKit */
body {
    font-family: Arial, sans-serif;
    margin: 0;
    padding: 0;
}
"""

    # ------------------------------
    # Body rendering
    # ------------------------------
    def render_body(self):
        """
        Imprime todos los bloques que no son hijos de nadie,
        respetando el orden de creación.
        """
        html = ""
        for block in Block._registry:
            if block._parent is None:  # solo bloques raíz
                html += block.render(indent=2)
        return html

    # ------------------------------
    # Document creation
    # ------------------------------
    def create_document(self):
        body_content = self.render_body()
        html_final = self._html_base.replace("{body_content}", body_content)
        css_final = self._css_base + "\n" + CSSRegistry.generate_css()

        try:
            with open(self.html_file, "w", encoding="utf-8") as f_html:
                f_html.write(html_final)
            with open(self.css_file, "w", encoding="utf-8") as f_css:
                f_css.write(css_final)
            print(f"✅ Documents created: {self.html_file} and {self.css_file}")
        except Exception as e:
            print(f"❌ Error creating files: {e}")
