# Repository: https://gitlab.com/qblox/packages/software/qblox-scheduler
# Licensed according to the LICENSE file on the main branch
"""Tests for the driver version check."""

import pytest
from packaging.version import InvalidVersion
from qblox_instruments import build

from qblox_scheduler.backends.qblox.driver_version_check import (
    SUPPORTED_DRIVER_VERSIONS,
    DriverVersionError,
    verify_qblox_instruments_version,
)


def test_verify_qblox_instruments_version(request):
    if request.config.getoption("--skip_qblox_driver_version_check"):
        pytest.xfail("Skipping test due to --skip_qblox_driver_version_check flag.")

    verify_qblox_instruments_version(build.__version__)

    version_numbers = build.__version__.split(".")
    version_numbers[-1] = "99999"
    version_ignore_patch = ".".join(map(str, version_numbers))
    verify_qblox_instruments_version(version_ignore_patch)

    nonsense_version = "nonsense.driver.version"
    with pytest.raises(InvalidVersion):
        verify_qblox_instruments_version(nonsense_version)
    outdated_version = "0.0.1"
    with pytest.raises(DriverVersionError) as wrong_version:
        verify_qblox_instruments_version(outdated_version)
    assert (
        wrong_version.value.args[0]
        == f"The installed Qblox driver (qblox-instruments) version {outdated_version} "
        "is not supported by backend. Please install one of the supported versions "
        f"({SUPPORTED_DRIVER_VERSIONS}) in order to use this backend."
    )

    with pytest.raises(DriverVersionError) as none_error:
        verify_qblox_instruments_version(None)

    assert (
        none_error.value.args[0]
        == "Version check for Qblox driver (qblox-instruments) could not be "
        "performed. Either the package is not installed correctly or a version "
        "<0.3.2 was found."
    )
