# -*- coding: utf-8 -*-
#
# Configuration file for the Sphinx documentation builder.
#
# This file does only contain a selection of the most common options. For a
# full list see the documentation:
# http://www.sphinx-doc.org/en/master/config

# -- Path setup --------------------------------------------------------------

# If extensions (or modules to document with autodoc) are in another directory,
# add these directories to sys.path here. If the directory is relative to the
# documentation root, use os.path.abspath to make it absolute, like shown here.
#
import os
import sys
from spinedb_api import DatabaseMapping

root_path = os.path.join(os.path.dirname(__file__), os.path.pardir, os.path.pardir)
sys.path.insert(0, os.path.abspath(root_path))


# -- Project information -----------------------------------------------------

project = "Spine Database API"
author = "Spine project consortium, Spine Database API contributors"
copyright = "2017-2022 {}".format(author)

# The short X.Y version
from spinedb_api import __version__ as spinedb_api_version

version = spinedb_api_version
# The full version, including alpha/beta/rc tags
release = spinedb_api_version


# -- General configuration ---------------------------------------------------

# If your documentation needs a minimal Sphinx version, state it here.
#
# needs_sphinx = '1.0'

# Add any Sphinx extension module names here, as strings. They can be
# extensions coming with Sphinx (named 'sphinx.ext.*') or your custom
# ones.
extensions = [
    "sphinx.ext.todo",
    "sphinx.ext.coverage",
    "sphinx.ext.ifconfig",
    "sphinx.ext.viewcode",
    "sphinx.ext.githubpages",
    "sphinx.ext.napoleon",
    "sphinx.ext.intersphinx",
    "recommonmark",
    "autoapi.extension",
]

# Add any paths that contain templates here, relative to this directory.
templates_path = ["_templates"]

# The suffix(es) of source filenames.
# You can specify multiple suffix as a list of string:
#
# source_suffix = ['.rst', '.md']
# source_suffix = ".rst"
# (note: since Sphinx 1.8 this is a dict rather than a list)
source_suffix = {".rst": "restructuredtext", ".md": "markdown"}  # support provided via the 'recommonmark' extension


# The master toctree document.
master_doc = "index"

# The language for content autogenerated by Sphinx. Refer to documentation
# for a list of supported languages.
#
# This is also used if you do content translation via gettext catalogs.
# Usually you set "language" from the command line for these cases.
language = "en"

# List of patterns, relative to source directory, that match files and
# directories to ignore when looking for source files.
# This pattern also affects html_static_path and html_extra_path.
exclude_patterns = []

# The name of the Pygments (syntax highlighting) style to use.
pygments_style = "sphinx"

# Settings for Sphinx AutoAPI
autoapi_options = ["members", "show-module-summary", "show-inheritance"]
autoapi_python_class_content = "both"
autoapi_add_toctree_entry = True
autoapi_root = "autoapi"
autoapi_dirs = ["../../spinedb_api"]  # package to be documented
autoapi_ignore = [
    "*/spinedb_api/alembic/*",
    "*/spinedb_api/arrow_value*",
    "*/spinedb_api/compatibility*",
    "*/spinedb_api/db_mapping_helpers*",
    "*/spinedb_api/exception*",
    "*/spinedb_api/export_functions*",
    "*/spinedb_api/graph_layout_generator*",
    "*/spinedb_api/helpers*",
    "*/spinedb_api/mapping*",
    "*/spinedb_api/perfect_split*",
    "*/spinedb_api/spine_db_client*",
    "*/spinedb_api/spine_db_server*",
    "*/spinedb_api/temp_id*",
    "*/spinedb_api/version*",
]  # ignored modules


def _spine_item_types():
    return ", ".join([f"``{x}``" for x in DatabaseMapping.item_types()])


def _process_docstring(app, what, name, obj, options, lines):
    if any(":meta private:" in line for line in lines):
        lines.clear()
    # Expand <spine_item_types>
    for k, line in enumerate(lines):
        if "<spine_item_types>" in line:
            lines[k] = line.replace("<spine_item_types>", _spine_item_types())


def _db_mapping_schema_lines():
    def type_(f_dict):
        return f_dict["type"].__name__ + (", optional" if f_dict.get("optional", False) else "")

    lines = [
        ".. _db_mapping_schema:",
        "",
        "DB mapping schema",
        "=================",
        "",
        "The DB mapping schema is a close cousin of the Spine DB schema with some extra flexibility, "
        "like the ability to specify references by name rather than by numerical id.",
        "",
        f"The schema defines the following item types: {_spine_item_types()}. "
        "As you can see, these follow the names of some of the tables in the Spine DB schema.",
        "",
        "The following subsections provide all the details you need to know about the different item types, namely, "
        "their fields, values, and unique keys.",
        "",
    ]
    for item_type in DatabaseMapping.item_types():
        factory = DatabaseMapping.item_factory(item_type)
        lines.extend([item_type, len(item_type) * "-", ""])
        lines.extend(
            [
                ".. list-table:: Fields and values",
                "   :header-rows: 1",
                "",
                "   * - field",
                "     - type",
                "     - value",
            ]
        )
        for f_name, f_dict in factory.fields.items():
            lines.extend([f"   * - {f_name}", f"     - {type_(f_dict)}", f"     - {f_dict['value']}"])
        lines.append("")
        lines.extend([".. list-table:: Unique keys", "   :header-rows: 0", ""])
        for f_names in factory.unique_keys:
            f_names = ", ".join(f_names)
            lines.append(f"   * - {f_names}")
        lines.append("")
    return lines


def setup(sphinx):
    sphinx.connect("autodoc-process-docstring", _process_docstring)
    with open(os.path.join(os.path.dirname(__file__), "db_mapping_schema.rst"), "w") as f:
        for line in _db_mapping_schema_lines():
            f.write(line + "\n")


# -- Options for HTML output -------------------------------------------------

# The theme to use for HTML and HTML Help pages.  See the documentation for
# a list of builtin themes.
#
html_theme = "sphinx_rtd_theme"

# Theme options are theme-specific and customize the look and feel of a theme
# further.  For a list of options available for each theme, see the
# documentation.
#
# html_theme_options = {}

# Add any paths that contain custom static files (such as style sheets) here,
# relative to this directory. They are copied after the builtin static files,
# so a file named "default.css" will overwrite the builtin "default.css".
html_static_path = []

# Custom sidebar templates, must be a dictionary that maps document names
# to template names.
#
# The default sidebars (for documents that don't match any pattern) are
# defined by theme itself.  Builtin themes are using these templates by
# default: ``['localtoc.html', 'relations.html', 'sourcelink.html',
# 'searchbox.html']``.
#
# html_sidebars = {}


# -- Options for HTMLHelp output ---------------------------------------------

# Output file base name for HTML help builder.
htmlhelp_basename = "SpineDatabaseAPIdoc"


# -- Options for LaTeX output ------------------------------------------------

latex_elements = {
    # The paper size ('letterpaper' or 'a4paper').
    #
    # 'papersize': 'letterpaper',
    # The font size ('10pt', '11pt' or '12pt').
    #
    # 'pointsize': '10pt',
    # Additional stuff for the LaTeX preamble.
    #
    # 'preamble': '',
    # Latex figure (float) alignment
    #
    # 'figure_align': 'htbp',
}

# Grouping the document tree into LaTeX files. List of tuples
# (source start file, target name, title,
#  author, documentclass [howto, manual, or own class]).
latex_documents = [
    (
        master_doc,
        "SpineDatabaseAPI.tex",
        "Spine Database API Documentation",
        "Spine project consortium, Spine Database API contributors",
        "manual",
    )
]


# -- Options for manual page output ------------------------------------------

# One entry per manual page. List of tuples
# (source start file, name, description, authors, manual section).
man_pages = [(master_doc, "spinedatabaseapi", "Spine Database API Documentation", [author], 1)]


# -- Options for Texinfo output ----------------------------------------------

# Grouping the document tree into Texinfo files. List of tuples
# (source start file, target name, title, author,
#  dir menu entry, description, category)
texinfo_documents = [
    (
        master_doc,
        "SpineDatabaseAPI",
        "Spine Database API Documentation",
        author,
        "SpineDatabaseAPI",
        "Database interface to Spine generic data model.",
        "",
    )
]


# -- Options for Epub output -------------------------------------------------

# Bibliographic Dublin Core info.
epub_title = project

# The unique identifier of the text. This can be a ISBN number
# or the project homepage.
#
# epub_identifier = ''

# A unique identification for the text.
#
# epub_uid = ''

# A list of files that should not be packed into the epub file.
epub_exclude_files = ["search.html"]


# -- Extension configuration -------------------------------------------------

# -- Options for intersphinx extension ---------------------------------------

# Example configuration for intersphinx: refer to the Python standard library.
intersphinx_mapping = {
    "python": ("https://docs.python.org/3/", None),
    "sqlalchemy": ("https://docs.sqlalchemy.org/en/14/", None),
}

# -- Options for todo extension ----------------------------------------------

# If true, `todo` and `todoList` produce output, else they produce nothing.
todo_include_todos = True

autodoc_member_order = "bysource"
