# This file was auto-generated by Fern from our API Definition.

import typing
from json.decoder import JSONDecodeError

from ..core.api_error import ApiError
from ..core.client_wrapper import AsyncClientWrapper, SyncClientWrapper
from ..core.http_response import AsyncHttpResponse, HttpResponse
from ..core.jsonable_encoder import jsonable_encoder
from ..core.pydantic_utilities import parse_obj_as
from ..core.request_options import RequestOptions
from ..core.serialization import convert_and_respect_annotation_metadata
from ..errors.unprocessable_entity_error import UnprocessableEntityError
from ..types.config_values import ConfigValues
from ..types.http_validation_error import HttpValidationError
from ..types.source_connection import SourceConnection
from ..types.source_connection_job import SourceConnectionJob
from ..types.source_connection_list_item import SourceConnectionListItem
from .types.source_connection_update_auth_fields import SourceConnectionUpdateAuthFields

# this is used as the default value for optional parameters
OMIT = typing.cast(typing.Any, ...)


class RawSourceConnectionsClient:
    def __init__(self, *, client_wrapper: SyncClientWrapper):
        self._client_wrapper = client_wrapper

    def list_source_connections(
        self,
        *,
        collection: typing.Optional[str] = None,
        skip: typing.Optional[int] = None,
        limit: typing.Optional[int] = None,
        request_options: typing.Optional[RequestOptions] = None,
    ) -> HttpResponse[typing.List[SourceConnectionListItem]]:
        """
        List source connections across your organization.

        <br/><br/>
        By default, returns ALL source connections from every collection in your
        organization. Use the 'collection' parameter to filter results to a specific
        collection. This is useful for getting an overview of all your data sources
        or managing connections within a particular collection.

        Parameters
        ----------
        collection : typing.Optional[str]
            Filter source connections by collection readable ID

        skip : typing.Optional[int]
            Number of source connections to skip for pagination

        limit : typing.Optional[int]
            Maximum number of source connections to return (1-1000)

        request_options : typing.Optional[RequestOptions]
            Request-specific configuration.

        Returns
        -------
        HttpResponse[typing.List[SourceConnectionListItem]]
            Successful Response
        """
        _response = self._client_wrapper.httpx_client.request(
            "source-connections",
            method="GET",
            params={
                "collection": collection,
                "skip": skip,
                "limit": limit,
            },
            request_options=request_options,
        )
        try:
            if 200 <= _response.status_code < 300:
                _data = typing.cast(
                    typing.List[SourceConnectionListItem],
                    parse_obj_as(
                        type_=typing.List[SourceConnectionListItem],  # type: ignore
                        object_=_response.json(),
                    ),
                )
                return HttpResponse(response=_response, data=_data)
            if _response.status_code == 422:
                raise UnprocessableEntityError(
                    headers=dict(_response.headers),
                    body=typing.cast(
                        HttpValidationError,
                        parse_obj_as(
                            type_=HttpValidationError,  # type: ignore
                            object_=_response.json(),
                        ),
                    ),
                )
            _response_json = _response.json()
        except JSONDecodeError:
            raise ApiError(status_code=_response.status_code, headers=dict(_response.headers), body=_response.text)
        raise ApiError(status_code=_response.status_code, headers=dict(_response.headers), body=_response_json)

    def create_source_connection(
        self,
        *,
        name: str,
        short_name: str,
        description: typing.Optional[str] = OMIT,
        config_fields: typing.Optional[ConfigValues] = OMIT,
        collection: typing.Optional[str] = OMIT,
        cron_schedule: typing.Optional[str] = OMIT,
        auth_fields: typing.Optional[ConfigValues] = OMIT,
        auth_provider: typing.Optional[str] = OMIT,
        auth_provider_config: typing.Optional[ConfigValues] = OMIT,
        sync_immediately: typing.Optional[bool] = OMIT,
        request_options: typing.Optional[RequestOptions] = None,
    ) -> HttpResponse[SourceConnection]:
        """
        Create a new source connection to sync data into your collection.

        <br/><br/>

        **This endpoint only works for sources that do not use OAuth2.0.**
        Sources that do use OAuth2.0 like Google Drive, Slack, or HubSpot must be
        connected through the UI where you can complete the OAuth consent flow.<br/><br/>

        Credentials for a source have to be provided using the `auth_fields` field.
        Currently, it is not automatically checked if the provided credentials are valid.
        If they are not valid, the data synchronization will fail.<br/><br/>

        Check the documentation of a specific source (for example
        [Github](https://docs.airweave.ai/docs/connectors/github)) to see what kind
        of authentication is used.

        Parameters
        ----------
        name : str
            Human-readable name for the source connection. This helps you identify the connection in the UI and should clearly describe what data it connects to.

        short_name : str
            Technical identifier of the source type that determines which connector to use for data synchronization.

        description : typing.Optional[str]
            Optional detailed description of what this source connection provides. Use this to document the purpose, data types, or any special considerations for this connection.

        config_fields : typing.Optional[ConfigValues]
            Source-specific configuration parameters required for data extraction. These vary by source type and control how data is retrieved (e.g., database queries, API filters, file paths). Check the documentation of a specific source (for example [Github](https://docs.airweave.ai/docs/connectors/github)) to see what is required.

        collection : typing.Optional[str]
            Readable ID of the collection where synced data will be stored. If not provided, a new collection will be automatically created.

        cron_schedule : typing.Optional[str]
            Cron expression for automatic data synchronization schedule. If not provided, data will only sync when manually triggered. Use standard cron format: minute hour day month weekday.

        auth_fields : typing.Optional[ConfigValues]
            Authentication credentials required to access the data source. The required fields vary by source type. Check the documentation of a specific source (for example [Github](https://docs.airweave.ai/docs/connectors/github)) to see what is required.

        auth_provider : typing.Optional[str]
            Unique readable ID of a connected auth provider to use for authentication instead of providing auth_fields directly. When specified, credentials for the source will be obtained and refreshed automatically by Airweave interaction with the auth provider. To see which auth providers are supported and learn more about how to use them, check [this page](https://docs.airweave.ai/docs/auth-providers).

        auth_provider_config : typing.Optional[ConfigValues]
            Configuration for the auth provider when using auth_provider field. Required fields vary by auth provider. For Composio, use auth_config_id and  account_id to specify which integration and account from Composio you want to use to connect to the source.

        sync_immediately : typing.Optional[bool]
            Whether to start an initial data synchronization immediately after creating the connection.

        request_options : typing.Optional[RequestOptions]
            Request-specific configuration.

        Returns
        -------
        HttpResponse[SourceConnection]
            Successful Response
        """
        _response = self._client_wrapper.httpx_client.request(
            "source-connections",
            method="POST",
            json={
                "name": name,
                "description": description,
                "config_fields": config_fields,
                "short_name": short_name,
                "collection": collection,
                "cron_schedule": cron_schedule,
                "auth_fields": auth_fields,
                "auth_provider": auth_provider,
                "auth_provider_config": auth_provider_config,
                "sync_immediately": sync_immediately,
            },
            headers={
                "content-type": "application/json",
            },
            request_options=request_options,
            omit=OMIT,
        )
        try:
            if 200 <= _response.status_code < 300:
                _data = typing.cast(
                    SourceConnection,
                    parse_obj_as(
                        type_=SourceConnection,  # type: ignore
                        object_=_response.json(),
                    ),
                )
                return HttpResponse(response=_response, data=_data)
            if _response.status_code == 422:
                raise UnprocessableEntityError(
                    headers=dict(_response.headers),
                    body=typing.cast(
                        HttpValidationError,
                        parse_obj_as(
                            type_=HttpValidationError,  # type: ignore
                            object_=_response.json(),
                        ),
                    ),
                )
            _response_json = _response.json()
        except JSONDecodeError:
            raise ApiError(status_code=_response.status_code, headers=dict(_response.headers), body=_response.text)
        raise ApiError(status_code=_response.status_code, headers=dict(_response.headers), body=_response_json)

    def get_source_connection(
        self,
        source_connection_id: str,
        *,
        show_auth_fields: typing.Optional[bool] = None,
        request_options: typing.Optional[RequestOptions] = None,
    ) -> HttpResponse[SourceConnection]:
        """
        Retrieve a specific source connection by its ID.

        Parameters
        ----------
        source_connection_id : str
            The unique identifier of the source connection

        show_auth_fields : typing.Optional[bool]
            Whether to reveal authentication credentials.

        request_options : typing.Optional[RequestOptions]
            Request-specific configuration.

        Returns
        -------
        HttpResponse[SourceConnection]
            Successful Response
        """
        _response = self._client_wrapper.httpx_client.request(
            f"source-connections/{jsonable_encoder(source_connection_id)}",
            method="GET",
            params={
                "show_auth_fields": show_auth_fields,
            },
            request_options=request_options,
        )
        try:
            if 200 <= _response.status_code < 300:
                _data = typing.cast(
                    SourceConnection,
                    parse_obj_as(
                        type_=SourceConnection,  # type: ignore
                        object_=_response.json(),
                    ),
                )
                return HttpResponse(response=_response, data=_data)
            if _response.status_code == 422:
                raise UnprocessableEntityError(
                    headers=dict(_response.headers),
                    body=typing.cast(
                        HttpValidationError,
                        parse_obj_as(
                            type_=HttpValidationError,  # type: ignore
                            object_=_response.json(),
                        ),
                    ),
                )
            _response_json = _response.json()
        except JSONDecodeError:
            raise ApiError(status_code=_response.status_code, headers=dict(_response.headers), body=_response.text)
        raise ApiError(status_code=_response.status_code, headers=dict(_response.headers), body=_response_json)

    def update_source_connection(
        self,
        source_connection_id: str,
        *,
        name: typing.Optional[str] = OMIT,
        description: typing.Optional[str] = OMIT,
        auth_fields: typing.Optional[SourceConnectionUpdateAuthFields] = OMIT,
        config_fields: typing.Optional[ConfigValues] = OMIT,
        cron_schedule: typing.Optional[str] = OMIT,
        connection_id: typing.Optional[str] = OMIT,
        white_label_id: typing.Optional[str] = OMIT,
        auth_provider: typing.Optional[str] = OMIT,
        auth_provider_config: typing.Optional[typing.Dict[str, typing.Optional[typing.Any]]] = OMIT,
        request_options: typing.Optional[RequestOptions] = None,
    ) -> HttpResponse[SourceConnection]:
        """
        Update a source connection's properties.

        <br/><br/>

        Modify the configuration of an existing source connection including its name,
        authentication credentials, configuration fields, sync schedule, or source-specific settings.

        Parameters
        ----------
        source_connection_id : str
            The unique identifier of the source connection to update

        name : typing.Optional[str]
            Updated name for the source connection. Must be between 4 and 42 characters.

        description : typing.Optional[str]
            Updated description of what this source connection provides.

        auth_fields : typing.Optional[SourceConnectionUpdateAuthFields]
            Updated authentication credentials for the data source. Provide new credentials to refresh or update authentication.

        config_fields : typing.Optional[ConfigValues]
            Source-specific configuration parameters required for data extraction. These vary by source type and control how data is retrieved (e.g., database queries, API filters, file paths). Check the documentation of a specific source (for example [Github](https://docs.airweave.ai/docs/connectors/github)) to see what is required.

        cron_schedule : typing.Optional[str]
            Updated cron expression for automatic synchronization schedule. Set to null to disable automatic syncing.

        connection_id : typing.Optional[str]
            Internal connection identifier. This is typically managed automatically and should not be modified manually.

        white_label_id : typing.Optional[str]
            ID of the white label integration. Used for custom OAuth integrations with your own branding.

        auth_provider : typing.Optional[str]
            Updated auth provider readable ID. Only relevant if the connection uses an auth provider.

        auth_provider_config : typing.Optional[typing.Dict[str, typing.Optional[typing.Any]]]
            Updated configuration for the auth provider. Only relevant if the connection uses an auth provider.

        request_options : typing.Optional[RequestOptions]
            Request-specific configuration.

        Returns
        -------
        HttpResponse[SourceConnection]
            Successful Response
        """
        _response = self._client_wrapper.httpx_client.request(
            f"source-connections/{jsonable_encoder(source_connection_id)}",
            method="PUT",
            json={
                "name": name,
                "description": description,
                "auth_fields": convert_and_respect_annotation_metadata(
                    object_=auth_fields, annotation=SourceConnectionUpdateAuthFields, direction="write"
                ),
                "config_fields": config_fields,
                "cron_schedule": cron_schedule,
                "connection_id": connection_id,
                "white_label_id": white_label_id,
                "auth_provider": auth_provider,
                "auth_provider_config": auth_provider_config,
            },
            headers={
                "content-type": "application/json",
            },
            request_options=request_options,
            omit=OMIT,
        )
        try:
            if 200 <= _response.status_code < 300:
                _data = typing.cast(
                    SourceConnection,
                    parse_obj_as(
                        type_=SourceConnection,  # type: ignore
                        object_=_response.json(),
                    ),
                )
                return HttpResponse(response=_response, data=_data)
            if _response.status_code == 422:
                raise UnprocessableEntityError(
                    headers=dict(_response.headers),
                    body=typing.cast(
                        HttpValidationError,
                        parse_obj_as(
                            type_=HttpValidationError,  # type: ignore
                            object_=_response.json(),
                        ),
                    ),
                )
            _response_json = _response.json()
        except JSONDecodeError:
            raise ApiError(status_code=_response.status_code, headers=dict(_response.headers), body=_response.text)
        raise ApiError(status_code=_response.status_code, headers=dict(_response.headers), body=_response_json)

    def delete_source_connection(
        self, source_connection_id: str, *, request_options: typing.Optional[RequestOptions] = None
    ) -> HttpResponse[SourceConnection]:
        """
        Delete a source connection and all associated data.

        <br/><br/>

        Permanently removes the source connection configuration, credentials, and all synced data
        from the destination systems. This action cannot be undone.

        Parameters
        ----------
        source_connection_id : str
            The unique identifier of the source connection to delete

        request_options : typing.Optional[RequestOptions]
            Request-specific configuration.

        Returns
        -------
        HttpResponse[SourceConnection]
            Successful Response
        """
        _response = self._client_wrapper.httpx_client.request(
            f"source-connections/{jsonable_encoder(source_connection_id)}",
            method="DELETE",
            request_options=request_options,
        )
        try:
            if 200 <= _response.status_code < 300:
                _data = typing.cast(
                    SourceConnection,
                    parse_obj_as(
                        type_=SourceConnection,  # type: ignore
                        object_=_response.json(),
                    ),
                )
                return HttpResponse(response=_response, data=_data)
            if _response.status_code == 422:
                raise UnprocessableEntityError(
                    headers=dict(_response.headers),
                    body=typing.cast(
                        HttpValidationError,
                        parse_obj_as(
                            type_=HttpValidationError,  # type: ignore
                            object_=_response.json(),
                        ),
                    ),
                )
            _response_json = _response.json()
        except JSONDecodeError:
            raise ApiError(status_code=_response.status_code, headers=dict(_response.headers), body=_response.text)
        raise ApiError(status_code=_response.status_code, headers=dict(_response.headers), body=_response_json)

    def run_source_connection(
        self,
        source_connection_id: str,
        *,
        access_token: typing.Optional[str] = OMIT,
        request_options: typing.Optional[RequestOptions] = None,
    ) -> HttpResponse[SourceConnectionJob]:
        """
        Manually trigger a data sync for this source connection.

        <br/><br/>
        Starts an immediate synchronization job that extracts fresh data from your source,
        transforms it according to your configuration, and updates the destination systems.
        The job runs asynchronously and endpoint returns immediately with tracking information.

        Parameters
        ----------
        source_connection_id : str
            The unique identifier of the source connection to sync

        access_token : typing.Optional[str]
            This parameter gives you the ability to start a sync job with an access token for an OAuth2.0 source directly instead of using the credentials that Airweave has stored for you. Learn more about direct token injection [here](https://docs.airweave.ai/direct-token-injection).

        request_options : typing.Optional[RequestOptions]
            Request-specific configuration.

        Returns
        -------
        HttpResponse[SourceConnectionJob]
            Successful Response
        """
        _response = self._client_wrapper.httpx_client.request(
            f"source-connections/{jsonable_encoder(source_connection_id)}/run",
            method="POST",
            json={
                "access_token": access_token,
            },
            headers={
                "content-type": "application/json",
            },
            request_options=request_options,
            omit=OMIT,
        )
        try:
            if 200 <= _response.status_code < 300:
                _data = typing.cast(
                    SourceConnectionJob,
                    parse_obj_as(
                        type_=SourceConnectionJob,  # type: ignore
                        object_=_response.json(),
                    ),
                )
                return HttpResponse(response=_response, data=_data)
            if _response.status_code == 422:
                raise UnprocessableEntityError(
                    headers=dict(_response.headers),
                    body=typing.cast(
                        HttpValidationError,
                        parse_obj_as(
                            type_=HttpValidationError,  # type: ignore
                            object_=_response.json(),
                        ),
                    ),
                )
            _response_json = _response.json()
        except JSONDecodeError:
            raise ApiError(status_code=_response.status_code, headers=dict(_response.headers), body=_response.text)
        raise ApiError(status_code=_response.status_code, headers=dict(_response.headers), body=_response_json)

    def list_source_connection_jobs(
        self, source_connection_id: str, *, request_options: typing.Optional[RequestOptions] = None
    ) -> HttpResponse[typing.List[SourceConnectionJob]]:
        """
        List all sync jobs for a source connection.

        <br/><br/>
        Returns the complete history of data synchronization jobs including successful syncs,
        failed attempts, and currently running operations.

        Parameters
        ----------
        source_connection_id : str
            The unique identifier of the source connection

        request_options : typing.Optional[RequestOptions]
            Request-specific configuration.

        Returns
        -------
        HttpResponse[typing.List[SourceConnectionJob]]
            Successful Response
        """
        _response = self._client_wrapper.httpx_client.request(
            f"source-connections/{jsonable_encoder(source_connection_id)}/jobs",
            method="GET",
            request_options=request_options,
        )
        try:
            if 200 <= _response.status_code < 300:
                _data = typing.cast(
                    typing.List[SourceConnectionJob],
                    parse_obj_as(
                        type_=typing.List[SourceConnectionJob],  # type: ignore
                        object_=_response.json(),
                    ),
                )
                return HttpResponse(response=_response, data=_data)
            if _response.status_code == 422:
                raise UnprocessableEntityError(
                    headers=dict(_response.headers),
                    body=typing.cast(
                        HttpValidationError,
                        parse_obj_as(
                            type_=HttpValidationError,  # type: ignore
                            object_=_response.json(),
                        ),
                    ),
                )
            _response_json = _response.json()
        except JSONDecodeError:
            raise ApiError(status_code=_response.status_code, headers=dict(_response.headers), body=_response.text)
        raise ApiError(status_code=_response.status_code, headers=dict(_response.headers), body=_response_json)

    def get_source_connection_job(
        self, source_connection_id: str, job_id: str, *, request_options: typing.Optional[RequestOptions] = None
    ) -> HttpResponse[SourceConnectionJob]:
        """
        Get detailed information about a specific sync job.

        Parameters
        ----------
        source_connection_id : str
            The unique identifier of the source connection

        job_id : str
            The unique identifier of the sync job

        request_options : typing.Optional[RequestOptions]
            Request-specific configuration.

        Returns
        -------
        HttpResponse[SourceConnectionJob]
            Successful Response
        """
        _response = self._client_wrapper.httpx_client.request(
            f"source-connections/{jsonable_encoder(source_connection_id)}/jobs/{jsonable_encoder(job_id)}",
            method="GET",
            request_options=request_options,
        )
        try:
            if 200 <= _response.status_code < 300:
                _data = typing.cast(
                    SourceConnectionJob,
                    parse_obj_as(
                        type_=SourceConnectionJob,  # type: ignore
                        object_=_response.json(),
                    ),
                )
                return HttpResponse(response=_response, data=_data)
            if _response.status_code == 422:
                raise UnprocessableEntityError(
                    headers=dict(_response.headers),
                    body=typing.cast(
                        HttpValidationError,
                        parse_obj_as(
                            type_=HttpValidationError,  # type: ignore
                            object_=_response.json(),
                        ),
                    ),
                )
            _response_json = _response.json()
        except JSONDecodeError:
            raise ApiError(status_code=_response.status_code, headers=dict(_response.headers), body=_response.text)
        raise ApiError(status_code=_response.status_code, headers=dict(_response.headers), body=_response_json)

    def cancel_source_connection_job(
        self, source_connection_id: str, job_id: str, *, request_options: typing.Optional[RequestOptions] = None
    ) -> HttpResponse[SourceConnectionJob]:
        """
        Cancel a running sync job.

        <br/><br/>
        Sends a cancellation signal to stop an in-progress data synchronization.
        The job will complete its current operation and then terminate gracefully.
        Only jobs in 'created', 'pending', or 'in_progress' states can be cancelled.

        Parameters
        ----------
        source_connection_id : str
            The unique identifier of the source connection

        job_id : str
            The unique identifier of the sync job to cancel

        request_options : typing.Optional[RequestOptions]
            Request-specific configuration.

        Returns
        -------
        HttpResponse[SourceConnectionJob]
            Successful Response
        """
        _response = self._client_wrapper.httpx_client.request(
            f"source-connections/{jsonable_encoder(source_connection_id)}/jobs/{jsonable_encoder(job_id)}/cancel",
            method="POST",
            request_options=request_options,
        )
        try:
            if 200 <= _response.status_code < 300:
                _data = typing.cast(
                    SourceConnectionJob,
                    parse_obj_as(
                        type_=SourceConnectionJob,  # type: ignore
                        object_=_response.json(),
                    ),
                )
                return HttpResponse(response=_response, data=_data)
            if _response.status_code == 422:
                raise UnprocessableEntityError(
                    headers=dict(_response.headers),
                    body=typing.cast(
                        HttpValidationError,
                        parse_obj_as(
                            type_=HttpValidationError,  # type: ignore
                            object_=_response.json(),
                        ),
                    ),
                )
            _response_json = _response.json()
        except JSONDecodeError:
            raise ApiError(status_code=_response.status_code, headers=dict(_response.headers), body=_response.text)
        raise ApiError(status_code=_response.status_code, headers=dict(_response.headers), body=_response_json)


class AsyncRawSourceConnectionsClient:
    def __init__(self, *, client_wrapper: AsyncClientWrapper):
        self._client_wrapper = client_wrapper

    async def list_source_connections(
        self,
        *,
        collection: typing.Optional[str] = None,
        skip: typing.Optional[int] = None,
        limit: typing.Optional[int] = None,
        request_options: typing.Optional[RequestOptions] = None,
    ) -> AsyncHttpResponse[typing.List[SourceConnectionListItem]]:
        """
        List source connections across your organization.

        <br/><br/>
        By default, returns ALL source connections from every collection in your
        organization. Use the 'collection' parameter to filter results to a specific
        collection. This is useful for getting an overview of all your data sources
        or managing connections within a particular collection.

        Parameters
        ----------
        collection : typing.Optional[str]
            Filter source connections by collection readable ID

        skip : typing.Optional[int]
            Number of source connections to skip for pagination

        limit : typing.Optional[int]
            Maximum number of source connections to return (1-1000)

        request_options : typing.Optional[RequestOptions]
            Request-specific configuration.

        Returns
        -------
        AsyncHttpResponse[typing.List[SourceConnectionListItem]]
            Successful Response
        """
        _response = await self._client_wrapper.httpx_client.request(
            "source-connections",
            method="GET",
            params={
                "collection": collection,
                "skip": skip,
                "limit": limit,
            },
            request_options=request_options,
        )
        try:
            if 200 <= _response.status_code < 300:
                _data = typing.cast(
                    typing.List[SourceConnectionListItem],
                    parse_obj_as(
                        type_=typing.List[SourceConnectionListItem],  # type: ignore
                        object_=_response.json(),
                    ),
                )
                return AsyncHttpResponse(response=_response, data=_data)
            if _response.status_code == 422:
                raise UnprocessableEntityError(
                    headers=dict(_response.headers),
                    body=typing.cast(
                        HttpValidationError,
                        parse_obj_as(
                            type_=HttpValidationError,  # type: ignore
                            object_=_response.json(),
                        ),
                    ),
                )
            _response_json = _response.json()
        except JSONDecodeError:
            raise ApiError(status_code=_response.status_code, headers=dict(_response.headers), body=_response.text)
        raise ApiError(status_code=_response.status_code, headers=dict(_response.headers), body=_response_json)

    async def create_source_connection(
        self,
        *,
        name: str,
        short_name: str,
        description: typing.Optional[str] = OMIT,
        config_fields: typing.Optional[ConfigValues] = OMIT,
        collection: typing.Optional[str] = OMIT,
        cron_schedule: typing.Optional[str] = OMIT,
        auth_fields: typing.Optional[ConfigValues] = OMIT,
        auth_provider: typing.Optional[str] = OMIT,
        auth_provider_config: typing.Optional[ConfigValues] = OMIT,
        sync_immediately: typing.Optional[bool] = OMIT,
        request_options: typing.Optional[RequestOptions] = None,
    ) -> AsyncHttpResponse[SourceConnection]:
        """
        Create a new source connection to sync data into your collection.

        <br/><br/>

        **This endpoint only works for sources that do not use OAuth2.0.**
        Sources that do use OAuth2.0 like Google Drive, Slack, or HubSpot must be
        connected through the UI where you can complete the OAuth consent flow.<br/><br/>

        Credentials for a source have to be provided using the `auth_fields` field.
        Currently, it is not automatically checked if the provided credentials are valid.
        If they are not valid, the data synchronization will fail.<br/><br/>

        Check the documentation of a specific source (for example
        [Github](https://docs.airweave.ai/docs/connectors/github)) to see what kind
        of authentication is used.

        Parameters
        ----------
        name : str
            Human-readable name for the source connection. This helps you identify the connection in the UI and should clearly describe what data it connects to.

        short_name : str
            Technical identifier of the source type that determines which connector to use for data synchronization.

        description : typing.Optional[str]
            Optional detailed description of what this source connection provides. Use this to document the purpose, data types, or any special considerations for this connection.

        config_fields : typing.Optional[ConfigValues]
            Source-specific configuration parameters required for data extraction. These vary by source type and control how data is retrieved (e.g., database queries, API filters, file paths). Check the documentation of a specific source (for example [Github](https://docs.airweave.ai/docs/connectors/github)) to see what is required.

        collection : typing.Optional[str]
            Readable ID of the collection where synced data will be stored. If not provided, a new collection will be automatically created.

        cron_schedule : typing.Optional[str]
            Cron expression for automatic data synchronization schedule. If not provided, data will only sync when manually triggered. Use standard cron format: minute hour day month weekday.

        auth_fields : typing.Optional[ConfigValues]
            Authentication credentials required to access the data source. The required fields vary by source type. Check the documentation of a specific source (for example [Github](https://docs.airweave.ai/docs/connectors/github)) to see what is required.

        auth_provider : typing.Optional[str]
            Unique readable ID of a connected auth provider to use for authentication instead of providing auth_fields directly. When specified, credentials for the source will be obtained and refreshed automatically by Airweave interaction with the auth provider. To see which auth providers are supported and learn more about how to use them, check [this page](https://docs.airweave.ai/docs/auth-providers).

        auth_provider_config : typing.Optional[ConfigValues]
            Configuration for the auth provider when using auth_provider field. Required fields vary by auth provider. For Composio, use auth_config_id and  account_id to specify which integration and account from Composio you want to use to connect to the source.

        sync_immediately : typing.Optional[bool]
            Whether to start an initial data synchronization immediately after creating the connection.

        request_options : typing.Optional[RequestOptions]
            Request-specific configuration.

        Returns
        -------
        AsyncHttpResponse[SourceConnection]
            Successful Response
        """
        _response = await self._client_wrapper.httpx_client.request(
            "source-connections",
            method="POST",
            json={
                "name": name,
                "description": description,
                "config_fields": config_fields,
                "short_name": short_name,
                "collection": collection,
                "cron_schedule": cron_schedule,
                "auth_fields": auth_fields,
                "auth_provider": auth_provider,
                "auth_provider_config": auth_provider_config,
                "sync_immediately": sync_immediately,
            },
            headers={
                "content-type": "application/json",
            },
            request_options=request_options,
            omit=OMIT,
        )
        try:
            if 200 <= _response.status_code < 300:
                _data = typing.cast(
                    SourceConnection,
                    parse_obj_as(
                        type_=SourceConnection,  # type: ignore
                        object_=_response.json(),
                    ),
                )
                return AsyncHttpResponse(response=_response, data=_data)
            if _response.status_code == 422:
                raise UnprocessableEntityError(
                    headers=dict(_response.headers),
                    body=typing.cast(
                        HttpValidationError,
                        parse_obj_as(
                            type_=HttpValidationError,  # type: ignore
                            object_=_response.json(),
                        ),
                    ),
                )
            _response_json = _response.json()
        except JSONDecodeError:
            raise ApiError(status_code=_response.status_code, headers=dict(_response.headers), body=_response.text)
        raise ApiError(status_code=_response.status_code, headers=dict(_response.headers), body=_response_json)

    async def get_source_connection(
        self,
        source_connection_id: str,
        *,
        show_auth_fields: typing.Optional[bool] = None,
        request_options: typing.Optional[RequestOptions] = None,
    ) -> AsyncHttpResponse[SourceConnection]:
        """
        Retrieve a specific source connection by its ID.

        Parameters
        ----------
        source_connection_id : str
            The unique identifier of the source connection

        show_auth_fields : typing.Optional[bool]
            Whether to reveal authentication credentials.

        request_options : typing.Optional[RequestOptions]
            Request-specific configuration.

        Returns
        -------
        AsyncHttpResponse[SourceConnection]
            Successful Response
        """
        _response = await self._client_wrapper.httpx_client.request(
            f"source-connections/{jsonable_encoder(source_connection_id)}",
            method="GET",
            params={
                "show_auth_fields": show_auth_fields,
            },
            request_options=request_options,
        )
        try:
            if 200 <= _response.status_code < 300:
                _data = typing.cast(
                    SourceConnection,
                    parse_obj_as(
                        type_=SourceConnection,  # type: ignore
                        object_=_response.json(),
                    ),
                )
                return AsyncHttpResponse(response=_response, data=_data)
            if _response.status_code == 422:
                raise UnprocessableEntityError(
                    headers=dict(_response.headers),
                    body=typing.cast(
                        HttpValidationError,
                        parse_obj_as(
                            type_=HttpValidationError,  # type: ignore
                            object_=_response.json(),
                        ),
                    ),
                )
            _response_json = _response.json()
        except JSONDecodeError:
            raise ApiError(status_code=_response.status_code, headers=dict(_response.headers), body=_response.text)
        raise ApiError(status_code=_response.status_code, headers=dict(_response.headers), body=_response_json)

    async def update_source_connection(
        self,
        source_connection_id: str,
        *,
        name: typing.Optional[str] = OMIT,
        description: typing.Optional[str] = OMIT,
        auth_fields: typing.Optional[SourceConnectionUpdateAuthFields] = OMIT,
        config_fields: typing.Optional[ConfigValues] = OMIT,
        cron_schedule: typing.Optional[str] = OMIT,
        connection_id: typing.Optional[str] = OMIT,
        white_label_id: typing.Optional[str] = OMIT,
        auth_provider: typing.Optional[str] = OMIT,
        auth_provider_config: typing.Optional[typing.Dict[str, typing.Optional[typing.Any]]] = OMIT,
        request_options: typing.Optional[RequestOptions] = None,
    ) -> AsyncHttpResponse[SourceConnection]:
        """
        Update a source connection's properties.

        <br/><br/>

        Modify the configuration of an existing source connection including its name,
        authentication credentials, configuration fields, sync schedule, or source-specific settings.

        Parameters
        ----------
        source_connection_id : str
            The unique identifier of the source connection to update

        name : typing.Optional[str]
            Updated name for the source connection. Must be between 4 and 42 characters.

        description : typing.Optional[str]
            Updated description of what this source connection provides.

        auth_fields : typing.Optional[SourceConnectionUpdateAuthFields]
            Updated authentication credentials for the data source. Provide new credentials to refresh or update authentication.

        config_fields : typing.Optional[ConfigValues]
            Source-specific configuration parameters required for data extraction. These vary by source type and control how data is retrieved (e.g., database queries, API filters, file paths). Check the documentation of a specific source (for example [Github](https://docs.airweave.ai/docs/connectors/github)) to see what is required.

        cron_schedule : typing.Optional[str]
            Updated cron expression for automatic synchronization schedule. Set to null to disable automatic syncing.

        connection_id : typing.Optional[str]
            Internal connection identifier. This is typically managed automatically and should not be modified manually.

        white_label_id : typing.Optional[str]
            ID of the white label integration. Used for custom OAuth integrations with your own branding.

        auth_provider : typing.Optional[str]
            Updated auth provider readable ID. Only relevant if the connection uses an auth provider.

        auth_provider_config : typing.Optional[typing.Dict[str, typing.Optional[typing.Any]]]
            Updated configuration for the auth provider. Only relevant if the connection uses an auth provider.

        request_options : typing.Optional[RequestOptions]
            Request-specific configuration.

        Returns
        -------
        AsyncHttpResponse[SourceConnection]
            Successful Response
        """
        _response = await self._client_wrapper.httpx_client.request(
            f"source-connections/{jsonable_encoder(source_connection_id)}",
            method="PUT",
            json={
                "name": name,
                "description": description,
                "auth_fields": convert_and_respect_annotation_metadata(
                    object_=auth_fields, annotation=SourceConnectionUpdateAuthFields, direction="write"
                ),
                "config_fields": config_fields,
                "cron_schedule": cron_schedule,
                "connection_id": connection_id,
                "white_label_id": white_label_id,
                "auth_provider": auth_provider,
                "auth_provider_config": auth_provider_config,
            },
            headers={
                "content-type": "application/json",
            },
            request_options=request_options,
            omit=OMIT,
        )
        try:
            if 200 <= _response.status_code < 300:
                _data = typing.cast(
                    SourceConnection,
                    parse_obj_as(
                        type_=SourceConnection,  # type: ignore
                        object_=_response.json(),
                    ),
                )
                return AsyncHttpResponse(response=_response, data=_data)
            if _response.status_code == 422:
                raise UnprocessableEntityError(
                    headers=dict(_response.headers),
                    body=typing.cast(
                        HttpValidationError,
                        parse_obj_as(
                            type_=HttpValidationError,  # type: ignore
                            object_=_response.json(),
                        ),
                    ),
                )
            _response_json = _response.json()
        except JSONDecodeError:
            raise ApiError(status_code=_response.status_code, headers=dict(_response.headers), body=_response.text)
        raise ApiError(status_code=_response.status_code, headers=dict(_response.headers), body=_response_json)

    async def delete_source_connection(
        self, source_connection_id: str, *, request_options: typing.Optional[RequestOptions] = None
    ) -> AsyncHttpResponse[SourceConnection]:
        """
        Delete a source connection and all associated data.

        <br/><br/>

        Permanently removes the source connection configuration, credentials, and all synced data
        from the destination systems. This action cannot be undone.

        Parameters
        ----------
        source_connection_id : str
            The unique identifier of the source connection to delete

        request_options : typing.Optional[RequestOptions]
            Request-specific configuration.

        Returns
        -------
        AsyncHttpResponse[SourceConnection]
            Successful Response
        """
        _response = await self._client_wrapper.httpx_client.request(
            f"source-connections/{jsonable_encoder(source_connection_id)}",
            method="DELETE",
            request_options=request_options,
        )
        try:
            if 200 <= _response.status_code < 300:
                _data = typing.cast(
                    SourceConnection,
                    parse_obj_as(
                        type_=SourceConnection,  # type: ignore
                        object_=_response.json(),
                    ),
                )
                return AsyncHttpResponse(response=_response, data=_data)
            if _response.status_code == 422:
                raise UnprocessableEntityError(
                    headers=dict(_response.headers),
                    body=typing.cast(
                        HttpValidationError,
                        parse_obj_as(
                            type_=HttpValidationError,  # type: ignore
                            object_=_response.json(),
                        ),
                    ),
                )
            _response_json = _response.json()
        except JSONDecodeError:
            raise ApiError(status_code=_response.status_code, headers=dict(_response.headers), body=_response.text)
        raise ApiError(status_code=_response.status_code, headers=dict(_response.headers), body=_response_json)

    async def run_source_connection(
        self,
        source_connection_id: str,
        *,
        access_token: typing.Optional[str] = OMIT,
        request_options: typing.Optional[RequestOptions] = None,
    ) -> AsyncHttpResponse[SourceConnectionJob]:
        """
        Manually trigger a data sync for this source connection.

        <br/><br/>
        Starts an immediate synchronization job that extracts fresh data from your source,
        transforms it according to your configuration, and updates the destination systems.
        The job runs asynchronously and endpoint returns immediately with tracking information.

        Parameters
        ----------
        source_connection_id : str
            The unique identifier of the source connection to sync

        access_token : typing.Optional[str]
            This parameter gives you the ability to start a sync job with an access token for an OAuth2.0 source directly instead of using the credentials that Airweave has stored for you. Learn more about direct token injection [here](https://docs.airweave.ai/direct-token-injection).

        request_options : typing.Optional[RequestOptions]
            Request-specific configuration.

        Returns
        -------
        AsyncHttpResponse[SourceConnectionJob]
            Successful Response
        """
        _response = await self._client_wrapper.httpx_client.request(
            f"source-connections/{jsonable_encoder(source_connection_id)}/run",
            method="POST",
            json={
                "access_token": access_token,
            },
            headers={
                "content-type": "application/json",
            },
            request_options=request_options,
            omit=OMIT,
        )
        try:
            if 200 <= _response.status_code < 300:
                _data = typing.cast(
                    SourceConnectionJob,
                    parse_obj_as(
                        type_=SourceConnectionJob,  # type: ignore
                        object_=_response.json(),
                    ),
                )
                return AsyncHttpResponse(response=_response, data=_data)
            if _response.status_code == 422:
                raise UnprocessableEntityError(
                    headers=dict(_response.headers),
                    body=typing.cast(
                        HttpValidationError,
                        parse_obj_as(
                            type_=HttpValidationError,  # type: ignore
                            object_=_response.json(),
                        ),
                    ),
                )
            _response_json = _response.json()
        except JSONDecodeError:
            raise ApiError(status_code=_response.status_code, headers=dict(_response.headers), body=_response.text)
        raise ApiError(status_code=_response.status_code, headers=dict(_response.headers), body=_response_json)

    async def list_source_connection_jobs(
        self, source_connection_id: str, *, request_options: typing.Optional[RequestOptions] = None
    ) -> AsyncHttpResponse[typing.List[SourceConnectionJob]]:
        """
        List all sync jobs for a source connection.

        <br/><br/>
        Returns the complete history of data synchronization jobs including successful syncs,
        failed attempts, and currently running operations.

        Parameters
        ----------
        source_connection_id : str
            The unique identifier of the source connection

        request_options : typing.Optional[RequestOptions]
            Request-specific configuration.

        Returns
        -------
        AsyncHttpResponse[typing.List[SourceConnectionJob]]
            Successful Response
        """
        _response = await self._client_wrapper.httpx_client.request(
            f"source-connections/{jsonable_encoder(source_connection_id)}/jobs",
            method="GET",
            request_options=request_options,
        )
        try:
            if 200 <= _response.status_code < 300:
                _data = typing.cast(
                    typing.List[SourceConnectionJob],
                    parse_obj_as(
                        type_=typing.List[SourceConnectionJob],  # type: ignore
                        object_=_response.json(),
                    ),
                )
                return AsyncHttpResponse(response=_response, data=_data)
            if _response.status_code == 422:
                raise UnprocessableEntityError(
                    headers=dict(_response.headers),
                    body=typing.cast(
                        HttpValidationError,
                        parse_obj_as(
                            type_=HttpValidationError,  # type: ignore
                            object_=_response.json(),
                        ),
                    ),
                )
            _response_json = _response.json()
        except JSONDecodeError:
            raise ApiError(status_code=_response.status_code, headers=dict(_response.headers), body=_response.text)
        raise ApiError(status_code=_response.status_code, headers=dict(_response.headers), body=_response_json)

    async def get_source_connection_job(
        self, source_connection_id: str, job_id: str, *, request_options: typing.Optional[RequestOptions] = None
    ) -> AsyncHttpResponse[SourceConnectionJob]:
        """
        Get detailed information about a specific sync job.

        Parameters
        ----------
        source_connection_id : str
            The unique identifier of the source connection

        job_id : str
            The unique identifier of the sync job

        request_options : typing.Optional[RequestOptions]
            Request-specific configuration.

        Returns
        -------
        AsyncHttpResponse[SourceConnectionJob]
            Successful Response
        """
        _response = await self._client_wrapper.httpx_client.request(
            f"source-connections/{jsonable_encoder(source_connection_id)}/jobs/{jsonable_encoder(job_id)}",
            method="GET",
            request_options=request_options,
        )
        try:
            if 200 <= _response.status_code < 300:
                _data = typing.cast(
                    SourceConnectionJob,
                    parse_obj_as(
                        type_=SourceConnectionJob,  # type: ignore
                        object_=_response.json(),
                    ),
                )
                return AsyncHttpResponse(response=_response, data=_data)
            if _response.status_code == 422:
                raise UnprocessableEntityError(
                    headers=dict(_response.headers),
                    body=typing.cast(
                        HttpValidationError,
                        parse_obj_as(
                            type_=HttpValidationError,  # type: ignore
                            object_=_response.json(),
                        ),
                    ),
                )
            _response_json = _response.json()
        except JSONDecodeError:
            raise ApiError(status_code=_response.status_code, headers=dict(_response.headers), body=_response.text)
        raise ApiError(status_code=_response.status_code, headers=dict(_response.headers), body=_response_json)

    async def cancel_source_connection_job(
        self, source_connection_id: str, job_id: str, *, request_options: typing.Optional[RequestOptions] = None
    ) -> AsyncHttpResponse[SourceConnectionJob]:
        """
        Cancel a running sync job.

        <br/><br/>
        Sends a cancellation signal to stop an in-progress data synchronization.
        The job will complete its current operation and then terminate gracefully.
        Only jobs in 'created', 'pending', or 'in_progress' states can be cancelled.

        Parameters
        ----------
        source_connection_id : str
            The unique identifier of the source connection

        job_id : str
            The unique identifier of the sync job to cancel

        request_options : typing.Optional[RequestOptions]
            Request-specific configuration.

        Returns
        -------
        AsyncHttpResponse[SourceConnectionJob]
            Successful Response
        """
        _response = await self._client_wrapper.httpx_client.request(
            f"source-connections/{jsonable_encoder(source_connection_id)}/jobs/{jsonable_encoder(job_id)}/cancel",
            method="POST",
            request_options=request_options,
        )
        try:
            if 200 <= _response.status_code < 300:
                _data = typing.cast(
                    SourceConnectionJob,
                    parse_obj_as(
                        type_=SourceConnectionJob,  # type: ignore
                        object_=_response.json(),
                    ),
                )
                return AsyncHttpResponse(response=_response, data=_data)
            if _response.status_code == 422:
                raise UnprocessableEntityError(
                    headers=dict(_response.headers),
                    body=typing.cast(
                        HttpValidationError,
                        parse_obj_as(
                            type_=HttpValidationError,  # type: ignore
                            object_=_response.json(),
                        ),
                    ),
                )
            _response_json = _response.json()
        except JSONDecodeError:
            raise ApiError(status_code=_response.status_code, headers=dict(_response.headers), body=_response.text)
        raise ApiError(status_code=_response.status_code, headers=dict(_response.headers), body=_response_json)
