import * as ec2 from '@aws-cdk/aws-ec2';
import * as iam from '@aws-cdk/aws-iam';
import { IResource, Resource } from '@aws-cdk/core';
import { Construct } from 'constructs';
import { ContainerDefinition, ContainerDefinitionOptions, PortMapping, Protocol } from '../container-definition';
import { FirelensLogRouter, FirelensLogRouterDefinitionOptions } from '../firelens-log-router';
import { PlacementConstraint } from '../placement';
import { ProxyConfiguration } from '../proxy-configuration/proxy-configuration';
/**
 * The interface for all task definitions.
 *
 * @stability stable
 */
export interface ITaskDefinition extends IResource {
    /**
     * ARN of this task definition.
     *
     * @stability stable
     * @attribute true
     */
    readonly taskDefinitionArn: string;
    /**
     * Execution role for this task definition.
     *
     * @stability stable
     */
    readonly executionRole?: iam.IRole;
    /**
     * What launch types this task definition should be compatible with.
     *
     * @stability stable
     */
    readonly compatibility: Compatibility;
    /**
     * Return true if the task definition can be run on an EC2 cluster.
     *
     * @stability stable
     */
    readonly isEc2Compatible: boolean;
    /**
     * Return true if the task definition can be run on a Fargate cluster.
     *
     * @stability stable
     */
    readonly isFargateCompatible: boolean;
    /**
     * The networking mode to use for the containers in the task.
     *
     * @stability stable
     */
    readonly networkMode: NetworkMode;
    /**
     * The name of the IAM role that grants containers in the task permission to call AWS APIs on your behalf.
     *
     * @stability stable
     */
    readonly taskRole: iam.IRole;
}
/**
 * The common properties for all task definitions.
 *
 * For more information, see
 * [Task Definition Parameters](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task_definition_parameters.html).
 *
 * @stability stable
 */
export interface CommonTaskDefinitionProps {
    /**
     * The name of a family that this task definition is registered to.
     *
     * A family groups multiple versions of a task definition.
     *
     * @default - Automatically generated name.
     * @stability stable
     */
    readonly family?: string;
    /**
     * The name of the IAM task execution role that grants the ECS agent to call AWS APIs on your behalf.
     *
     * The role will be used to retrieve container images from ECR and create CloudWatch log groups.
     *
     * @default - An execution role will be automatically created if you use ECR images in your task definition.
     * @stability stable
     */
    readonly executionRole?: iam.IRole;
    /**
     * The name of the IAM role that grants containers in the task permission to call AWS APIs on your behalf.
     *
     * @default - A task role is automatically created for you.
     * @stability stable
     */
    readonly taskRole?: iam.IRole;
    /**
     * The configuration details for the App Mesh proxy.
     *
     * @default - No proxy configuration.
     * @stability stable
     */
    readonly proxyConfiguration?: ProxyConfiguration;
    /**
     * The list of volume definitions for the task.
     *
     * For more information, see
     * [Task Definition Parameter Volumes](https://docs.aws.amazon.com/AmazonECS/latest/developerguide//task_definition_parameters.html#volumes).
     *
     * @default - No volumes are passed to the Docker daemon on a container instance.
     * @stability stable
     */
    readonly volumes?: Volume[];
}
/**
 * The properties for task definitions.
 *
 * @stability stable
 */
export interface TaskDefinitionProps extends CommonTaskDefinitionProps {
    /**
     * The networking mode to use for the containers in the task.
     *
     * On Fargate, the only supported networking mode is AwsVpc.
     *
     * @default - NetworkMode.Bridge for EC2 tasks, AwsVpc for Fargate tasks.
     * @stability stable
     */
    readonly networkMode?: NetworkMode;
    /**
     * The placement constraints to use for tasks in the service.
     *
     * You can specify a maximum of 10 constraints per task (this limit includes
     * constraints in the task definition and those specified at run time).
     *
     * Not supported in Fargate.
     *
     * @default - No placement constraints.
     * @stability stable
     */
    readonly placementConstraints?: PlacementConstraint[];
    /**
     * The task launch type compatiblity requirement.
     *
     * @stability stable
     */
    readonly compatibility: Compatibility;
    /**
     * The number of cpu units used by the task.
     *
     * If you are using the EC2 launch type, this field is optional and any value can be used.
     * If you are using the Fargate launch type, this field is required and you must use one of the following values,
     * which determines your range of valid values for the memory parameter:
     *
     * 256 (.25 vCPU) - Available memory values: 512 (0.5 GB), 1024 (1 GB), 2048 (2 GB)
     *
     * 512 (.5 vCPU) - Available memory values: 1024 (1 GB), 2048 (2 GB), 3072 (3 GB), 4096 (4 GB)
     *
     * 1024 (1 vCPU) - Available memory values: 2048 (2 GB), 3072 (3 GB), 4096 (4 GB), 5120 (5 GB), 6144 (6 GB), 7168 (7 GB), 8192 (8 GB)
     *
     * 2048 (2 vCPU) - Available memory values: Between 4096 (4 GB) and 16384 (16 GB) in increments of 1024 (1 GB)
     *
     * 4096 (4 vCPU) - Available memory values: Between 8192 (8 GB) and 30720 (30 GB) in increments of 1024 (1 GB)
     *
     * @default - CPU units are not specified.
     * @stability stable
     */
    readonly cpu?: string;
    /**
     * The amount (in MiB) of memory used by the task.
     *
     * If using the EC2 launch type, this field is optional and any value can be used.
     * If using the Fargate launch type, this field is required and you must use one of the following values,
     * which determines your range of valid values for the cpu parameter:
     *
     * 512 (0.5 GB), 1024 (1 GB), 2048 (2 GB) - Available cpu values: 256 (.25 vCPU)
     *
     * 1024 (1 GB), 2048 (2 GB), 3072 (3 GB), 4096 (4 GB) - Available cpu values: 512 (.5 vCPU)
     *
     * 2048 (2 GB), 3072 (3 GB), 4096 (4 GB), 5120 (5 GB), 6144 (6 GB), 7168 (7 GB), 8192 (8 GB) - Available cpu values: 1024 (1 vCPU)
     *
     * Between 4096 (4 GB) and 16384 (16 GB) in increments of 1024 (1 GB) - Available cpu values: 2048 (2 vCPU)
     *
     * Between 8192 (8 GB) and 30720 (30 GB) in increments of 1024 (1 GB) - Available cpu values: 4096 (4 vCPU)
     *
     * @default - Memory used by task is not specified.
     * @stability stable
     */
    readonly memoryMiB?: string;
    /**
     * The IPC resource namespace to use for the containers in the task.
     *
     * Not supported in Fargate and Windows containers.
     *
     * @default - IpcMode used by the task is not specified
     * @stability stable
     */
    readonly ipcMode?: IpcMode;
    /**
     * The process namespace to use for the containers in the task.
     *
     * Not supported in Fargate and Windows containers.
     *
     * @default - PidMode used by the task is not specified
     * @stability stable
     */
    readonly pidMode?: PidMode;
    /**
     * The inference accelerators to use for the containers in the task.
     *
     * Not supported in Fargate.
     *
     * @default - No inference accelerators.
     * @stability stable
     */
    readonly inferenceAccelerators?: InferenceAccelerator[];
}
/**
 * The common task definition attributes used across all types of task definitions.
 *
 * @stability stable
 */
export interface CommonTaskDefinitionAttributes {
    /**
     * The arn of the task definition.
     *
     * @stability stable
     */
    readonly taskDefinitionArn: string;
    /**
     * The networking mode to use for the containers in the task.
     *
     * @default Network mode cannot be provided to the imported task.
     * @stability stable
     */
    readonly networkMode?: NetworkMode;
    /**
     * The name of the IAM role that grants containers in the task permission to call AWS APIs on your behalf.
     *
     * @default Permissions cannot be granted to the imported task.
     * @stability stable
     */
    readonly taskRole?: iam.IRole;
}
/**
 * A reference to an existing task definition.
 *
 * @stability stable
 */
export interface TaskDefinitionAttributes extends CommonTaskDefinitionAttributes {
    /**
     * What launch types this task definition should be compatible with.
     *
     * @default Compatibility.EC2_AND_FARGATE
     * @stability stable
     */
    readonly compatibility?: Compatibility;
}
declare abstract class TaskDefinitionBase extends Resource implements ITaskDefinition {
    abstract readonly compatibility: Compatibility;
    abstract readonly networkMode: NetworkMode;
    abstract readonly taskDefinitionArn: string;
    abstract readonly taskRole: iam.IRole;
    abstract readonly executionRole?: iam.IRole;
    /**
     * Return true if the task definition can be run on an EC2 cluster.
     *
     * @stability stable
     */
    get isEc2Compatible(): boolean;
    /**
     * Return true if the task definition can be run on a Fargate cluster.
     *
     * @stability stable
     */
    get isFargateCompatible(): boolean;
}
/**
 * The base class for all task definitions.
 *
 * @stability stable
 */
export declare class TaskDefinition extends TaskDefinitionBase {
    /**
     * Imports a task definition from the specified task definition ARN.
     *
     * The task will have a compatibility of EC2+Fargate.
     *
     * @stability stable
     */
    static fromTaskDefinitionArn(scope: Construct, id: string, taskDefinitionArn: string): ITaskDefinition;
    /**
     * Create a task definition from a task definition reference.
     *
     * @stability stable
     */
    static fromTaskDefinitionAttributes(scope: Construct, id: string, attrs: TaskDefinitionAttributes): ITaskDefinition;
    /**
     * The name of a family that this task definition is registered to.
     *
     * A family groups multiple versions of a task definition.
     *
     * @stability stable
     */
    readonly family: string;
    /**
     * The full Amazon Resource Name (ARN) of the task definition.
     *
     * @stability stable
     * @attribute true
     */
    readonly taskDefinitionArn: string;
    /**
     * The name of the IAM role that grants containers in the task permission to call AWS APIs on your behalf.
     *
     * @stability stable
     */
    readonly taskRole: iam.IRole;
    /**
     * The networking mode to use for the containers in the task.
     *
     * @stability stable
     */
    readonly networkMode: NetworkMode;
    /**
     * Default container for this task.
     *
     * Load balancers will send traffic to this container. The first
     * essential container that is added to this task will become the default
     * container.
     *
     * @stability stable
     */
    defaultContainer?: ContainerDefinition;
    /**
     * The task launch type compatibility requirement.
     *
     * @stability stable
     */
    readonly compatibility: Compatibility;
    /**
     * The container definitions.
     *
     * @stability stable
     */
    protected readonly containers: ContainerDefinition[];
    /**
     * All volumes
     */
    private readonly volumes;
    /**
     * Placement constraints for task instances
     */
    private readonly placementConstraints;
    /**
     * Inference accelerators for task instances
     */
    private readonly _inferenceAccelerators;
    private _executionRole?;
    private _referencesSecretJsonField?;
    /**
     * Constructs a new instance of the TaskDefinition class.
     *
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: TaskDefinitionProps);
    /**
     * Execution role for this task definition.
     *
     * @stability stable
     */
    get executionRole(): iam.IRole | undefined;
    /**
     * Public getter method to access list of inference accelerators attached to the instance.
     *
     * @stability stable
     */
    get inferenceAccelerators(): InferenceAccelerator[];
    private renderVolumes;
    private renderInferenceAccelerators;
    /**
     * Validate the existence of the input target and set default values.
     *
     * @internal
     */
    _validateTarget(options: LoadBalancerTargetOptions): LoadBalancerTarget;
    /**
     * Returns the port range to be opened that match the provided container name and container port.
     *
     * @internal
     */
    _portRangeFromPortMapping(portMapping: PortMapping): ec2.Port;
    /**
     * Adds a policy statement to the task IAM role.
     *
     * @stability stable
     */
    addToTaskRolePolicy(statement: iam.PolicyStatement): void;
    /**
     * Adds a policy statement to the task execution IAM role.
     *
     * @stability stable
     */
    addToExecutionRolePolicy(statement: iam.PolicyStatement): void;
    /**
     * Adds a new container to the task definition.
     *
     * @stability stable
     */
    addContainer(id: string, props: ContainerDefinitionOptions): ContainerDefinition;
    /**
     * Adds a firelens log router to the task definition.
     *
     * @stability stable
     */
    addFirelensLogRouter(id: string, props: FirelensLogRouterDefinitionOptions): FirelensLogRouter;
    /**
     * Links a container to this task definition.
     * @internal
     */
    _linkContainer(container: ContainerDefinition): void;
    /**
     * Adds a volume to the task definition.
     *
     * @stability stable
     */
    addVolume(volume: Volume): void;
    /**
     * Adds the specified placement constraint to the task definition.
     *
     * @stability stable
     */
    addPlacementConstraint(constraint: PlacementConstraint): void;
    /**
     * Adds the specified extention to the task definition.
     *
     * Extension can be used to apply a packaged modification to
     * a task definition.
     *
     * @stability stable
     */
    addExtension(extension: ITaskDefinitionExtension): void;
    /**
     * Adds an inference accelerator to the task definition.
     *
     * @stability stable
     */
    addInferenceAccelerator(inferenceAccelerator: InferenceAccelerator): void;
    /**
     * Creates the task execution IAM role if it doesn't already exist.
     *
     * @stability stable
     */
    obtainExecutionRole(): iam.IRole;
    /**
     * Whether this task definition has at least a container that references a specific JSON field of a secret stored in Secrets Manager.
     *
     * @stability stable
     */
    get referencesSecretJsonField(): boolean | undefined;
    /**
     * Validates the task definition.
     *
     * @stability stable
     */
    protected validate(): string[];
    /**
     * Returns the container that match the provided containerName.
     */
    private findContainer;
    private renderNetworkMode;
    private renderContainers;
}
/**
 * The networking mode to use for the containers in the task.
 *
 * @stability stable
 */
export declare enum NetworkMode {
    /**
     * The task's containers do not have external connectivity and port mappings can't be specified in the container definition.
     *
     * @stability stable
     */
    NONE = "none",
    /**
     * The task utilizes Docker's built-in virtual network which runs inside each container instance.
     *
     * @stability stable
     */
    BRIDGE = "bridge",
    /**
     * The task is allocated an elastic network interface.
     *
     * @stability stable
     */
    AWS_VPC = "awsvpc",
    /**
     * The task bypasses Docker's built-in virtual network and maps container ports directly to the EC2 instance's network interface directly.
     *
     * In this mode, you can't run multiple instantiations of the same task on a
     * single container instance when port mappings are used.
     *
     * @stability stable
     */
    HOST = "host",
    /**
     * The task utilizes NAT network mode required by Windows containers.
     *
     * This is the only supported network mode for Windows containers. For more information, see
     * [Task Definition Parameters](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task_definition_parameters.html#network_mode).
     *
     * @stability stable
     */
    NAT = "nat"
}
/**
 * The IPC resource namespace to use for the containers in the task.
 *
 * @stability stable
 */
export declare enum IpcMode {
    /**
     * If none is specified, then IPC resources within the containers of a task are private and not shared with other containers in a task or on the container instance.
     *
     * @stability stable
     */
    NONE = "none",
    /**
     * If host is specified, then all containers within the tasks that specified the host IPC mode on the same container instance share the same IPC resources with the host Amazon EC2 instance.
     *
     * @stability stable
     */
    HOST = "host",
    /**
     * If task is specified, all containers within the specified task share the same IPC resources.
     *
     * @stability stable
     */
    TASK = "task"
}
/**
 * The process namespace to use for the containers in the task.
 *
 * @stability stable
 */
export declare enum PidMode {
    /**
     * If host is specified, then all containers within the tasks that specified the host PID mode on the same container instance share the same process namespace with the host Amazon EC2 instance.
     *
     * @stability stable
     */
    HOST = "host",
    /**
     * If task is specified, all containers within the specified task share the same process namespace.
     *
     * @stability stable
     */
    TASK = "task"
}
/**
 * Elastic Inference Accelerator.
 *
 * For more information, see [Elastic Inference Basics](https://docs.aws.amazon.com/elastic-inference/latest/developerguide/basics.html)
 *
 * @stability stable
 */
export interface InferenceAccelerator {
    /**
     * The Elastic Inference accelerator device name.
     *
     * @default - empty
     * @stability stable
     */
    readonly deviceName?: string;
    /**
     * The Elastic Inference accelerator type to use.
     *
     * The allowed values are: eia2.medium, eia2.large and eia2.xlarge.
     *
     * @default - empty
     * @stability stable
     */
    readonly deviceType?: string;
}
/**
 * A data volume used in a task definition.
 *
 * For tasks that use a Docker volume, specify a DockerVolumeConfiguration.
 * For tasks that use a bind mount host volume, specify a host and optional sourcePath.
 *
 * For more information, see [Using Data Volumes in Tasks](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_data_volumes.html).
 *
 * @stability stable
 */
export interface Volume {
    /**
     * This property is specified when you are using bind mount host volumes.
     *
     * Bind mount host volumes are supported when you are using either the EC2 or Fargate launch types.
     * The contents of the host parameter determine whether your bind mount host volume persists on the
     * host container instance and where it is stored. If the host parameter is empty, then the Docker
     * daemon assigns a host path for your data volume. However, the data is not guaranteed to persist
     * after the containers associated with it stop running.
     *
     * @stability stable
     */
    readonly host?: Host;
    /**
     * The name of the volume.
     *
     * Up to 255 letters (uppercase and lowercase), numbers, and hyphens are allowed.
     * This name is referenced in the sourceVolume parameter of container definition mountPoints.
     *
     * @stability stable
     */
    readonly name: string;
    /**
     * This property is specified when you are using Docker volumes.
     *
     * Docker volumes are only supported when you are using the EC2 launch type.
     * Windows containers only support the use of the local driver.
     * To use bind mounts, specify a host instead.
     *
     * @stability stable
     */
    readonly dockerVolumeConfiguration?: DockerVolumeConfiguration;
    /**
     * This property is specified when you are using Amazon EFS.
     *
     * When specifying Amazon EFS volumes in tasks using the Fargate launch type,
     * Fargate creates a supervisor container that is responsible for managing the Amazon EFS volume.
     * The supervisor container uses a small amount of the task's memory.
     * The supervisor container is visible when querying the task metadata version 4 endpoint,
     * but is not visible in CloudWatch Container Insights.
     *
     * @default No Elastic FileSystem is setup
     * @stability stable
     */
    readonly efsVolumeConfiguration?: EfsVolumeConfiguration;
}
/**
 * The details on a container instance bind mount host volume.
 *
 * @stability stable
 */
export interface Host {
    /**
     * Specifies the path on the host container instance that is presented to the container.
     *
     * If the sourcePath value does not exist on the host container instance, the Docker daemon creates it.
     * If the location does exist, the contents of the source path folder are exported.
     *
     * This property is not supported for tasks that use the Fargate launch type.
     *
     * @stability stable
     */
    readonly sourcePath?: string;
}
/**
 * Properties for an ECS target.
 *
 * @internal
 */
export interface LoadBalancerTarget {
    /**
     * The name of the container.
     */
    readonly containerName: string;
    /**
     * The port mapping of the target.
     */
    readonly portMapping: PortMapping;
}
/**
 * Properties for defining an ECS target.
 *
 * The port mapping for it must already have been created through addPortMapping().
 *
 * @stability stable
 */
export interface LoadBalancerTargetOptions {
    /**
     * The name of the container.
     *
     * @stability stable
     */
    readonly containerName: string;
    /**
     * The port number of the container.
     *
     * Only applicable when using application/network load balancers.
     *
     * @default - Container port of the first added port mapping.
     * @stability stable
     */
    readonly containerPort?: number;
    /**
     * The protocol used for the port mapping.
     *
     * Only applicable when using application load balancers.
     *
     * @default Protocol.TCP
     * @stability stable
     */
    readonly protocol?: Protocol;
}
/**
 * The configuration for a Docker volume.
 *
 * Docker volumes are only supported when you are using the EC2 launch type.
 *
 * @stability stable
 */
export interface DockerVolumeConfiguration {
    /**
     * Specifies whether the Docker volume should be created if it does not already exist.
     *
     * If true is specified, the Docker volume will be created for you.
     *
     * @default false
     * @stability stable
     */
    readonly autoprovision?: boolean;
    /**
     * The Docker volume driver to use.
     *
     * @stability stable
     */
    readonly driver: string;
    /**
     * A map of Docker driver-specific options passed through.
     *
     * @default No options
     * @stability stable
     */
    readonly driverOpts?: {
        [key: string]: string;
    };
    /**
     * Custom metadata to add to your Docker volume.
     *
     * @default No labels
     * @stability stable
     */
    readonly labels?: {
        [key: string]: string;
    };
    /**
     * The scope for the Docker volume that determines its lifecycle.
     *
     * @stability stable
     */
    readonly scope: Scope;
}
/**
 * The authorization configuration details for the Amazon EFS file system.
 *
 * @stability stable
 */
export interface AuthorizationConfig {
    /**
     * The access point ID to use.
     *
     * If an access point is specified, the root directory value will be
     * relative to the directory set for the access point.
     * If specified, transit encryption must be enabled in the EFSVolumeConfiguration.
     *
     * @default No id
     * @stability stable
     */
    readonly accessPointId?: string;
    /**
     * Whether or not to use the Amazon ECS task IAM role defined in a task definition when mounting the Amazon EFS file system.
     *
     * If enabled, transit encryption must be enabled in the EFSVolumeConfiguration.
     *
     * Valid values: ENABLED | DISABLED
     *
     * @default If this parameter is omitted, the default value of DISABLED is used.
     * @stability stable
     */
    readonly iam?: string;
}
/**
 * The configuration for an Elastic FileSystem volume.
 *
 * @stability stable
 */
export interface EfsVolumeConfiguration {
    /**
     * The Amazon EFS file system ID to use.
     *
     * @stability stable
     */
    readonly fileSystemId: string;
    /**
     * The directory within the Amazon EFS file system to mount as the root directory inside the host.
     *
     * Specifying / will have the same effect as omitting this parameter.
     *
     * @default The root of the Amazon EFS volume
     * @stability stable
     */
    readonly rootDirectory?: string;
    /**
     * Whether or not to enable encryption for Amazon EFS data in transit between the Amazon ECS host and the Amazon EFS server.
     *
     * Transit encryption must be enabled if Amazon EFS IAM authorization is used.
     *
     * Valid values: ENABLED | DISABLED
     *
     * @default DISABLED
     * @stability stable
     */
    readonly transitEncryption?: string;
    /**
     * The port to use when sending encrypted data between the Amazon ECS host and the Amazon EFS server.
     *
     * EFS mount helper uses.
     *
     * @default Port selection strategy that the Amazon EFS mount helper uses.
     * @stability stable
     */
    readonly transitEncryptionPort?: number;
    /**
     * The authorization configuration details for the Amazon EFS file system.
     *
     * @default No configuration.
     * @stability stable
     */
    readonly authorizationConfig?: AuthorizationConfig;
}
/**
 * The scope for the Docker volume that determines its lifecycle.
 *
 * Docker volumes that are scoped to a task are automatically provisioned when the task starts and destroyed when the task stops.
 * Docker volumes that are scoped as shared persist after the task stops.
 *
 * @stability stable
 */
export declare enum Scope {
    /**
     * Docker volumes that are scoped to a task are automatically provisioned when the task starts and destroyed when the task stops.
     *
     * @stability stable
     */
    TASK = "task",
    /**
     * Docker volumes that are scoped as shared persist after the task stops.
     *
     * @stability stable
     */
    SHARED = "shared"
}
/**
 * The task launch type compatibility requirement.
 *
 * @stability stable
 */
export declare enum Compatibility {
    /**
     * The task should specify the EC2 launch type.
     *
     * @stability stable
     */
    EC2 = 0,
    /**
     * The task should specify the Fargate launch type.
     *
     * @stability stable
     */
    FARGATE = 1,
    /**
     * The task can specify either the EC2 or Fargate launch types.
     *
     * @stability stable
     */
    EC2_AND_FARGATE = 2
}
/**
 * An extension for Task Definitions.
 *
 * Classes that want to make changes to a TaskDefinition (such as
 * adding helper containers) can implement this interface, and can
 * then be "added" to a TaskDefinition like so:
 *
 *     taskDefinition.addExtension(new MyExtension("some_parameter"));
 *
 * @stability stable
 */
export interface ITaskDefinitionExtension {
    /**
     * Apply the extension to the given TaskDefinition.
     *
     * @param taskDefinition [disable-awslint:ref-via-interface].
     * @stability stable
     */
    extend(taskDefinition: TaskDefinition): void;
}
/**
 * Return true if the given task definition can be run on an EC2 cluster
 */
export declare function isEc2Compatible(compatibility: Compatibility): boolean;
/**
 * Return true if the given task definition can be run on a Fargate cluster
 */
export declare function isFargateCompatible(compatibility: Compatibility): boolean;
export {};
