import * as autoscaling from '@aws-cdk/aws-autoscaling';
import * as cloudwatch from '@aws-cdk/aws-cloudwatch';
import * as ec2 from '@aws-cdk/aws-ec2';
import * as kms from '@aws-cdk/aws-kms';
import * as cloudmap from '@aws-cdk/aws-servicediscovery';
import { Duration, IResource, Resource } from '@aws-cdk/core';
import { Construct } from 'constructs';
import { Construct as CoreConstruct } from '@aws-cdk/core';
/**
 * The properties used to define an ECS cluster.
 *
 * @stability stable
 */
export interface ClusterProps {
    /**
     * The name for the cluster.
     *
     * @default CloudFormation-generated name
     * @stability stable
     */
    readonly clusterName?: string;
    /**
     * The VPC where your ECS instances will be running or your ENIs will be deployed.
     *
     * @default - creates a new VPC with two AZs
     * @stability stable
     */
    readonly vpc?: ec2.IVpc;
    /**
     * The service discovery namespace created in this cluster.
     *
     * @default - no service discovery namespace created, you can use `addDefaultCloudMapNamespace` to add a
     * default service discovery namespace later.
     * @stability stable
     */
    readonly defaultCloudMapNamespace?: CloudMapNamespaceOptions;
    /**
     * The ec2 capacity to add to the cluster.
     *
     * @default - no EC2 capacity will be added, you can use `addCapacity` to add capacity later.
     * @stability stable
     */
    readonly capacity?: AddCapacityOptions;
    /**
     * (deprecated) The capacity providers to add to the cluster.
     *
     * @default - None. Currently only FARGATE and FARGATE_SPOT are supported.
     * @deprecated Use {@link ClusterProps.enableFargateCapacityProviders} instead.
     */
    readonly capacityProviders?: string[];
    /**
     * Whether to enable Fargate Capacity Providers.
     *
     * @default false
     * @stability stable
     */
    readonly enableFargateCapacityProviders?: boolean;
    /**
     * If true CloudWatch Container Insights will be enabled for the cluster.
     *
     * @default - Container Insights will be disabled for this cluser.
     * @stability stable
     */
    readonly containerInsights?: boolean;
}
/**
 * The machine image type.
 *
 * @stability stable
 */
export declare enum MachineImageType {
    /**
     * Amazon ECS-optimized Amazon Linux 2 AMI.
     *
     * @stability stable
     */
    AMAZON_LINUX_2 = 0,
    /**
     * Bottlerocket AMI.
     *
     * @stability stable
     */
    BOTTLEROCKET = 1
}
/**
 * A regional grouping of one or more container instances on which you can run tasks and services.
 *
 * @stability stable
 */
export declare class Cluster extends Resource implements ICluster {
    /**
     * Import an existing cluster to the stack from its attributes.
     *
     * @stability stable
     */
    static fromClusterAttributes(scope: Construct, id: string, attrs: ClusterAttributes): ICluster;
    /**
     * Manage the allowed network connections for the cluster with Security Groups.
     *
     * @stability stable
     */
    readonly connections: ec2.Connections;
    /**
     * The VPC associated with the cluster.
     *
     * @stability stable
     */
    readonly vpc: ec2.IVpc;
    /**
     * The Amazon Resource Name (ARN) that identifies the cluster.
     *
     * @stability stable
     */
    readonly clusterArn: string;
    /**
     * The name of the cluster.
     *
     * @stability stable
     */
    readonly clusterName: string;
    /**
     * The cluster-level (FARGATE, FARGATE_SPOT) capacity providers.
     */
    private _fargateCapacityProviders;
    /**
     * The EC2 Auto Scaling Group capacity providers associated with the cluster.
     */
    private _asgCapacityProviders;
    /**
     * The AWS Cloud Map namespace to associate with the cluster.
     */
    private _defaultCloudMapNamespace?;
    /**
     * Specifies whether the cluster has EC2 instance capacity.
     */
    private _hasEc2Capacity;
    /**
     * The autoscaling group for added Ec2 capacity
     */
    private _autoscalingGroup?;
    /**
     * Constructs a new instance of the Cluster class.
     *
     * @stability stable
     */
    constructor(scope: Construct, id: string, props?: ClusterProps);
    /**
     * Enable the Fargate capacity providers for this cluster.
     *
     * @stability stable
     */
    enableFargateCapacityProviders(): void;
    /**
     * Add an AWS Cloud Map DNS namespace for this cluster.
     *
     * NOTE: HttpNamespaces are not supported, as ECS always requires a DNSConfig when registering an instance to a Cloud
     * Map service.
     *
     * @stability stable
     */
    addDefaultCloudMapNamespace(options: CloudMapNamespaceOptions): cloudmap.INamespace;
    /**
     * Getter for namespace added to cluster.
     *
     * @stability stable
     */
    get defaultCloudMapNamespace(): cloudmap.INamespace | undefined;
    /**
     * (deprecated) This method adds compute capacity to a cluster by creating an AutoScalingGroup with the specified options.
     *
     * Returns the AutoScalingGroup so you can add autoscaling settings to it.
     *
     * @deprecated Use {@link Cluster.addAsgCapacityProvider} instead.
     */
    addCapacity(id: string, options: AddCapacityOptions): autoscaling.AutoScalingGroup;
    /**
     * This method adds an Auto Scaling Group Capacity Provider to a cluster.
     *
     * @param provider the capacity provider to add to this cluster.
     * @stability stable
     */
    addAsgCapacityProvider(provider: AsgCapacityProvider, options?: AddAutoScalingGroupCapacityOptions): void;
    /**
     * (deprecated) This method adds compute capacity to a cluster using the specified AutoScalingGroup.
     *
     * @param autoScalingGroup the ASG to add to this cluster.
     * @deprecated Use {@link Cluster.addAsgCapacityProvider} instead.
     */
    addAutoScalingGroup(autoScalingGroup: autoscaling.AutoScalingGroup, options?: AddAutoScalingGroupCapacityOptions): void;
    private configureAutoScalingGroup;
    /**
     * (deprecated) This method enables the Fargate or Fargate Spot capacity providers on the cluster.
     *
     * @param provider the capacity provider to add to this cluster.
     * @see {@link addAsgCapacityProvider} to add an Auto Scaling Group capacity provider to the cluster.
     * @deprecated Use {@link enableFargateCapacityProviders} instead.
     */
    addCapacityProvider(provider: string): void;
    private configureWindowsAutoScalingGroup;
    /**
     * Getter for autoscaling group added to cluster.
     *
     * @stability stable
     */
    get autoscalingGroup(): autoscaling.IAutoScalingGroup | undefined;
    /**
     * Whether the cluster has EC2 capacity associated with it.
     *
     * @stability stable
     */
    get hasEc2Capacity(): boolean;
    /**
     * This method returns the CloudWatch metric for this clusters CPU reservation.
     *
     * @default average over 5 minutes
     * @stability stable
     */
    metricCpuReservation(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * This method returns the CloudWatch metric for this clusters CPU utilization.
     *
     * @default average over 5 minutes
     * @stability stable
     */
    metricCpuUtilization(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * This method returns the CloudWatch metric for this clusters memory reservation.
     *
     * @default average over 5 minutes
     * @stability stable
     */
    metricMemoryReservation(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * This method returns the CloudWatch metric for this clusters memory utilization.
     *
     * @default average over 5 minutes
     * @stability stable
     */
    metricMemoryUtilization(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * This method returns the specifed CloudWatch metric for this cluster.
     *
     * @stability stable
     */
    metric(metricName: string, props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    private cannedMetric;
}
/**
 * ECS-optimized Windows version list.
 *
 * @stability stable
 */
export declare enum WindowsOptimizedVersion {
    /**
     * @stability stable
     */
    SERVER_2019 = "2019",
    /**
     * @stability stable
     */
    SERVER_2016 = "2016"
}
/**
 * (deprecated) The properties that define which ECS-optimized AMI is used.
 *
 * @deprecated see {@link EcsOptimizedImage}
 */
export interface EcsOptimizedAmiProps {
    /**
     * (deprecated) The Amazon Linux generation to use.
     *
     * @default AmazonLinuxGeneration.AmazonLinux2
     * @deprecated
     */
    readonly generation?: ec2.AmazonLinuxGeneration;
    /**
     * (deprecated) The Windows Server version to use.
     *
     * @default none, uses Linux generation
     * @deprecated
     */
    readonly windowsVersion?: WindowsOptimizedVersion;
    /**
     * (deprecated) The ECS-optimized AMI variant to use.
     *
     * @default AmiHardwareType.Standard
     * @deprecated
     */
    readonly hardwareType?: AmiHardwareType;
}
/**
 * (deprecated) Construct a Linux or Windows machine image from the latest ECS Optimized AMI published in SSM.
 *
 * @deprecated see {@link EcsOptimizedImage#amazonLinux}, {@link EcsOptimizedImage#amazonLinux} and {@link EcsOptimizedImage#windows}
 */
export declare class EcsOptimizedAmi implements ec2.IMachineImage {
    private readonly generation?;
    private readonly windowsVersion?;
    private readonly hwType;
    private readonly amiParameterName;
    /**
     * (deprecated) Constructs a new instance of the EcsOptimizedAmi class.
     *
     * @deprecated
     */
    constructor(props?: EcsOptimizedAmiProps);
    /**
     * (deprecated) Return the correct image.
     *
     * @deprecated
     */
    getImage(scope: CoreConstruct): ec2.MachineImageConfig;
}
/**
 * Construct a Linux or Windows machine image from the latest ECS Optimized AMI published in SSM.
 *
 * @stability stable
 */
export declare class EcsOptimizedImage implements ec2.IMachineImage {
    /**
     * Construct an Amazon Linux 2 image from the latest ECS Optimized AMI published in SSM.
     *
     * @param hardwareType ECS-optimized AMI variant to use.
     * @stability stable
     */
    static amazonLinux2(hardwareType?: AmiHardwareType): EcsOptimizedImage;
    /**
     * Construct an Amazon Linux AMI image from the latest ECS Optimized AMI published in SSM.
     *
     * @stability stable
     */
    static amazonLinux(): EcsOptimizedImage;
    /**
     * Construct a Windows image from the latest ECS Optimized AMI published in SSM.
     *
     * @param windowsVersion Windows Version to use.
     * @stability stable
     */
    static windows(windowsVersion: WindowsOptimizedVersion): EcsOptimizedImage;
    private readonly generation?;
    private readonly windowsVersion?;
    private readonly hwType?;
    private readonly amiParameterName;
    /**
     * Constructs a new instance of the EcsOptimizedAmi class.
     */
    private constructor();
    /**
     * Return the correct image.
     *
     * @stability stable
     */
    getImage(scope: CoreConstruct): ec2.MachineImageConfig;
}
/**
 * Amazon ECS variant
 */
export declare enum BottlerocketEcsVariant {
    /**
     * Amazon ECS variant.
     *
     * @stability stable
     */
    AWS_ECS_1 = "aws-ecs-1"
}
/**
 * Properties for BottleRocketImage.
 *
 * @stability stable
 */
export interface BottleRocketImageProps {
    /**
     * The Amazon ECS variant to use.
     *
     * Only `aws-ecs-1` is currently available
     *
     * @default - BottlerocketEcsVariant.AWS_ECS_1
     * @stability stable
     */
    readonly variant?: BottlerocketEcsVariant;
}
/**
 * Construct an Bottlerocket image from the latest AMI published in SSM.
 *
 * @stability stable
 */
export declare class BottleRocketImage implements ec2.IMachineImage {
    private readonly amiParameterName;
    /**
     * Amazon ECS variant for Bottlerocket AMI
     */
    private readonly variant;
    /**
     * Constructs a new instance of the BottleRocketImage class.
     *
     * @stability stable
     */
    constructor(props?: BottleRocketImageProps);
    /**
     * Return the correct image.
     *
     * @stability stable
     */
    getImage(scope: CoreConstruct): ec2.MachineImageConfig;
}
/**
 * A regional grouping of one or more container instances on which you can run tasks and services.
 *
 * @stability stable
 */
export interface ICluster extends IResource {
    /**
     * The name of the cluster.
     *
     * @stability stable
     * @attribute true
     */
    readonly clusterName: string;
    /**
     * The Amazon Resource Name (ARN) that identifies the cluster.
     *
     * @stability stable
     * @attribute true
     */
    readonly clusterArn: string;
    /**
     * The VPC associated with the cluster.
     *
     * @stability stable
     */
    readonly vpc: ec2.IVpc;
    /**
     * Manage the allowed network connections for the cluster with Security Groups.
     *
     * @stability stable
     */
    readonly connections: ec2.Connections;
    /**
     * Specifies whether the cluster has EC2 instance capacity.
     *
     * @stability stable
     */
    readonly hasEc2Capacity: boolean;
    /**
     * The AWS Cloud Map namespace to associate with the cluster.
     *
     * @stability stable
     */
    readonly defaultCloudMapNamespace?: cloudmap.INamespace;
    /**
     * The autoscaling group added to the cluster if capacity is associated to the cluster.
     *
     * @stability stable
     */
    readonly autoscalingGroup?: autoscaling.IAutoScalingGroup;
}
/**
 * The properties to import from the ECS cluster.
 *
 * @stability stable
 */
export interface ClusterAttributes {
    /**
     * The name of the cluster.
     *
     * @stability stable
     */
    readonly clusterName: string;
    /**
     * The Amazon Resource Name (ARN) that identifies the cluster.
     *
     * @default Derived from clusterName
     * @stability stable
     */
    readonly clusterArn?: string;
    /**
     * The VPC associated with the cluster.
     *
     * @stability stable
     */
    readonly vpc: ec2.IVpc;
    /**
     * The security groups associated with the container instances registered to the cluster.
     *
     * @stability stable
     */
    readonly securityGroups: ec2.ISecurityGroup[];
    /**
     * Specifies whether the cluster has EC2 instance capacity.
     *
     * @default true
     * @stability stable
     */
    readonly hasEc2Capacity?: boolean;
    /**
     * The AWS Cloud Map namespace to associate with the cluster.
     *
     * @default - No default namespace
     * @stability stable
     */
    readonly defaultCloudMapNamespace?: cloudmap.INamespace;
    /**
     * Autoscaling group added to the cluster if capacity is added.
     *
     * @default - No default autoscaling group
     * @stability stable
     */
    readonly autoscalingGroup?: autoscaling.IAutoScalingGroup;
}
/**
 * The properties for adding an AutoScalingGroup.
 *
 * @stability stable
 */
export interface AddAutoScalingGroupCapacityOptions {
    /**
     * Specifies whether the containers can access the container instance role.
     *
     * @default false
     * @stability stable
     */
    readonly canContainersAccessInstanceRole?: boolean;
    /**
     * (deprecated) The time period to wait before force terminating an instance that is draining.
     *
     * This creates a Lambda function that is used by a lifecycle hook for the
     * AutoScalingGroup that will delay instance termination until all ECS tasks
     * have drained from the instance. Set to 0 to disable task draining.
     *
     * Set to 0 to disable task draining.
     *
     * @default Duration.minutes(5)
     * @deprecated The lifecycle draining hook is not configured if using the EC2 Capacity Provider. Enable managed termination protection instead.
     */
    readonly taskDrainTime?: Duration;
    /**
     * Specify whether to enable Automated Draining for Spot Instances running Amazon ECS Services.
     *
     * For more information, see [Using Spot Instances](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/container-instance-spot.html).
     *
     * @default false
     * @stability stable
     */
    readonly spotInstanceDraining?: boolean;
    /**
     * If {@link AddAutoScalingGroupCapacityOptions.taskDrainTime} is non-zero, then the ECS cluster creates an SNS Topic to as part of a system to drain instances of tasks when the instance is being shut down. If this property is provided, then this key will be used to encrypt the contents of that SNS Topic. See [SNS Data Encryption](https://docs.aws.amazon.com/sns/latest/dg/sns-data-encryption.html) for more information.
     *
     * @default The SNS Topic will not be encrypted.
     * @stability stable
     */
    readonly topicEncryptionKey?: kms.IKey;
    /**
     * Specify the machine image type.
     *
     * @default MachineImageType.AMAZON_LINUX_2
     * @stability stable
     */
    readonly machineImageType?: MachineImageType;
}
/**
 * The properties for adding instance capacity to an AutoScalingGroup.
 *
 * @stability stable
 */
export interface AddCapacityOptions extends AddAutoScalingGroupCapacityOptions, autoscaling.CommonAutoScalingGroupProps {
    /**
     * The EC2 instance type to use when launching instances into the AutoScalingGroup.
     *
     * @stability stable
     */
    readonly instanceType: ec2.InstanceType;
    /**
     * The ECS-optimized AMI variant to use.
     *
     * For more information, see
     * [Amazon ECS-optimized AMIs](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html).
     * You must define either `machineImage` or `machineImageType`, not both.
     *
     * @default - Amazon Linux 2
     * @stability stable
     */
    readonly machineImage?: ec2.IMachineImage;
}
/**
 * The options for creating an AWS Cloud Map namespace.
 *
 * @stability stable
 */
export interface CloudMapNamespaceOptions {
    /**
     * The name of the namespace, such as example.com.
     *
     * @stability stable
     */
    readonly name: string;
    /**
     * The type of CloudMap Namespace to create.
     *
     * @default PrivateDns
     * @stability stable
     */
    readonly type?: cloudmap.NamespaceType;
    /**
     * The VPC to associate the namespace with.
     *
     * This property is required for private DNS namespaces.
     *
     * @default VPC of the cluster for Private DNS Namespace, otherwise none
     * @stability stable
     */
    readonly vpc?: ec2.IVpc;
}
/**
 * The ECS-optimized AMI variant to use.
 *
 * For more information, see
 * [Amazon ECS-optimized AMIs](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html).
 *
 * @stability stable
 */
export declare enum AmiHardwareType {
    /**
     * Use the standard Amazon ECS-optimized AMI.
     *
     * @stability stable
     */
    STANDARD = "Standard",
    /**
     * Use the Amazon ECS GPU-optimized AMI.
     *
     * @stability stable
     */
    GPU = "GPU",
    /**
     * Use the Amazon ECS-optimized Amazon Linux 2 (arm64) AMI.
     *
     * @stability stable
     */
    ARM = "ARM64"
}
/**
 * A Capacity Provider strategy to use for the service.
 *
 * NOTE: defaultCapacityProviderStrategy on cluster not currently supported.
 *
 * @stability stable
 */
export interface CapacityProviderStrategy {
    /**
     * The name of the capacity provider.
     *
     * @stability stable
     */
    readonly capacityProvider: string;
    /**
     * The base value designates how many tasks, at a minimum, to run on the specified capacity provider.
     *
     * Only one
     * capacity provider in a capacity provider strategy can have a base defined. If no value is specified, the default
     * value of 0 is used.
     *
     * @default - none
     * @stability stable
     */
    readonly base?: number;
    /**
     * The weight value designates the relative percentage of the total number of tasks launched that should use the specified capacity provider.
     *
     * The weight value is taken into consideration after the base value, if defined, is satisfied.
     *
     * @default - 0
     * @stability stable
     */
    readonly weight?: number;
}
/**
 * The options for creating an Auto Scaling Group Capacity Provider.
 *
 * @stability stable
 */
export interface AsgCapacityProviderProps extends AddAutoScalingGroupCapacityOptions {
    /**
     * The name for the capacity provider.
     *
     * @default CloudFormation-generated name
     * @stability stable
     */
    readonly capacityProviderName?: string;
    /**
     * The autoscaling group to add as a Capacity Provider.
     *
     * @stability stable
     */
    readonly autoScalingGroup: autoscaling.IAutoScalingGroup;
    /**
     * Whether to enable managed scaling.
     *
     * @default true
     * @stability stable
     */
    readonly enableManagedScaling?: boolean;
    /**
     * Whether to enable managed termination protection.
     *
     * @default true
     * @stability stable
     */
    readonly enableManagedTerminationProtection?: boolean;
    /**
     * Maximum scaling step size.
     *
     * In most cases this should be left alone.
     *
     * @default 1000
     * @stability stable
     */
    readonly maximumScalingStepSize?: number;
    /**
     * Minimum scaling step size.
     *
     * In most cases this should be left alone.
     *
     * @default 1
     * @stability stable
     */
    readonly minimumScalingStepSize?: number;
    /**
     * Target capacity percent.
     *
     * In most cases this should be left alone.
     *
     * @default 100
     * @stability stable
     */
    readonly targetCapacityPercent?: number;
}
/**
 * An Auto Scaling Group Capacity Provider.
 *
 * This allows an ECS cluster to target
 * a specific EC2 Auto Scaling Group for the placement of tasks. Optionally (and
 * recommended), ECS can manage the number of instances in the ASG to fit the
 * tasks, and can ensure that instances are not prematurely terminated while
 * there are still tasks running on them.
 *
 * @stability stable
 */
export declare class AsgCapacityProvider extends CoreConstruct {
    /**
     * Capacity provider name.
     *
     * @default Chosen by CloudFormation
     * @stability stable
     */
    readonly capacityProviderName: string;
    /**
     * Auto Scaling Group.
     *
     * @stability stable
     */
    readonly autoScalingGroup: autoscaling.AutoScalingGroup;
    /**
     * Whether managed termination protection is enabled.
     *
     * @stability stable
     */
    readonly enableManagedTerminationProtection?: boolean;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: AsgCapacityProviderProps);
}
