import * as ecr from '@aws-cdk/aws-ecr';
import { ContainerDefinition } from './container-definition';
import { CfnTaskDefinition } from './ecs.generated';
import { Construct as CoreConstruct } from '@aws-cdk/core';
/**
 * Constructs for types of container images.
 *
 * @stability stable
 */
export declare abstract class ContainerImage {
    /**
     * Reference an image on DockerHub or another online registry.
     *
     * @stability stable
     */
    static fromRegistry(name: string, props?: RepositoryImageProps): RepositoryImage;
    /**
     * Reference an image in an ECR repository.
     *
     * @stability stable
     */
    static fromEcrRepository(repository: ecr.IRepository, tag?: string): EcrImage;
    /**
     * Reference an image that's constructed directly from sources on disk.
     *
     * If you already have a `DockerImageAsset` instance, you can use the
     * `ContainerImage.fromDockerImageAsset` method instead.
     *
     * @param directory The directory containing the Dockerfile.
     * @stability stable
     */
    static fromAsset(directory: string, props?: AssetImageProps): AssetImage;
    /**
     * Use an existing `DockerImageAsset` for this container image.
     *
     * @param asset The `DockerImageAsset` to use for this container definition.
     * @stability stable
     */
    static fromDockerImageAsset(asset: DockerImageAsset): ContainerImage;
    /**
     * Called when the image is used by a ContainerDefinition.
     *
     * @stability stable
     */
    abstract bind(scope: CoreConstruct, containerDefinition: ContainerDefinition): ContainerImageConfig;
}
/**
 * The configuration for creating a container image.
 *
 * @stability stable
 */
export interface ContainerImageConfig {
    /**
     * Specifies the name of the container image.
     *
     * @stability stable
     */
    readonly imageName: string;
    /**
     * Specifies the credentials used to access the image repository.
     *
     * @stability stable
     */
    readonly repositoryCredentials?: CfnTaskDefinition.RepositoryCredentialsProperty;
}
import { DockerImageAsset } from '@aws-cdk/aws-ecr-assets';
import { AssetImage, AssetImageProps } from './images/asset-image';
import { EcrImage } from './images/ecr';
import { RepositoryImage, RepositoryImageProps } from './images/repository';
