"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.InstanceDrainHook = void 0;
const fs = require("fs");
const path = require("path");
const autoscaling = require("@aws-cdk/aws-autoscaling");
const hooks = require("@aws-cdk/aws-autoscaling-hooktargets");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_1 = require("@aws-cdk/core");
/**
 * A hook to drain instances from ECS traffic before they're terminated
 */
class InstanceDrainHook extends core_1.Construct {
    /**
     * Constructs a new instance of the InstanceDrainHook class.
     */
    constructor(scope, id, props) {
        super(scope, id);
        const drainTime = props.drainTime || cdk.Duration.minutes(5);
        // Invoke Lambda via SNS Topic
        const fn = new lambda.Function(this, 'Function', {
            code: lambda.Code.fromInline(fs.readFileSync(path.join(__dirname, 'lambda-source', 'index.py'), { encoding: 'utf-8' })),
            handler: 'index.lambda_handler',
            runtime: lambda.Runtime.PYTHON_3_6,
            // Timeout: some extra margin for additional API calls made by the Lambda,
            // up to a maximum of 15 minutes.
            timeout: cdk.Duration.seconds(Math.min(drainTime.toSeconds() + 10, 900)),
            environment: {
                CLUSTER: props.cluster.clusterName,
            },
        });
        // Hook everything up: ASG -> Topic, Topic -> Lambda
        props.autoScalingGroup.addLifecycleHook('DrainHook', {
            lifecycleTransition: autoscaling.LifecycleTransition.INSTANCE_TERMINATING,
            defaultResult: autoscaling.DefaultResult.CONTINUE,
            notificationTarget: new hooks.FunctionHook(fn, props.topicEncryptionKey),
            heartbeatTimeout: drainTime,
        });
        // Describe actions cannot be restricted and restrict the CompleteLifecycleAction to the ASG arn
        // https://docs.aws.amazon.com/autoscaling/ec2/userguide/control-access-using-iam.html
        fn.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'ec2:DescribeInstances',
                'ec2:DescribeInstanceAttribute',
                'ec2:DescribeInstanceStatus',
                'ec2:DescribeHosts',
            ],
            resources: ['*'],
        }));
        // Restrict to the ASG
        fn.addToRolePolicy(new iam.PolicyStatement({
            actions: ['autoscaling:CompleteLifecycleAction'],
            resources: [props.autoScalingGroup.autoScalingGroupArn],
        }));
        fn.addToRolePolicy(new iam.PolicyStatement({
            actions: ['ecs:DescribeContainerInstances', 'ecs:DescribeTasks'],
            resources: ['*'],
            conditions: {
                ArnEquals: { 'ecs:cluster': props.cluster.clusterArn },
            },
        }));
        // Restrict to the ECS Cluster
        fn.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'ecs:ListContainerInstances',
                'ecs:SubmitContainerStateChange',
                'ecs:SubmitTaskStateChange',
            ],
            resources: [props.cluster.clusterArn],
        }));
        // Restrict the container-instance operations to the ECS Cluster
        fn.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'ecs:UpdateContainerInstancesState',
                'ecs:ListTasks',
            ],
            conditions: {
                ArnEquals: { 'ecs:cluster': props.cluster.clusterArn },
            },
            resources: ['*'],
        }));
    }
}
exports.InstanceDrainHook = InstanceDrainHook;
//# sourceMappingURL=data:application/json;base64,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