import * as ec2 from '@aws-cdk/aws-ec2';
import { Construct } from 'constructs';
import { BaseService, BaseServiceOptions, IBaseService, IService, PropagatedTagSource } from '../base/base-service';
import { TaskDefinition } from '../base/task-definition';
import { ICluster } from '../cluster';
import { PlacementConstraint, PlacementStrategy } from '../placement';
/**
 * The properties for defining a service using the EC2 launch type.
 *
 * @stability stable
 */
export interface Ec2ServiceProps extends BaseServiceOptions {
    /**
     * The task definition to use for tasks in the service.
     *
     * [disable-awslint:ref-via-interface]
     *
     * @stability stable
     */
    readonly taskDefinition: TaskDefinition;
    /**
     * Specifies whether the task's elastic network interface receives a public IP address.
     *
     * If true, each task will receive a public IP address.
     *
     * This property is only used for tasks that use the awsvpc network mode.
     *
     * @default false
     * @stability stable
     */
    readonly assignPublicIp?: boolean;
    /**
     * The subnets to associate with the service.
     *
     * This property is only used for tasks that use the awsvpc network mode.
     *
     * @default - Public subnets if `assignPublicIp` is set, otherwise the first available one of Private, Isolated, Public, in that order.
     * @stability stable
     */
    readonly vpcSubnets?: ec2.SubnetSelection;
    /**
     * (deprecated) The security groups to associate with the service.
     *
     * If you do not specify a security group, the default security group for the VPC is used.
     *
     * This property is only used for tasks that use the awsvpc network mode.
     *
     * @default - A new security group is created.
     * @deprecated use securityGroups instead.
     */
    readonly securityGroup?: ec2.ISecurityGroup;
    /**
     * The security groups to associate with the service.
     *
     * If you do not specify a security group, the default security group for the VPC is used.
     *
     * This property is only used for tasks that use the awsvpc network mode.
     *
     * @default - A new security group is created.
     * @stability stable
     */
    readonly securityGroups?: ec2.ISecurityGroup[];
    /**
     * The placement constraints to use for tasks in the service.
     *
     * For more information, see
     * [Amazon ECS Task Placement Constraints](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-placement-constraints.html).
     *
     * @default - No constraints.
     * @stability stable
     */
    readonly placementConstraints?: PlacementConstraint[];
    /**
     * The placement strategies to use for tasks in the service.
     *
     * For more information, see
     * [Amazon ECS Task Placement Strategies](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-placement-strategies.html).
     *
     * @default - No strategies.
     * @stability stable
     */
    readonly placementStrategies?: PlacementStrategy[];
    /**
     * Specifies whether the service will use the daemon scheduling strategy.
     *
     * If true, the service scheduler deploys exactly one task on each container instance in your cluster.
     *
     * When you are using this strategy, do not specify a desired number of tasks orany task placement strategies.
     *
     * @default false
     * @stability stable
     */
    readonly daemon?: boolean;
    /**
     * (deprecated) Specifies whether to propagate the tags from the task definition or the service to the tasks in the service.
     *
     * Tags can only be propagated to the tasks within the service during service creation.
     *
     * @default PropagatedTagSource.NONE
     * @deprecated Use `propagateTags` instead.
     */
    readonly propagateTaskTagsFrom?: PropagatedTagSource;
}
/**
 * The interface for a service using the EC2 launch type on an ECS cluster.
 *
 * @stability stable
 */
export interface IEc2Service extends IService {
}
/**
 * The properties to import from the service using the EC2 launch type.
 *
 * @stability stable
 */
export interface Ec2ServiceAttributes {
    /**
     * The cluster that hosts the service.
     *
     * @stability stable
     */
    readonly cluster: ICluster;
    /**
     * The service ARN.
     *
     * @default - either this, or {@link serviceName}, is required
     * @stability stable
     */
    readonly serviceArn?: string;
    /**
     * The name of the service.
     *
     * @default - either this, or {@link serviceArn}, is required
     * @stability stable
     */
    readonly serviceName?: string;
}
/**
 * This creates a service using the EC2 launch type on an ECS cluster.
 *
 * @stability stable
 * @resource AWS::ECS::Service
 */
export declare class Ec2Service extends BaseService implements IEc2Service {
    /**
     * Imports from the specified service ARN.
     *
     * @stability stable
     */
    static fromEc2ServiceArn(scope: Construct, id: string, ec2ServiceArn: string): IEc2Service;
    /**
     * Imports from the specified service attrributes.
     *
     * @stability stable
     */
    static fromEc2ServiceAttributes(scope: Construct, id: string, attrs: Ec2ServiceAttributes): IBaseService;
    private readonly constraints;
    private readonly strategies;
    private readonly daemon;
    /**
     * Constructs a new instance of the Ec2Service class.
     *
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: Ec2ServiceProps);
    /**
     * Adds one or more placement strategies to use for tasks in the service.
     *
     * For more information, see
     * [Amazon ECS Task Placement Strategies](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-placement-strategies.html).
     *
     * @stability stable
     */
    addPlacementStrategies(...strategies: PlacementStrategy[]): void;
    /**
     * Adds one or more placement constraints to use for tasks in the service.
     *
     * For more information, see
     * [Amazon ECS Task Placement Constraints](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-placement-constraints.html).
     *
     * @stability stable
     */
    addPlacementConstraints(...constraints: PlacementConstraint[]): void;
    /**
     * Validates this Ec2Service.
     *
     * @stability stable
     */
    protected validate(): string[];
}
/**
 * The built-in container instance attributes.
 *
 * @stability stable
 */
export declare class BuiltInAttributes {
    /**
     * The id of the instance.
     *
     * @stability stable
     */
    static readonly INSTANCE_ID = "instanceId";
    /**
     * The AvailabilityZone where the instance is running in.
     *
     * @stability stable
     */
    static readonly AVAILABILITY_ZONE = "attribute:ecs.availability-zone";
    /**
     * The AMI id the instance is using.
     *
     * @stability stable
     */
    static readonly AMI_ID = "attribute:ecs.ami-id";
    /**
     * The EC2 instance type.
     *
     * @stability stable
     */
    static readonly INSTANCE_TYPE = "attribute:ecs.instance-type";
    /**
     * The operating system of the instance.
     *
     * Either 'linux' or 'windows'.
     *
     * @stability stable
     */
    static readonly OS_TYPE = "attribute:ecs.os-type";
}
