import { IBucket, Location } from '@aws-cdk/aws-s3';
import { AssetOptions } from '@aws-cdk/aws-s3-assets';
import { Construct } from '@aws-cdk/core';
/**
 * Constructs for types of environment files.
 *
 * @stability stable
 */
export declare abstract class EnvironmentFile {
    /**
     * Loads the environment file from a local disk path.
     *
     * @param path Local disk path.
     * @stability stable
     */
    static fromAsset(path: string, options?: AssetOptions): AssetEnvironmentFile;
    /**
     * Loads the environment file from an S3 bucket.
     *
     * @param bucket The S3 bucket.
     * @param key The object key.
     * @param objectVersion Optional S3 object version.
     * @returns `S3EnvironmentFile` associated with the specified S3 object.
     * @stability stable
     */
    static fromBucket(bucket: IBucket, key: string, objectVersion?: string): S3EnvironmentFile;
    /**
     * Called when the container is initialized to allow this object to bind to the stack.
     *
     * @param scope The binding scope.
     * @stability stable
     */
    abstract bind(scope: Construct): EnvironmentFileConfig;
}
/**
 * Environment file from a local directory.
 *
 * @stability stable
 */
export declare class AssetEnvironmentFile extends EnvironmentFile {
    readonly path: string;
    private readonly options;
    private asset?;
    /**
     * @param path The path to the asset file or directory.
     * @stability stable
     */
    constructor(path: string, options?: AssetOptions);
    /**
     * Called when the container is initialized to allow this object to bind to the stack.
     *
     * @stability stable
     */
    bind(scope: Construct): EnvironmentFileConfig;
}
/**
 * Environment file from S3.
 *
 * @stability stable
 */
export declare class S3EnvironmentFile extends EnvironmentFile {
    private key;
    private objectVersion?;
    private readonly bucketName;
    /**
     * @stability stable
     */
    constructor(bucket: IBucket, key: string, objectVersion?: string | undefined);
    /**
     * Called when the container is initialized to allow this object to bind to the stack.
     *
     * @stability stable
     */
    bind(_scope: Construct): EnvironmentFileConfig;
}
/**
 * Configuration for the environment file.
 *
 * @stability stable
 */
export interface EnvironmentFileConfig {
    /**
     * The type of environment file.
     *
     * @stability stable
     */
    readonly fileType: EnvironmentFileType;
    /**
     * The location of the environment file in S3.
     *
     * @stability stable
     */
    readonly s3Location: Location;
}
/**
 * Type of environment file to be included in the container definition
 */
export declare enum EnvironmentFileType {
    /**
     * Type of environment file to be included in the container definition.
     *
     * @stability stable
     */
    S3 = "s3"
}
