import * as ec2 from '@aws-cdk/aws-ec2';
import { Construct } from 'constructs';
import { BaseService, BaseServiceOptions, IBaseService, IService, PropagatedTagSource } from '../base/base-service';
import { TaskDefinition } from '../base/task-definition';
import { ICluster } from '../cluster';
/**
 * The properties for defining a service using the Fargate launch type.
 *
 * @stability stable
 */
export interface FargateServiceProps extends BaseServiceOptions {
    /**
     * The task definition to use for tasks in the service.
     *
     * [disable-awslint:ref-via-interface]
     *
     * @stability stable
     */
    readonly taskDefinition: TaskDefinition;
    /**
     * Specifies whether the task's elastic network interface receives a public IP address.
     *
     * If true, each task will receive a public IP address.
     *
     * @default false
     * @stability stable
     */
    readonly assignPublicIp?: boolean;
    /**
     * The subnets to associate with the service.
     *
     * @default - Public subnets if `assignPublicIp` is set, otherwise the first available one of Private, Isolated, Public, in that order.
     * @stability stable
     */
    readonly vpcSubnets?: ec2.SubnetSelection;
    /**
     * (deprecated) The security groups to associate with the service.
     *
     * If you do not specify a security group, the default security group for the VPC is used.
     *
     * @default - A new security group is created.
     * @deprecated use securityGroups instead.
     */
    readonly securityGroup?: ec2.ISecurityGroup;
    /**
     * The security groups to associate with the service.
     *
     * If you do not specify a security group, the default security group for the VPC is used.
     *
     * @default - A new security group is created.
     * @stability stable
     */
    readonly securityGroups?: ec2.ISecurityGroup[];
    /**
     * The platform version on which to run your service.
     *
     * If one is not specified, the LATEST platform version is used by default. For more information, see
     * [AWS Fargate Platform Versions](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/platform_versions.html)
     * in the Amazon Elastic Container Service Developer Guide.
     *
     * @default Latest
     * @stability stable
     */
    readonly platformVersion?: FargatePlatformVersion;
    /**
     * (deprecated) Specifies whether to propagate the tags from the task definition or the service to the tasks in the service.
     *
     * Tags can only be propagated to the tasks within the service during service creation.
     *
     * @default PropagatedTagSource.NONE
     * @deprecated Use `propagateTags` instead.
     */
    readonly propagateTaskTagsFrom?: PropagatedTagSource;
}
/**
 * The interface for a service using the Fargate launch type on an ECS cluster.
 *
 * @stability stable
 */
export interface IFargateService extends IService {
}
/**
 * The properties to import from the service using the Fargate launch type.
 *
 * @stability stable
 */
export interface FargateServiceAttributes {
    /**
     * The cluster that hosts the service.
     *
     * @stability stable
     */
    readonly cluster: ICluster;
    /**
     * The service ARN.
     *
     * @default - either this, or {@link serviceName}, is required
     * @stability stable
     */
    readonly serviceArn?: string;
    /**
     * The name of the service.
     *
     * @default - either this, or {@link serviceArn}, is required
     * @stability stable
     */
    readonly serviceName?: string;
}
/**
 * This creates a service using the Fargate launch type on an ECS cluster.
 *
 * @stability stable
 * @resource AWS::ECS::Service
 */
export declare class FargateService extends BaseService implements IFargateService {
    /**
     * Imports from the specified service ARN.
     *
     * @stability stable
     */
    static fromFargateServiceArn(scope: Construct, id: string, fargateServiceArn: string): IFargateService;
    /**
     * Imports from the specified service attrributes.
     *
     * @stability stable
     */
    static fromFargateServiceAttributes(scope: Construct, id: string, attrs: FargateServiceAttributes): IBaseService;
    /**
     * Constructs a new instance of the FargateService class.
     *
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: FargateServiceProps);
}
/**
 * The platform version on which to run your service.
 *
 * @see https://docs.aws.amazon.com/AmazonECS/latest/developerguide/platform_versions.html
 * @stability stable
 */
export declare enum FargatePlatformVersion {
    /**
     * The latest, recommended platform version.
     *
     * @stability stable
     */
    LATEST = "LATEST",
    /**
     * Version 1.4.0.
     *
     * Supports EFS endpoints, CAP_SYS_PTRACE Linux capability,
     * network performance metrics in CloudWatch Container Insights,
     * consolidated 20 GB ephemeral volume.
     *
     * @stability stable
     */
    VERSION1_4 = "1.4.0",
    /**
     * Version 1.3.0.
     *
     * Supports secrets, task recycling.
     *
     * @stability stable
     */
    VERSION1_3 = "1.3.0",
    /**
     * Version 1.2.0.
     *
     * Supports private registries.
     *
     * @stability stable
     */
    VERSION1_2 = "1.2.0",
    /**
     * Version 1.1.0.
     *
     * Supports task metadata, health checks, service discovery.
     *
     * @stability stable
     */
    VERSION1_1 = "1.1.0",
    /**
     * Initial release.
     *
     * Based on Amazon Linux 2017.09.
     *
     * @stability stable
     */
    VERSION1_0 = "1.0.0"
}
