"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.FirelensLogRouter = exports.obtainDefaultFluentBitECRImage = exports.FirelensConfigFileType = exports.FirelensLogRouterType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("@aws-cdk/aws-iam");
const ssm = require("@aws-cdk/aws-ssm");
const cdk = require("@aws-cdk/core");
const container_definition_1 = require("./container-definition");
const container_image_1 = require("./container-image");
/**
 * Firelens log router type, fluentbit or fluentd.
 *
 * https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html
 *
 * @stability stable
 */
var FirelensLogRouterType;
(function (FirelensLogRouterType) {
    FirelensLogRouterType["FLUENTBIT"] = "fluentbit";
    FirelensLogRouterType["FLUENTD"] = "fluentd";
})(FirelensLogRouterType = exports.FirelensLogRouterType || (exports.FirelensLogRouterType = {}));
/**
 * Firelens configuration file type, s3 or file path.
 *
 * https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html#firelens-taskdef-customconfig
 *
 * @stability stable
 */
var FirelensConfigFileType;
(function (FirelensConfigFileType) {
    FirelensConfigFileType["S3"] = "s3";
    FirelensConfigFileType["FILE"] = "file";
})(FirelensConfigFileType = exports.FirelensConfigFileType || (exports.FirelensConfigFileType = {}));
/**
 * Render to CfnTaskDefinition.FirelensConfigurationProperty from FirelensConfig
 */
function renderFirelensConfig(firelensConfig) {
    if (!firelensConfig.options) {
        return { type: firelensConfig.type };
    }
    else {
        // firelensConfig.options.configFileType has been filled with s3 or file type in constructor.
        return {
            type: firelensConfig.type,
            options: {
                'enable-ecs-log-metadata': firelensConfig.options.enableECSLogMetadata ? 'true' : 'false',
                'config-file-type': firelensConfig.options.configFileType,
                'config-file-value': firelensConfig.options.configFileValue,
            },
        };
    }
}
/**
 * SSM parameters for latest fluent bit docker image in ECR
 * https://github.com/aws/aws-for-fluent-bit#using-ssm-to-find-available-versions
 */
const fluentBitImageSSMPath = '/aws/service/aws-for-fluent-bit';
/**
 * Obtain Fluent Bit image in Amazon ECR and setup corresponding IAM permissions.
 * ECR image pull permissions will be granted in task execution role.
 * Cloudwatch logs, Kinesis data stream or firehose permissions will be grant by check options in logDriverConfig.
 * https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html#firelens-using-fluentbit
 */
function obtainDefaultFluentBitECRImage(task, logDriverConfig, imageTag) {
    // grant ECR image pull permissions to executor role
    task.addToExecutionRolePolicy(new iam.PolicyStatement({
        actions: [
            'ecr:GetAuthorizationToken',
            'ecr:BatchCheckLayerAvailability',
            'ecr:GetDownloadUrlForLayer',
            'ecr:BatchGetImage',
        ],
        resources: ['*'],
    }));
    // grant cloudwatch or firehose permissions to task role
    const logName = logDriverConfig && logDriverConfig.logDriver === 'awsfirelens'
        && logDriverConfig.options && logDriverConfig.options.Name;
    if (logName === 'cloudwatch') {
        task.addToTaskRolePolicy(new iam.PolicyStatement({
            actions: [
                'logs:CreateLogGroup',
                'logs:CreateLogStream',
                'logs:DescribeLogStreams',
                'logs:PutLogEvents',
            ],
            resources: ['*'],
        }));
    }
    else if (logName === 'firehose') {
        task.addToTaskRolePolicy(new iam.PolicyStatement({
            actions: [
                'firehose:PutRecordBatch',
            ],
            resources: ['*'],
        }));
    }
    else if (logName === 'kinesis') {
        task.addToTaskRolePolicy(new iam.PolicyStatement({
            actions: [
                'kinesis:PutRecords',
            ],
            resources: ['*'],
        }));
    }
    const fluentBitImageTag = imageTag || 'latest';
    const fluentBitImage = `${fluentBitImageSSMPath}/${fluentBitImageTag}`;
    // Not use ContainerImage.fromEcrRepository since it's not support parsing ECR repo URI,
    // use repo ARN might result in complex Fn:: functions in cloudformation template.
    return container_image_1.ContainerImage.fromRegistry(ssm.StringParameter.valueForStringParameter(task, fluentBitImage));
}
exports.obtainDefaultFluentBitECRImage = obtainDefaultFluentBitECRImage;
/**
 * Firelens log router.
 *
 * @stability stable
 */
class FirelensLogRouter extends container_definition_1.ContainerDefinition {
    /**
     * Constructs a new instance of the FirelensLogRouter class.
     *
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id, props);
        const options = props.firelensConfig.options;
        if (options) {
            const enableECSLogMetadata = options.enableECSLogMetadata || options.enableECSLogMetadata === undefined;
            const configFileType = (options.configFileType === undefined || options.configFileType === FirelensConfigFileType.S3) &&
                (cdk.Token.isUnresolved(options.configFileValue) || /arn:aws[a-zA-Z-]*:s3:::.+/.test(options.configFileValue))
                ? FirelensConfigFileType.S3 : FirelensConfigFileType.FILE;
            this.firelensConfig = {
                type: props.firelensConfig.type,
                options: {
                    enableECSLogMetadata,
                    configFileType,
                    configFileValue: options.configFileValue,
                },
            };
            // grant s3 access permissions
            if (configFileType === FirelensConfigFileType.S3) {
                props.taskDefinition.addToExecutionRolePolicy(new iam.PolicyStatement({
                    actions: [
                        's3:GetObject',
                    ],
                    resources: [options.configFileValue],
                }));
                props.taskDefinition.addToExecutionRolePolicy(new iam.PolicyStatement({
                    actions: [
                        's3:GetBucketLocation',
                    ],
                    resources: [options.configFileValue.split('/')[0]],
                }));
            }
        }
        else {
            this.firelensConfig = props.firelensConfig;
        }
    }
    /**
     * Render this container definition to a CloudFormation object.
     *
     * @stability stable
     */
    renderContainerDefinition(_taskDefinition) {
        return {
            ...(super.renderContainerDefinition()),
            firelensConfiguration: this.firelensConfig && renderFirelensConfig(this.firelensConfig),
        };
    }
}
exports.FirelensLogRouter = FirelensLogRouter;
_a = JSII_RTTI_SYMBOL_1;
FirelensLogRouter[_a] = { fqn: "@aws-cdk/aws-ecs.FirelensLogRouter", version: "1.106.0" };
//# sourceMappingURL=data:application/json;base64,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