import { Construct } from 'constructs';
import { CfnTaskDefinition } from './ecs.generated';
import { Construct as CoreConstruct } from '@aws-cdk/core';
/**
 * The properties for defining Linux-specific options that are applied to the container.
 *
 * @stability stable
 */
export interface LinuxParametersProps {
    /**
     * Specifies whether to run an init process inside the container that forwards signals and reaps processes.
     *
     * @default false
     * @stability stable
     */
    readonly initProcessEnabled?: boolean;
    /**
     * The value for the size (in MiB) of the /dev/shm volume.
     *
     * @default No shared memory.
     * @stability stable
     */
    readonly sharedMemorySize?: number;
}
/**
 * Linux-specific options that are applied to the container.
 *
 * @stability stable
 */
export declare class LinuxParameters extends CoreConstruct {
    /**
     * Whether the init process is enabled
     */
    private readonly initProcessEnabled?;
    /**
     * The shared memory size. Not valid for Fargate launch type
     */
    private readonly sharedMemorySize?;
    /**
     * Capabilities to be added
     */
    private readonly capAdd;
    /**
     * Capabilities to be dropped
     */
    private readonly capDrop;
    /**
     * Device mounts
     */
    private readonly devices;
    /**
     * TmpFs mounts
     */
    private readonly tmpfs;
    /**
     * Constructs a new instance of the LinuxParameters class.
     *
     * @stability stable
     */
    constructor(scope: Construct, id: string, props?: LinuxParametersProps);
    /**
     * Adds one or more Linux capabilities to the Docker configuration of a container.
     *
     * Only works with EC2 launch type.
     *
     * @stability stable
     */
    addCapabilities(...cap: Capability[]): void;
    /**
     * Removes one or more Linux capabilities to the Docker configuration of a container.
     *
     * Only works with EC2 launch type.
     *
     * @stability stable
     */
    dropCapabilities(...cap: Capability[]): void;
    /**
     * Adds one or more host devices to a container.
     *
     * @stability stable
     */
    addDevices(...device: Device[]): void;
    /**
     * Specifies the container path, mount options, and size (in MiB) of the tmpfs mount for a container.
     *
     * Only works with EC2 launch type.
     *
     * @stability stable
     */
    addTmpfs(...tmpfs: Tmpfs[]): void;
    /**
     * Renders the Linux parameters to a CloudFormation object.
     *
     * @stability stable
     */
    renderLinuxParameters(): CfnTaskDefinition.LinuxParametersProperty;
}
/**
 * A container instance host device.
 *
 * @stability stable
 */
export interface Device {
    /**
     * The path inside the container at which to expose the host device.
     *
     * @default Same path as the host
     * @stability stable
     */
    readonly containerPath?: string;
    /**
     * The path for the device on the host container instance.
     *
     * @stability stable
     */
    readonly hostPath: string;
    /**
     * The explicit permissions to provide to the container for the device.
     *
     * By default, the container has permissions for read, write, and mknod for the device.
     *
     * @default Readonly
     * @stability stable
     */
    readonly permissions?: DevicePermission[];
}
/**
 * The details of a tmpfs mount for a container.
 *
 * @stability stable
 */
export interface Tmpfs {
    /**
     * The absolute file path where the tmpfs volume is to be mounted.
     *
     * @stability stable
     */
    readonly containerPath: string;
    /**
     * The size (in MiB) of the tmpfs volume.
     *
     * @stability stable
     */
    readonly size: number;
    /**
     * The list of tmpfs volume mount options.
     *
     * For more information, see
     * [TmpfsMountOptions](https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_Tmpfs.html).
     *
     * @stability stable
     */
    readonly mountOptions?: TmpfsMountOption[];
}
/**
 * A Linux capability.
 *
 * @stability stable
 */
export declare enum Capability {
    /**
     * @stability stable
     */
    ALL = "ALL",
    /**
     * @stability stable
     */
    AUDIT_CONTROL = "AUDIT_CONTROL",
    /**
     * @stability stable
     */
    AUDIT_WRITE = "AUDIT_WRITE",
    /**
     * @stability stable
     */
    BLOCK_SUSPEND = "BLOCK_SUSPEND",
    /**
     * @stability stable
     */
    CHOWN = "CHOWN",
    /**
     * @stability stable
     */
    DAC_OVERRIDE = "DAC_OVERRIDE",
    /**
     * @stability stable
     */
    DAC_READ_SEARCH = "DAC_READ_SEARCH",
    /**
     * @stability stable
     */
    FOWNER = "FOWNER",
    /**
     * @stability stable
     */
    FSETID = "FSETID",
    /**
     * @stability stable
     */
    IPC_LOCK = "IPC_LOCK",
    /**
     * @stability stable
     */
    IPC_OWNER = "IPC_OWNER",
    /**
     * @stability stable
     */
    KILL = "KILL",
    /**
     * @stability stable
     */
    LEASE = "LEASE",
    /**
     * @stability stable
     */
    LINUX_IMMUTABLE = "LINUX_IMMUTABLE",
    /**
     * @stability stable
     */
    MAC_ADMIN = "MAC_ADMIN",
    /**
     * @stability stable
     */
    MAC_OVERRIDE = "MAC_OVERRIDE",
    /**
     * @stability stable
     */
    MKNOD = "MKNOD",
    /**
     * @stability stable
     */
    NET_ADMIN = "NET_ADMIN",
    /**
     * @stability stable
     */
    NET_BIND_SERVICE = "NET_BIND_SERVICE",
    /**
     * @stability stable
     */
    NET_BROADCAST = "NET_BROADCAST",
    /**
     * @stability stable
     */
    NET_RAW = "NET_RAW",
    /**
     * @stability stable
     */
    SETFCAP = "SETFCAP",
    /**
     * @stability stable
     */
    SETGID = "SETGID",
    /**
     * @stability stable
     */
    SETPCAP = "SETPCAP",
    /**
     * @stability stable
     */
    SETUID = "SETUID",
    /**
     * @stability stable
     */
    SYS_ADMIN = "SYS_ADMIN",
    /**
     * @stability stable
     */
    SYS_BOOT = "SYS_BOOT",
    /**
     * @stability stable
     */
    SYS_CHROOT = "SYS_CHROOT",
    /**
     * @stability stable
     */
    SYS_MODULE = "SYS_MODULE",
    /**
     * @stability stable
     */
    SYS_NICE = "SYS_NICE",
    /**
     * @stability stable
     */
    SYS_PACCT = "SYS_PACCT",
    /**
     * @stability stable
     */
    SYS_PTRACE = "SYS_PTRACE",
    /**
     * @stability stable
     */
    SYS_RAWIO = "SYS_RAWIO",
    /**
     * @stability stable
     */
    SYS_RESOURCE = "SYS_RESOURCE",
    /**
     * @stability stable
     */
    SYS_TIME = "SYS_TIME",
    /**
     * @stability stable
     */
    SYS_TTY_CONFIG = "SYS_TTY_CONFIG",
    /**
     * @stability stable
     */
    SYSLOG = "SYSLOG",
    /**
     * @stability stable
     */
    WAKE_ALARM = "WAKE_ALARM"
}
/**
 * Permissions for device access.
 *
 * @stability stable
 */
export declare enum DevicePermission {
    /**
     * Read.
     *
     * @stability stable
     */
    READ = "read",
    /**
     * Write.
     *
     * @stability stable
     */
    WRITE = "write",
    /**
     * Make a node.
     *
     * @stability stable
     */
    MKNOD = "mknod"
}
/**
 * The supported options for a tmpfs mount for a container.
 *
 * @stability stable
 */
export declare enum TmpfsMountOption {
    /**
     * @stability stable
     */
    DEFAULTS = "defaults",
    /**
     * @stability stable
     */
    RO = "ro",
    /**
     * @stability stable
     */
    RW = "rw",
    /**
     * @stability stable
     */
    SUID = "suid",
    /**
     * @stability stable
     */
    NOSUID = "nosuid",
    /**
     * @stability stable
     */
    DEV = "dev",
    /**
     * @stability stable
     */
    NODEV = "nodev",
    /**
     * @stability stable
     */
    EXEC = "exec",
    /**
     * @stability stable
     */
    NOEXEC = "noexec",
    /**
     * @stability stable
     */
    SYNC = "sync",
    /**
     * @stability stable
     */
    ASYNC = "async",
    /**
     * @stability stable
     */
    DIRSYNC = "dirsync",
    /**
     * @stability stable
     */
    REMOUNT = "remount",
    /**
     * @stability stable
     */
    MAND = "mand",
    /**
     * @stability stable
     */
    NOMAND = "nomand",
    /**
     * @stability stable
     */
    ATIME = "atime",
    /**
     * @stability stable
     */
    NOATIME = "noatime",
    /**
     * @stability stable
     */
    DIRATIME = "diratime",
    /**
     * @stability stable
     */
    NODIRATIME = "nodiratime",
    /**
     * @stability stable
     */
    BIND = "bind",
    /**
     * @stability stable
     */
    RBIND = "rbind",
    /**
     * @stability stable
     */
    UNBINDABLE = "unbindable",
    /**
     * @stability stable
     */
    RUNBINDABLE = "runbindable",
    /**
     * @stability stable
     */
    PRIVATE = "private",
    /**
     * @stability stable
     */
    RPRIVATE = "rprivate",
    /**
     * @stability stable
     */
    SHARED = "shared",
    /**
     * @stability stable
     */
    RSHARED = "rshared",
    /**
     * @stability stable
     */
    SLAVE = "slave",
    /**
     * @stability stable
     */
    RSLAVE = "rslave",
    /**
     * @stability stable
     */
    RELATIME = "relatime",
    /**
     * @stability stable
     */
    NORELATIME = "norelatime",
    /**
     * @stability stable
     */
    STRICTATIME = "strictatime",
    /**
     * @stability stable
     */
    NOSTRICTATIME = "nostrictatime",
    /**
     * @stability stable
     */
    MODE = "mode",
    /**
     * @stability stable
     */
    UID = "uid",
    /**
     * @stability stable
     */
    GID = "gid",
    /**
     * @stability stable
     */
    NR_INODES = "nr_inodes",
    /**
     * @stability stable
     */
    NR_BLOCKS = "nr_blocks",
    /**
     * @stability stable
     */
    MPOL = "mpol"
}
