"""Infrastructure for expressing validation rules against GeneralManager instances."""

from __future__ import annotations
import ast
import inspect
import re
import textwrap
from typing import Callable, Dict, Generic, List, Optional, Tuple, TypeVar
from decimal import Decimal

from django.conf import settings
from django.utils.module_loading import import_string

from general_manager.rule.handler import (
    BaseRuleHandler,
    LenHandler,
    MaxHandler,
    MinHandler,
    SumHandler,
)
from general_manager.manager.generalManager import GeneralManager

GeneralManagerType = TypeVar("GeneralManagerType", bound=GeneralManager)

NOTEXISTENT = object()


class NonexistentAttributeError(AttributeError):
    """Raised when a referenced attribute does not exist on the GeneralManager instance."""

    def __init__(self, attribute: str) -> None:
        """
        Initialize the exception indicating that a referenced attribute does not exist.

        Parameters:
            attribute (str): The name of the nonexistent attribute; this name will be included in the exception message.
        """
        super().__init__(f"The attribute '{attribute}' does not exist.")


class MissingErrorTemplateVariableError(ValueError):
    """Raised when a custom error template omits required variables."""

    def __init__(self, missing: List[str]) -> None:
        """
        Initialize the exception indicating that one or more variables referenced by a rule are missing from a custom error template.

        Parameters:
            missing (List[str]): Names of the variables that are required by the template but were not found; these names will be included in the exception message.
        """
        super().__init__(
            f"The custom error message does not contain all used variables: {missing}."
        )


class ErrorMessageGenerationError(ValueError):
    """Raised when generating an error message before evaluating any input."""

    def __init__(self) -> None:
        """
        Exception raised when no input is available for generating an error message.

        This exception is initialized with the default message "No input provided for error message generation."
        """
        super().__init__("No input provided for error message generation.")


class Rule(Generic[GeneralManagerType]):
    """
    Encapsulate a boolean predicate and derive contextual error messages from its AST.

    When the predicate evaluates to False, the rule inspects the parsed abstract syntax tree
    to determine which variables failed and crafts either autogenerated or custom error messages.
    """

    _func: Callable[[GeneralManagerType], bool]
    _custom_error_message: Optional[str]
    _ignore_if_none: bool
    _last_result: Optional[bool]
    _last_input: Optional[GeneralManagerType]
    _last_args: Dict[str, object]
    _param_names: Tuple[str, ...]
    _primary_param: Optional[str]
    _tree: ast.AST
    _variables: List[str]
    _handlers: Dict[str, BaseRuleHandler]

    def __init__(
        self,
        func: Callable[[GeneralManagerType], bool],
        custom_error_message: Optional[str] = None,
        ignore_if_none: bool = True,
    ) -> None:
        """
        Initialize a Rule that wraps a predicate function, captures its source AST to discover referenced variables, and prepares handlers for generating error messages.

        Parameters:
            func (Callable[[GeneralManagerType], bool]): Predicate that evaluates a GeneralManager instance.
            custom_error_message (Optional[str]): Optional template used to format generated error messages; placeholders must match variables referenced by the predicate.
            ignore_if_none (bool): If True, evaluation will be skipped (result recorded as None) when any referenced variable resolves to None.
        """
        self._func = func
        self._custom_error_message = custom_error_message
        self._ignore_if_none = ignore_if_none
        self._last_result = None
        self._last_input = None
        self._last_args = {}

        parameters = inspect.signature(func).parameters
        self._param_names = tuple(parameters.keys())
        self._primary_param = self._param_names[0] if self._param_names else None

        # 1) Extract source, strip decorators, and dedent
        src = textwrap.dedent(inspect.getsource(func))

        # 2) Parse AST and attach parent references
        self._tree = ast.parse(src)
        for parent in ast.walk(self._tree):
            for child in ast.iter_child_nodes(parent):
                child.parent = parent  # type: ignore

        # 3) Extract referenced variables
        self._variables = self._extract_variables()

        # 4) Register handlers
        self._handlers = {}  # type: Dict[str, BaseRuleHandler]
        for cls in (LenHandler, MaxHandler, MinHandler, SumHandler):
            inst: BaseRuleHandler = cls()
            self._handlers[inst.function_name] = inst
        for path in getattr(settings, "RULE_HANDLERS", []):
            handler_cls: type[BaseRuleHandler] = import_string(path)
            inst = handler_cls()
            self._handlers[inst.function_name] = inst

    @property
    def func(self) -> Callable[[GeneralManagerType], bool]:
        return self._func

    @property
    def customErrorMessage(self) -> Optional[str]:
        return self._custom_error_message

    @property
    def variables(self) -> List[str]:
        return self._variables

    @property
    def lastEvaluationResult(self) -> Optional[bool]:
        return self._last_result

    @property
    def lastEvaluationInput(self) -> Optional[GeneralManagerType]:
        return self._last_input

    @property
    def ignoreIfNone(self) -> bool:
        return self._ignore_if_none

    def evaluate(self, x: GeneralManagerType) -> Optional[bool]:
        """
        Evaluate the rule's predicate against a GeneralManager instance and record evaluation context.

        Binds the primary parameter to the provided input, extracts referenced variable values, and sets the last evaluation result to the predicate outcome. If `ignore_if_none` is true and any referenced variable value is `None`, the evaluation is skipped and the last result is set to `None`.

        Parameters:
            x (GeneralManagerType): Manager instance supplied to the predicate.

        Returns:
            `True` if the predicate evaluates to true, `False` if it evaluates to false, `None` if evaluation was skipped because a referenced value was `None` and `ignore_if_none` is enabled.
        """
        self._last_input = x
        self._last_args = {}
        if self._primary_param is not None:
            self._last_args[self._primary_param] = x

        vals = self._extract_variable_values(x)
        if self._ignore_if_none and any(v is None for v in vals.values()):
            self._last_result = None
            return None

        self._last_result = self._func(x)
        return self._last_result

    def validateCustomErrorMessage(self) -> None:
        """
        Validate that a provided custom error message template includes placeholders for every variable referenced by the rule.

        Raises:
            MissingErrorTemplateVariableError: If one or more extracted variables are not present as `{name}` placeholders in the custom template.
        """
        if not self._custom_error_message:
            return

        vars_in_msg = set(re.findall(r"{([^}]+)}", self._custom_error_message))
        missing = [v for v in self._variables if v not in vars_in_msg]
        if missing:
            raise MissingErrorTemplateVariableError(missing)

    def getErrorMessage(self) -> Optional[Dict[str, str]]:
        """
        Constructs error messages for the last failed evaluation and returns them keyed by variable name.

        Returns:
            dict[str, str] | None: Mapping from each referenced variable name to its error message, or `None` if the predicate passed or was not evaluated.

        Raises:
            ErrorMessageGenerationError: If called before any input has been evaluated.
        """
        if self._last_result or self._last_result is None:
            return None
        if self._last_input is None:
            raise ErrorMessageGenerationError()

        # Validate and substitute template placeholders
        self.validateCustomErrorMessage()
        vals = self._extract_variable_values(self._last_input)

        if self._custom_error_message:
            formatted = re.sub(
                r"{([^}]+)}",
                lambda m: str(vals.get(m.group(1), m.group(0))),
                self._custom_error_message,
            )
            return {v: formatted for v in self._variables}

        errors = self._generate_error_messages(vals)
        return errors or None

    def _extract_variables(self) -> List[str]:
        """
        Collects the dotted attribute names referenced on the rule's parameters.

        Scans the predicate's AST and returns a sorted list of attribute access paths that originate from any of the predicate's parameter names (for example, "user.name" or "order.total"). If the predicate has no parameters, returns an empty list.

        Returns:
            List[str]: Sorted list of dotted variable names referenced from the predicate's parameters.
        """
        param_names = set(self._param_names)
        if not param_names:
            return []

        class VarVisitor(ast.NodeVisitor):
            def __init__(self, params: set[str]) -> None:
                """
                Initialize visitor state with a set of parameter names and an empty collection for discovered variables.

                Parameters:
                    params (set[str]): Names of function parameters to consider when extracting referenced variables.
                """
                self.vars: set[str] = set()
                self.params = params

            def visit_Attribute(self, node: ast.Attribute) -> None:
                """
                Record dotted attribute accesses that originate from allowed parameter names.

                Visits an ast.Attribute node, collects the dotted name components (e.g., "a.b.c") if the base is an ast.Name present in self.params, and adds the joined name to self.vars. Continues generic traversal after handling the node.

                Parameters:
                        node (ast.Attribute): The attribute node being visited.
                """
                parts: list[str] = []
                curr: ast.AST = node
                while isinstance(curr, ast.Attribute):
                    parts.append(curr.attr)
                    curr = curr.value
                if isinstance(curr, ast.Name) and curr.id in self.params:
                    self.vars.add(".".join(reversed(parts)))
                self.generic_visit(node)

        visitor = VarVisitor(param_names)
        visitor.visit(self._tree)
        return sorted(visitor.vars)

    def _extract_variable_values(
        self, x: GeneralManagerType
    ) -> Dict[str, Optional[object]]:
        out: Dict[str, Optional[object]] = {}
        for var in self._variables:
            obj: object = x  # type: ignore
            for part in var.split("."):
                obj = getattr(obj, part, NOTEXISTENT)
                if obj is NOTEXISTENT:
                    raise NonexistentAttributeError(var)
                if obj is None:
                    break
            out[var] = obj
        return out

    def _extract_comparisons(self) -> list[ast.Compare]:
        """
        Collect all comparison (ast.Compare) nodes from the predicate AST.

        Searches the rule's parsed AST stored on self._tree and returns every ast.Compare node found in traversal order.

        Returns:
            comps (list[ast.Compare]): The list of comparison nodes present in the predicate AST.
        """

        class CompVisitor(ast.NodeVisitor):
            def __init__(self) -> None:
                self.comps: list[ast.Compare] = []

            def visit_Compare(self, node: ast.Compare) -> None:
                self.comps.append(node)
                self.generic_visit(node)

        visitor = CompVisitor()
        visitor.visit(self._tree)
        return visitor.comps

    def _contains_logical_ops(self) -> bool:
        class LogicVisitor(ast.NodeVisitor):
            found: bool = False

            def visit_BoolOp(self, node: ast.BoolOp) -> None:
                if isinstance(node.op, (ast.And, ast.Or)):
                    self.found = True
                self.generic_visit(node)

        visitor = LogicVisitor()
        visitor.visit(self._tree)
        return visitor.found

    def _generate_error_messages(
        self, var_values: Dict[str, Optional[object]]
    ) -> Dict[str, str]:
        """
        Generate human-readable error messages for each referenced variable using the resolved variable values.

        Given a mapping of variable names to their evaluated values, produce a dictionary mapping each variable to an explanatory error message derived from the rule's predicate. If specialized rule handlers are registered for particular function calls in the predicate, those handlers will be used to produce messages for the affected variables. When the predicate contains boolean combinations and the last evaluation failed, all referenced variables receive a combined invalid-combination message. If no explicit comparisons are present in the predicate, a generic combination-invalid message is returned for every referenced variable.

        Parameters:
            var_values (Dict[str, Optional[object]]): Mapping from variable names (as extracted from the predicate) to their resolved values for the last-evaluated input.

        Returns:
            Dict[str, str]: Mapping from variable name to its generated error message.
        """
        errors: Dict[str, str] = {}
        comparisons = self._extract_comparisons()
        logical = self._contains_logical_ops()

        if comparisons:
            for cmp in comparisons:
                left, rights, ops = cmp.left, cmp.comparators, cmp.ops
                for right, op in zip(rights, ops, strict=False):
                    # Special handler?
                    if isinstance(left, ast.Call):
                        fn = self._get_node_name(left.func)
                        handler = self._handlers.get(fn)
                        if handler:
                            errors.update(
                                handler.handle(cmp, left, right, op, var_values, self)
                            )
                            continue

                    # Standard error message
                    lnm = self._get_node_name(left)
                    rnm = self._get_node_name(right)
                    lval = self._eval_node(left)
                    rval = self._eval_node(right)
                    ldisp = f"[{lnm}] ({lval})" if lnm in var_values else str(lval)
                    rdisp = f"[{rnm}] ({rval})" if rnm in var_values else str(rval)
                    sym = self._get_op_symbol(op)
                    msg = f"{ldisp} must be {sym} {rdisp}!"
                    if lnm in var_values:
                        errors[lnm] = msg
                    if rnm in var_values and rnm != lnm:
                        errors[rnm] = msg

            if logical and not self._last_result:
                combo = ", ".join(f"[{v}]" for v in self._variables)
                msg = f"{combo} combination is not valid"
                for v in self._variables:
                    errors.setdefault(v, msg)  # keep specific messages

            return errors

        # No comparisons present → fall back to a generic message
        combo = ", ".join(f"[{v}]" for v in self._variables)
        return {v: f"{combo} combination is not valid" for v in self._variables}

    def _get_op_symbol(self, op: Optional[ast.cmpop]) -> str:
        if op is None:
            return "?"
        return {
            ast.Lt: "<",
            ast.LtE: "<=",
            ast.Gt: ">",
            ast.GtE: ">=",
            ast.Eq: "==",
            ast.NotEq: "!=",
            ast.Is: "is",
            ast.IsNot: "is not",
            ast.In: "in",
            ast.NotIn: "not in",
        }.get(type(op), "?")

    def _get_node_name(self, node: ast.AST) -> str:
        """
        Produce a concise, human-readable name for the given AST node.

        For attribute access returns the dotted attribute path (e.g., "a.b.c"); for a Name node returns its identifier; for a Call node returns "func(arg1, arg2)" using the same naming rules for subnodes; for Constant nodes or nodes that cannot be represented returns an empty string.

        Parameters:
            node (ast.AST): The AST node to derive a readable name from.

        Returns:
            str: The human-readable name or an empty string if no sensible name can be produced.
        """
        if isinstance(node, ast.Attribute):
            parts: list[str] = []
            curr: ast.AST = node
            while isinstance(curr, ast.Attribute):
                parts.insert(0, curr.attr)
                curr = curr.value
            return ".".join(parts)
        if isinstance(node, ast.Name):
            return node.id
        if isinstance(node, ast.Constant):
            return ""
        if isinstance(node, ast.Call):
            fn = self._get_node_name(node.func)
            args = ", ".join(self._get_node_name(a) for a in node.args)
            return f"{fn}({args})"
        try:
            # ast.unparse returns a string representation
            return ast.unparse(node)
        except (AttributeError, ValueError, TypeError):
            return ""

    def _eval_node(self, node: ast.expr) -> Optional[object]:
        """
        Evaluate an AST expression against the Rule's last-evaluated input and bound argument context.

        Parameters:
            node (ast.expr): The AST expression node to evaluate.

        Returns:
            result (Optional[object]): The evaluated value of the expression when resolvable; `None` if the node is not an expression or cannot be evaluated in the current context.

        Description:
            Supported evaluations:
            - Attribute access: resolves chained attributes from the last input or resolved base value.
            - Name: returns a bound argument value if present, otherwise looks up the name in the predicate function's global namespace.
            - Unary negative: evaluates numeric negation for integer, float, or Decimal operands.
            If the expression cannot be resolved (including missing intermediate attributes or unsupported node kinds), `None` is returned.
        """
        if not isinstance(node, ast.expr):
            return None
        try:
            return ast.literal_eval(node)
        except (ValueError, TypeError):
            if isinstance(node, ast.Attribute):
                base = self._eval_node(node.value)
                if base is None:
                    return None
                return getattr(base, node.attr, None)
            if isinstance(node, ast.Name):
                if node.id in self._last_args:
                    return self._last_args[node.id]
                return self._func.__globals__.get(node.id)
            if isinstance(node, ast.UnaryOp) and isinstance(node.op, ast.USub):
                operand = self._eval_node(node.operand)
                if isinstance(operand, (int, float, Decimal)):
                    return -operand
        return None
