import ctypes
from collections.abc import Iterable
from typing import Self

import numpy as np

from pyautd3.driver.firmware.fpga.emit_intensity import Intensity
from pyautd3.driver.geometry import Geometry
from pyautd3.driver.utils import _validate_nonzero_u32
from pyautd3.gain.holo.amplitude import Amplitude
from pyautd3.gain.holo.constraint import EmissionConstraint
from pyautd3.gain.holo.holo import Holo
from pyautd3.native_methods.autd3capi_driver import GainPtr
from pyautd3.native_methods.autd3capi_gain_holo import EmissionConstraintWrap
from pyautd3.native_methods.autd3capi_gain_holo import GSOption as GSOption_
from pyautd3.native_methods.autd3capi_gain_holo import NativeMethods as GainHolo
from pyautd3.native_methods.structs import Point3


class GSOption:
    repeat: int
    constraint: EmissionConstraintWrap

    def __init__(self: Self, *, repeat: int = 100, constraint: EmissionConstraintWrap | None = None) -> None:
        self.repeat = _validate_nonzero_u32(repeat)
        self.constraint = constraint or EmissionConstraint.Clamp(Intensity.MIN, Intensity.MAX)

    def _inner(self: Self) -> GSOption_:
        return GSOption_(self.constraint, self.repeat)


class GS(Holo["GS"]):
    option: GSOption

    def __init__(self: Self, foci: Iterable[tuple[np.ndarray, Amplitude]], option: GSOption) -> None:
        super().__init__(foci)
        self.option = option

    def _gain_ptr(self: Self, _: Geometry) -> GainPtr:
        size = len(self.foci)
        points = np.fromiter((np.void(Point3(d)) for d, _ in self.foci), dtype=Point3)  # type: ignore[type-var,call-overload]
        amps = np.fromiter((d.pascal() for _, d in self.foci), dtype=ctypes.c_float)  # type: ignore[type-var,call-overload]
        return GainHolo().gain_holo_gs_sphere(
            points.ctypes.data_as(ctypes.POINTER(Point3)),  # type: ignore[arg-type]
            amps.ctypes.data_as(ctypes.POINTER(ctypes.c_float)),  # type: ignore[arg-type]
            size,
            self.option._inner(),
        )
