# PySploit - Advanced Vulnerability Analysis Library

**Comprehensive Security Intelligence Platform**

PySploit is an advanced vulnerability analysis library featuring comprehensive security intelligence from multiple authoritative sources. The library provides extensive vulnerability assessment capabilities with an integrated database containing over 55,000 vulnerabilities.

**Core Capabilities:**
- **Comprehensive Vulnerability Database** with 55,712 unique vulnerabilities
- **Multi-Source Intelligence Integration** - CVE, ExploitDB, NVD, RouterSploit  
- **Advanced Security Assessment** with pattern-based threat detection
- **High-Performance Architecture** with optimized in-memory database
- **Extensive Threat Analysis** and comprehensive security intelligence

PySploit delivers comprehensive vulnerability assessment capabilities through integration with authoritative security databases including CVE, ExploitDB, NVD, and RouterSploit. The library features high-performance architecture designed for advanced security analysis and vulnerability management.

## Installation

### Version Support
Python 3.8+ is supported. Works on Windows, Linux, and macOS.

### All Platforms
Simply run the following to install the latest version:
```bash
pip install pysploit
```

Or install from the git repository:
```bash
git clone https://github.com/D14b0l1c/PySploit.git
cd PySploit
pip install -e .
```

## Quick Start

### Basic Vulnerability Assessment

```python
import pysploit

# Initialize the analysis engine
analyzer = pysploit.PySploitAnalyzer()

# Define target services for assessment
services = [
    {'port': 80, 'service': 'http', 'version': '2.4.29', 'banner': 'Apache/2.4.29'},
    {'port': 22, 'service': 'ssh', 'version': '7.4', 'banner': 'OpenSSH_7.4'},
    {'port': 443, 'service': 'https', 'version': '1.14.2', 'banner': 'nginx/1.14.2'}
]

# Perform comprehensive vulnerability analysis
results = analyzer.analyze_service_vulnerabilities('192.168.1.100', services)

# Review assessment results
print(f"Services analyzed: {results['services_analyzed']}")
print(f"Vulnerabilities identified: {len(results['vulnerabilities_found'])}")
print(f"Security risk score: {results['risk_score']:.2f}")

# Display vulnerability details
for vulnerability in results['vulnerabilities_found']:
    print(f"- {vulnerability['title']} (Severity: {vulnerability['severity']})")
```

## Key Features

### Comprehensive Vulnerability Intelligence
- **55,712 Vulnerability Entries**: Complete database with multi-source integration
- **ExploitDB Integration**: 46,453 exploit signatures for comprehensive threat analysis
- **NVD Coverage**: 9,191 National Vulnerability Database entries
- **RouterSploit Modules**: 68 specialized router and IoT security assessments
- **Vulnerability Matching**: Match network signatures against known vulnerabilities
- **Report Generation**: Create detailed security assessment reports
- **Modular Design**: Import only the components you need

## API Documentation

### Core Modules

#### `pysploit.VulnerabilityDatabase`
```python
from pysploit import VulnerabilityDatabase

# Initialize database
db = VulnerabilityDatabase()

# Load vulnerability data
cves = db.load_cves()
exploits = db.load_exploits()
routersploit_data = db.load_routersploit()

# Search vulnerabilities
results = db.search("CVE-2021-44228")
router_vulns = db.search_by_category("router")
```

#### `pysploit.PcapAnalyzer`
```python
from pysploit import PcapAnalyzer

analyzer = PcapAnalyzer()

# Analyze PCAP file
results = analyzer.analyze("capture.pcap")

# Extract specific protocol data
http_data = analyzer.extract_http("capture.pcap")
dns_data = analyzer.extract_dns("capture.pcap")

# Apply vulnerability filters
router_traffic = analyzer.filter_router_traffic(results)
suspicious_patterns = analyzer.detect_suspicious_patterns(results)
```

#### `pysploit.NmapAnalyzer`
```python
from pysploit import NmapAnalyzer

analyzer = NmapAnalyzer()

# Parse Nmap XML
scan_results = analyzer.parse_xml("nmap_scan.xml")

# Extract service information
services = analyzer.extract_services(scan_results)

# Identify potential vulnerabilities
vulnerabilities = analyzer.identify_vulnerabilities(services)
```

#### `pysploit.VulnerabilityMatcher`
```python
from pysploit import VulnerabilityMatcher, VulnerabilityDatabase

db = VulnerabilityDatabase()
matcher = VulnerabilityMatcher(db)

# Match network data against vulnerabilities
pcap_matches = matcher.match_pcap_data(pcap_results)
nmap_matches = matcher.match_nmap_data(nmap_results)

# Get detailed match information
for match in pcap_matches:
    print(f"CVE: {match.cve_id}")
    print(f"Severity: {match.severity}")
    print(f"Description: {match.description}")
```

## Library Structure

```
pysploit/
├── __init__.py              # Main library interface
├── core/                    # Core functionality
│   ├── database.py          # Vulnerability database management
│   ├── pcap_analyzer.py     # PCAP analysis tools
│   ├── nmap_analyzer.py     # Nmap XML parsing
│   └── matcher.py           # Vulnerability matching engine
├── data/                    # Data management
│   ├── nvd.py              # NVD API integration
│   ├── exploitdb.py        # ExploitDB data handling
│   └── routersploit.py     # RouterSploit data integration
├── utils/                   # Utility functions
│   ├── filters.py          # Traffic and data filters
│   ├── reports.py          # Report generation
│   └── helpers.py          # Helper functions
└── examples/               # Usage examples
    ├── basic_analysis.py
    ├── pcap_scanning.py
    └── vulnerability_research.py
```

## Examples

### Basic Vulnerability Analysis
```python
import pysploit

# Initialize
db = pysploit.VulnerabilityDatabase()
analyzer = pysploit.PcapAnalyzer()

# Analyze network traffic
traffic_data = analyzer.analyze("network.pcap")

# Find vulnerabilities
matcher = pysploit.VulnerabilityMatcher(db)
vulnerabilities = matcher.match_traffic(traffic_data)

# Generate report
report = pysploit.generate_report(vulnerabilities, format="json")
```

### Router Vulnerability Assessment
```python
import pysploit

# Load router-specific vulnerability data
db = pysploit.VulnerabilityDatabase()
router_cves = db.search_by_category("router")

# Analyze router traffic
analyzer = pysploit.PcapAnalyzer()
router_traffic = analyzer.filter_router_traffic("capture.pcap")

# Match against router vulnerabilities
matcher = pysploit.VulnerabilityMatcher(db)
router_vulns = matcher.match_router_signatures(router_traffic)
```

### Integration with Existing Tools
```python
# Use with pandas for data analysis
import pandas as pd
import pysploit

db = pysploit.VulnerabilityDatabase()
cves_df = pd.DataFrame(db.load_cves())

# Filter high-severity CVEs
critical_cves = cves_df[cves_df['severity'] == 'CRITICAL']

# Use with requests for API integration
import requests
import pysploit

# Fetch live vulnerability feeds
nvd_updater = pysploit.NVDUpdater()
latest_cves = nvd_updater.fetch_recent_cves(days=7)
```

## Data Sources

- **National Vulnerability Database (NVD)**: https://services.nvd.nist.gov/rest/json/cves/2.0
- **ExploitDB**: https://gitlab.com/exploit-database/exploitdb
- **RouterSploit Framework**: https://github.com/threat9/routersploit

## License

MIT License - see LICENSE file for details.

## Contributing

1. Fork the repository
2. Create a feature branch
3. Add tests for new functionality
4. Submit a pull request