"""
PySploit Usage Examples
This file demonstrates how to use the PySploit library for vulnerability analysis
"""

import sys
from pathlib import Path

# Add parent directory to path for imports
sys.path.insert(0, str(Path(__file__).parent.parent))

import pysploit
from datetime import datetime, timedelta

def example_basic_usage():
    """Basic usage example"""
    print("=== Basic PySploit Usage ===")
    
    # Initialize database
    with pysploit.VulnerabilityDatabase() as db:
        # Search for a specific CVE
        results = db.search_by_cve("CVE-2021-44228")  # Log4j vulnerability
        print(f"Found {len(results)} results for Log4j CVE")
        
        if results:
            cve = results[0]
            print(f"CVE: {cve.get('id', 'N/A')}")
            print(f"Score: {cve.get('base_score', 'N/A')}")
            print(f"Description: {cve.get('description', 'N/A')[:100]}...")

def example_pcap_analysis():
    """PCAP analysis example"""
    print("\n=== PCAP Analysis Example ===")
    
    # Initialize PCAP analyzer
    analyzer = pysploit.PcapAnalyzer()
    
    # Example: analyze a PCAP file (replace with actual file path)
    pcap_file = "sample_traffic.pcap"
    
    try:
        # Analyze the PCAP file
        with analyzer:
            results = analyzer.analyze(pcap_file)
            
        print(f"Analysis completed for {pcap_file}")
        print(f"Total packets: {results.get('total_packets', 0)}")
        print(f"Suspicious packets: {len(results.get('suspicious_packets', []))}")
        print(f"Vulnerabilities detected: {len(results.get('vulnerabilities', []))}")
        
        # Show first few vulnerabilities
        for i, vuln in enumerate(results.get('vulnerabilities', [])[:3]):
            print(f"  {i+1}. {vuln.get('type', 'Unknown')} - {vuln.get('description', 'No description')}")
            
    except FileNotFoundError:
        print(f"PCAP file {pcap_file} not found. Create a sample file to test this.")
    except Exception as e:
        print(f"Error analyzing PCAP: {e}")

def example_nmap_analysis():
    """Nmap XML analysis example"""
    print("\n=== Nmap Analysis Example ===")
    
    # Initialize Nmap analyzer
    analyzer = pysploit.NmapAnalyzer()
    
    # Example: analyze Nmap XML output
    nmap_file = "nmap_scan.xml"
    
    try:
        results = analyzer.parse_xml(nmap_file)
        
        print(f"Nmap scan analysis for {nmap_file}")
        print(f"Hosts discovered: {len(results.get('hosts', []))}")
        
        for host in results.get('hosts', [])[:3]:  # Show first 3 hosts
            print(f"Host: {host.get('address', 'Unknown')}")
            print(f"  Status: {host.get('status', 'Unknown')}")
            print(f"  Open ports: {len(host.get('ports', []))}")
            
            for port in host.get('ports', [])[:5]:  # Show first 5 ports
                print(f"    Port {port.get('port')}/{port.get('protocol')} - {port.get('service', 'Unknown service')}")
                
    except FileNotFoundError:
        print(f"Nmap file {nmap_file} not found. Run an nmap scan and save as XML to test this.")
    except Exception as e:
        print(f"Error analyzing Nmap XML: {e}")

def example_vulnerability_matching():
    """Vulnerability matching example"""
    print("\n=== Vulnerability Matching Example ===")
    
    # Initialize components
    with pysploit.VulnerabilityDatabase() as db:
        matcher = pysploit.VulnerabilityMatcher(db)
        
        # Example service information (typically from Nmap scan)
        services = [
            {
                'service': 'Apache httpd',
                'version': '2.4.41',
                'port': 80,
                'host': '192.168.1.100'
            },
            {
                'service': 'OpenSSH',
                'version': '7.4',
                'port': 22,
                'host': '192.168.1.100'
            },
            {
                'service': 'MySQL',
                'version': '5.7.30',
                'port': 3306,
                'host': '192.168.1.101'
            }
        ]
        
        print("Checking services for vulnerabilities...")
        
        for service in services:
            vulnerabilities = matcher.match_service_vulnerabilities(
                service['service'],
                service['version']
            )
            
            print(f"\n{service['service']} {service['version']} on {service['host']}:{service['port']}")
            if vulnerabilities:
                print(f"  Found {len(vulnerabilities)} potential vulnerabilities:")
                for vuln in vulnerabilities[:3]:  # Show first 3
                    print(f"    - {vuln.get('id', 'N/A')} (Score: {vuln.get('base_score', 'N/A')})")
            else:
                print("  No known vulnerabilities found")

def example_quick_analysis():
    """Quick analysis using convenience functions"""
    print("\n=== Quick Analysis Example ===")
    
    # Use convenience functions for quick analysis
    
    # Quick vulnerability search
    results = pysploit.search_cves("buffer overflow", limit=5)
    print(f"Found {len(results)} CVEs for 'buffer overflow'")
    
    # Get router vulnerabilities
    router_vulns = pysploit.get_router_vulnerabilities(limit=3)
    print(f"Found {len(router_vulns)} router vulnerabilities")
    
    # Quick analysis (if you had both PCAP and Nmap files)
    try:
        analysis_results = pysploit.quick_analyze(
            pcap_file="sample_traffic.pcap",
            nmap_file="nmap_scan.xml"
        )
        print("Quick analysis completed successfully")
        print(f"Network vulnerabilities: {len(analysis_results.get('network_vulns', []))}")
        print(f"Service vulnerabilities: {len(analysis_results.get('service_vulns', []))}")
    except FileNotFoundError:
        print("Sample files not found for quick analysis demo")
    except Exception as e:
        print(f"Quick analysis error: {e}")

def example_reporting():
    """Report generation example"""
    print("\n=== Report Generation Example ===")
    
    from pysploit.utils.reports import ReportGenerator
    
    # Sample data for report
    sample_data = {
        'scan_info': {
            'timestamp': datetime.now().isoformat(),
            'target': '192.168.1.0/24',
            'scan_type': 'Comprehensive Security Assessment'
        },
        'network_analysis': {
            'total_packets': 15000,
            'suspicious_packets': 25,
            'protocols_detected': ['HTTP', 'HTTPS', 'SSH', 'FTP']
        },
        'host_discovery': {
            'total_hosts': 12,
            'active_hosts': 8,
            'hosts': [
                {'ip': '192.168.1.100', 'status': 'up', 'ports': [22, 80, 443]},
                {'ip': '192.168.1.101', 'status': 'up', 'ports': [22, 3306]}
            ]
        },
        'vulnerabilities': [
            {
                'id': 'CVE-2021-44228',
                'severity': 'CRITICAL',
                'base_score': 10.0,
                'description': 'Apache Log4j2 Remote Code Execution',
                'affected_hosts': ['192.168.1.100']
            },
            {
                'id': 'CVE-2020-1472',
                'severity': 'CRITICAL', 
                'base_score': 9.8,
                'description': 'Windows Netlogon Elevation of Privilege',
                'affected_hosts': ['192.168.1.102']
            }
        ]
    }
    
    # Generate reports in different formats
    generator = ReportGenerator()
    
    try:
        # Generate JSON report
        json_report = generator.generate_json_report(sample_data)
        print("JSON report generated successfully")
        
        # Generate HTML report
        html_report = generator.generate_html_report(sample_data)
        print("HTML report generated successfully")
        
        # Generate CSV report
        csv_report = generator.generate_csv_report(sample_data)
        print("CSV report generated successfully")
        
        # Save reports to files
        generator.save_report(json_report, "security_report.json")
        generator.save_report(html_report, "security_report.html")
        generator.save_report(csv_report, "security_report.csv")
        
        print("Reports saved to files")
        
    except Exception as e:
        print(f"Error generating reports: {e}")

def example_data_sources():
    """CVE data source example"""
    print("\n=== CVE Data Sources Example ===")
    
    from pysploit.data.cve_sources import CVEDataManager
    
    # Initialize data manager
    data_manager = CVEDataManager()
    
    try:
        # Search for recent high-severity CVEs
        print("Fetching recent high-severity CVEs...")
        recent_cves = data_manager.get_high_severity_cves(days=7)
        
        print(f"Found {len(recent_cves)} high/critical severity CVEs in the last 7 days")
        
        for cve in recent_cves[:3]:  # Show first 3
            print(f"  {cve.get('id', 'N/A')} - Score: {cve.get('base_score', 'N/A')} - {cve.get('severity', 'N/A')}")
            
        # Search for specific vulnerability
        print("\nSearching for Log4j vulnerabilities...")
        log4j_cves = data_manager.search_cves("log4j", severity_filter="HIGH")
        print(f"Found {len(log4j_cves)} Log4j related CVEs")
        
    except Exception as e:
        print(f"Error fetching CVE data: {e}")
        print("Note: This requires internet connection and may be rate-limited")

def example_comprehensive_assessment():
    """Comprehensive vulnerability assessment example (like Analysis/Vulnerability)"""
    print("\n=== Comprehensive Vulnerability Assessment ===")
    
    try:
        # Initialize the comprehensive assessment engine
        with pysploit.PySploitVulnerabilityAssessment() as assessor:
            
            # Example Nmap scan data (normally from actual scan)
            sample_nmap_data = {
                'hosts': [
                    {
                        'address': '192.168.1.1',
                        'hostname': 'router.local',
                        'os': {'name': 'Linux 2.6.x'},
                        'ports': [
                            {'port': 80, 'service': 'http', 'version': '1.0', 'product': 'linksys'},
                            {'port': 23, 'service': 'telnet', 'version': '', 'product': ''}
                        ]
                    },
                    {
                        'address': '192.168.1.100', 
                        'hostname': 'camera.local',
                        'os': {'name': 'Embedded Linux'},
                        'ports': [
                            {'port': 80, 'service': 'http', 'version': '2.0', 'product': 'hikvision'},
                            {'port': 554, 'service': 'rtsp', 'version': '1.0', 'product': 'hikvision'}
                        ]
                    }
                ]
            }
            
            # Perform network service assessment  
            print("Performing network service assessment...")
            nmap_assessment = assessor.assess_network_services(sample_nmap_data)
            
            print(f"Assessment Results:")
            print(f"  Total hosts scanned: {nmap_assessment['total_hosts']}")
            print(f"  Vulnerable hosts: {len(nmap_assessment['vulnerable_hosts'])}")
            print(f"  Critical findings: {len(nmap_assessment['critical_findings'])}")
            print(f"  RouterSploit matches: {len(nmap_assessment['routersploit_matches'])}")
            
            # Show vulnerable hosts
            for host in nmap_assessment['vulnerable_hosts']:
                print(f"\n  Host {host['address']} ({host['hostname']}):")
                print(f"    Risk Score: {host['risk_score']:.1f}/10.0")
                print(f"    RouterSploit Applicable: {host['routersploit_applicable']}")
                print(f"    Vulnerabilities found: {len(host['vulnerabilities'])}")
                
                for vuln in host['vulnerabilities'][:2]:  # Show first 2 vulns
                    print(f"      - {vuln.get('cve_id', 'N/A')} ({vuln.get('severity', 'N/A')}) - {vuln.get('source', 'N/A')}")
            
            # Example PCAP data assessment
            sample_pcap_data = {
                'total_packets': 5000,
                'packets': [
                    {
                        'ip.src': '192.168.1.100',
                        'ip.dst': '192.168.1.1', 
                        'http.host': 'router.local',
                        'http.request.uri': '/cgi-bin/setup.cgi?submit_button=1',
                        '_ws.col.Protocol': 'HTTP'
                    },
                    {
                        'ip.src': '192.168.1.50',
                        'dns.qry.name': 'mirai.botnet.com',
                        '_ws.col.Protocol': 'DNS'
                    }
                ]
            }
            
            print("\nPerforming traffic analysis...")
            pcap_assessment = assessor.assess_pcap_traffic(sample_pcap_data)
            
            print(f"Traffic Analysis Results:")
            print(f"  Total packets analyzed: {pcap_assessment['total_packets']}")
            print(f"  Suspicious indicators: {len(pcap_assessment['suspicious_indicators'])}")
            print(f"  Potentially compromised hosts: {len(pcap_assessment['compromised_hosts'])}")
            
            # Generate comprehensive report
            print("\nGenerating comprehensive assessment report...")
            report = assessor.generate_comprehensive_report(
                nmap_assessment=nmap_assessment,
                pcap_assessment=pcap_assessment
            )
            
            print(f"Assessment Report Summary:")
            print(f"  Assessment ID: {report['assessment_id']}")
            print(f"  Overall Risk Level: {report['summary']['risk_level']}")
            print(f"  Total Vulnerabilities: {report['summary']['total_vulnerabilities']}")
            print(f"  Critical Vulnerabilities: {report['summary']['critical_vulnerabilities']}")
            print(f"  RouterSploit Applicable: {report['summary']['routersploit_applicable']}")
            
            print(f"\nTop Recommendations:")
            for i, rec in enumerate(report['recommendations'][:3], 1):
                print(f"  {i}. {rec}")
    
    except Exception as e:
        print(f"Error in comprehensive assessment: {e}")

def example_routersploit_integration():
    """RouterSploit integration example"""
    print("\n=== RouterSploit Integration Example ===")
    
    try:
        # Get RouterSploit vulnerability database
        routersploit_db = pysploit.get_routersploit_database()
        
        # Get database statistics
        stats = routersploit_db.get_statistics()
        print(f"RouterSploit Database Statistics:")
        print(f"  Total vulnerabilities: {stats['total_vulnerabilities']}")
        print(f"  Exploitable vulnerabilities: {stats['exploitable_count']}")
        print(f"  CVE-mapped vulnerabilities: {stats['cve_mapped']}")
        
        print(f"\nBy Category:")
        for category, count in stats['by_category'].items():
            print(f"    {category}: {count}")
        
        print(f"\nBy Severity:")
        for severity, count in stats['by_severity'].items():
            if count > 0:
                print(f"    {severity}: {count}")
        
        # Test device assessment
        print(f"\nTesting RouterSploit compatibility assessment...")
        
        # Example devices to test
        test_devices = [
            {'vendor': 'Linksys', 'model': 'E1500'},
            {'vendor': 'Netgear', 'model': 'R7000'},  
            {'vendor': 'Hikvision', 'model': 'DS-2CD'},
            {'vendor': 'Unknown', 'model': 'Generic'}
        ]
        
        for device in test_devices:
            assessment = pysploit.assess_device_routersploit_compatibility(
                vendor=device['vendor'],
                model=device['model'],
                services=[{'port': 80, 'name': 'http'}]
            )
            
            print(f"\nDevice: {device['vendor']} {device['model']}")
            print(f"  RouterSploit Applicable: {assessment['routersploit_applicable']}")
            print(f"  Risk Level: {assessment['risk_level']}")
            print(f"  Applicable Modules: {len(assessment['applicable_modules'])}")
            
            if assessment['applicable_modules']:
                for module in assessment['applicable_modules'][:2]:  # Show first 2
                    vuln = module['vulnerability']
                    print(f"    - {vuln['module_name']}: {vuln['description'][:60]}...")
        
        # Show critical RouterSploit vulnerabilities
        print(f"\nCritical RouterSploit Vulnerabilities:")
        critical_vulns = routersploit_db.get_critical_vulnerabilities()
        
        for vuln in critical_vulns[:3]:  # Show first 3
            print(f"  - {vuln['vendor']} {vuln['device_models'][0]}: {vuln['description']}")
            if vuln.get('cve_references'):
                print(f"    CVE: {', '.join(vuln['cve_references'])}")
    
    except Exception as e:
        print(f"Error in RouterSploit integration: {e}")

if __name__ == "__main__":
    print("PySploit Library Examples")
    print("=" * 50)
    
    # Run all examples
    example_basic_usage()
    example_pcap_analysis()
    example_nmap_analysis() 
    example_vulnerability_matching()
    example_quick_analysis()
    example_reporting()
    example_data_sources()
    example_comprehensive_assessment()
    example_routersploit_integration()
    
    print("\n" + "=" * 50)
    print("Examples completed!")
    print("\nTo use PySploit in your own projects:")
    print("1. Install: pip install -e .")
    print("2. Import: import pysploit")
    print("3. Use: db = pysploit.VulnerabilityDatabase()")