"""
PySploit Advanced Vulnerability Analysis Demonstration
Advanced security assessment capabilities with comprehensive vulnerability intelligence
"""

import pysploit

def demo_basic_usage():
    """Advanced vulnerability analysis demonstration"""
    print("PySploit Advanced Security Analysis Platform")
    print("=" * 55)
    
    # Initialize advanced analyzer
    analyzer = pysploit.PySploitAnalyzer()
    
    # Get comprehensive database information
    db_info = analyzer.get_database_info()
    stats = db_info['statistics']
    print(f"Vulnerability Database: {stats['total_vulnerabilities']} entries")
    print(f"Database Type: {db_info['database_type']}")
    print(f"Data Sources: {len(db_info['data_sources'])}")
    print(f"Analysis Capabilities: {len(db_info['capabilities'])}")
    print()

def demo_host_analysis():
    """Demonstrate professional vulnerability assessment"""
    print("Enterprise Security Assessment")
    print("-" * 35)
    
    # Example: Analyze a Linksys router
    results = pysploit.analyze_host('192.168.1.1', [
        {'port': 80, 'service': 'http', 'version': '1.0', 'banner': 'Linksys E1500'},
        {'port': 23, 'service': 'telnet', 'version': '', 'banner': ''}
    ])
    
    print(f"Host: {results['host']}")
    print(f"Risk Score: {results['risk_score']:.1f}/10.0")
    print(f"Vulnerabilities Found: {len(results['vulnerabilities_found'])}")
    
    for vuln in results['vulnerabilities_found']:
        print(f"  [CRITICAL] {vuln['title']}")
        print(f"      Severity: {vuln['severity']} (Score: {vuln['base_score']})")
        print(f"      CVE: {vuln['cve_id']}")
        print(f"      Description: {vuln['description'][:80]}...")
        print()

def demo_traffic_analysis():
    """Demonstrate network traffic analysis"""
    print("Network Traffic Analysis")
    print("-" * 30)
    
    # Example traffic packets
    sample_traffic = [
        {
            'src_ip': '192.168.1.100',
            'dst_ip': '192.168.1.1',
            'payload': 'GET /cgi-bin/tmUnblock.cgi HTTP/1.1',
            'uri': '/cgi-bin/tmUnblock.cgi'
        },
        {
            'src_ip': '192.168.1.50',
            'payload': 'DNS query to mirai.botnet.com',
            'host': 'mirai.botnet.com'
        },
        {
            'src_ip': '10.0.0.5',
            'payload': 'GET /admin/../../../etc/passwd HTTP/1.1',
            'uri': '/admin/../../../etc/passwd'
        }
    ]
    
    results = pysploit.analyze_traffic(sample_traffic)
    
    print(f"Packets Analyzed: {results['packets_analyzed']}")
    print(f"Suspicious Indicators: {len(results['suspicious_indicators'])}")
    print(f"Compromised Hosts: {len(results['compromised_hosts'])}")
    
    print("\nSuspicious Activities:")
    for indicator in results['suspicious_indicators'][:5]:  # Show first 5
        print(f"  [WARNING] {indicator['type']} ({indicator['severity']})")
        print(f"      Source: {indicator['src_ip']}")
        print(f"      Pattern: {indicator['pattern_matched']}")
        print()

def demo_quick_assessment():
    """Demonstrate comprehensive quick assessment"""
    print("Quick Vulnerability Assessment")
    print("-" * 30)
    
    # Example network with multiple hosts
    test_network = [
        {
            'ip': '192.168.1.1',
            'services': [
                {'port': 80, 'service': 'http', 'banner': 'Linksys E1500'},
                {'port': 23, 'service': 'telnet', 'banner': ''}
            ]
        },
        {
            'ip': '192.168.1.100', 
            'services': [
                {'port': 80, 'service': 'http', 'banner': 'Hikvision DS-2CD'},
                {'port': 554, 'service': 'rtsp', 'banner': 'Hikvision'}
            ]
        },
        {
            'ip': '192.168.1.200',
            'services': [
                {'port': 80, 'service': 'http', 'banner': 'Apache/2.4'},
                {'port': 22, 'service': 'ssh', 'banner': 'OpenSSH 7.4'}
            ]
        }
    ]
    
    # Example suspicious traffic
    suspicious_traffic = [
        {'src_ip': '192.168.1.100', 'payload': 'GET /cgi-bin/setup.cgi?submit_button=1'},
        {'src_ip': '192.168.1.50', 'payload': 'DNS mirai.botnet.com'},
        {'src_ip': '10.0.0.1', 'payload': 'GET /admin/../../etc/passwd'}
    ]
    
    # Run comprehensive assessment
    assessment = pysploit.quick_scan(hosts=test_network, traffic=suspicious_traffic)
    
    print(f"Assessment ID: {assessment['assessment_id']}")
    print(f"Overall Risk Level: {assessment['overall_risk']}")
    print(f"Critical Findings: {len(assessment['critical_findings'])}")
    
    if assessment['network_assessment']:
        na = assessment['network_assessment']
        print(f"\nNetwork Assessment:")
        print(f"  Hosts Analyzed: {na['hosts_analyzed']}")
        print(f"  Vulnerable Hosts: {na['vulnerable_hosts']}")
        print(f"  Total Vulnerabilities: {na['total_vulnerabilities']}")
    
    if assessment['traffic_assessment']:
        ta = assessment['traffic_assessment']
        print(f"\nTraffic Assessment:")
        print(f"  Packets Analyzed: {ta['packets_analyzed']}")
        print(f"  Suspicious Indicators: {ta['summary']['total_indicators']}")
        print(f"  Compromised Hosts: {ta['summary']['potentially_compromised_hosts']}")
    
    print(f"\nTop Recommendations:")
    for i, rec in enumerate(assessment['recommendations'][:3], 1):
        print(f"  {i}. {rec}")

def demo_vulnerability_lookup():
    """Demonstrate vulnerability database lookup"""
    print("Vulnerability Database Lookup")
    print("-" * 30)
    
    # Search for specific vulnerabilities
    print("Critical Vulnerabilities:")
    critical_vulns = pysploit.get_critical_vulnerabilities()
    
    for vuln in critical_vulns[:3]:  # Show first 3
        print(f"  [CRITICAL] {vuln['cve_id']} - {vuln['severity']}")
        print(f"      Score: {vuln['base_score']}")
        print(f"      Description: {vuln['description'][:60]}...")
        print()
    
    # Search by vendor
    print("Linksys Vulnerabilities:")
    linksys_vulns = pysploit.search_vulnerabilities("Linksys")
    
    for vuln in linksys_vulns[:2]:  # Show first 2
        print(f"  [WARNING] {vuln['id']}")
        print(f"      Severity: {vuln['severity']} (Score: {vuln['base_score']})")
        print(f"      Description: {vuln['description'][:60]}...")
        print()
    
    # RouterSploit specific vulnerabilities
    print("RouterSploit Vulnerabilities:")
    router_vulns = pysploit.get_router_vulnerabilities()
    
    for vuln in router_vulns[:2]:  # Show first 2
        print(f"  [ROUTER] {vuln['id']}")
        print(f"      Vendor: {vuln.get('vendors', 'N/A')}")
        print(f"      Category: {vuln['category']}")
        print()

if __name__ == "__main__":
    print("PySploit - Complete Offline Vulnerability Analysis")
    print("================================================================")
    print("[+] No internet required")
    print("[+] No external database setup") 
    print("[+] No configuration needed")
    print("[+] Just import and use!")
    print()
    
    # Run all demos
    demo_basic_usage()
    print()
    
    demo_host_analysis()
    print()
    
    demo_traffic_analysis()
    print()
    
    demo_quick_assessment()
    print()
    
    demo_vulnerability_lookup()
    
    print("\n" + "=" * 50)
    print("PySploit Demo Complete!")
    print("\nKey Features Demonstrated:")
    print("- Embedded vulnerability database (works offline)")
    print("- Host vulnerability analysis")  
    print("- Network traffic analysis")
    print("- Comprehensive security assessment")
    print("- RouterSploit integration")
    print("- CVE database lookup")
    print("\nPySploit is ready to use - no setup required!")