"""
PySploit - Advanced Vulnerability Analysis Library

Advanced vulnerability assessment platform with comprehensive security intelligence.
Features integrated database with 55,000+ vulnerabilities from multiple sources.

Key Features:
- Comprehensive vulnerability database with 55,000+ entries
- Integrated CVE, ExploitDB, NVD, and RouterSploit intelligence  
- Advanced pattern-based threat detection algorithms
- Comprehensive vulnerability assessment capabilities
- High-performance in-memory database with optimized indexing
- Enterprise-ready vulnerability analysis and reporting

Quick Start:
    import pysploit
    
    # Analyze a single host
    results = pysploit.analyze_host('192.168.1.1', [
        {'port': 80, 'service': 'http', 'banner': 'Apache/2.4'}
    ])
    
    # Quick vulnerability assessment
    assessment = pysploit.quick_scan(hosts=[
        {'ip': '192.168.1.1', 'services': [{'port': 80, 'service': 'http'}]}
    ])
    
    # Get database info
    info = pysploit.get_database_info()
    print(f"Loaded {info['statistics']['total_vulnerabilities']} vulnerabilities")
"""

__version__ = "1.0.0"
__author__ = "D14b0l1c"
__license__ = "MIT"

# Core imports - works offline with embedded data
from .database.embedded import EmbeddedVulnerabilityDatabase, get_embedded_database
from .analysis.core import (
    PySploitAnalyzer, 
    analyze_host, 
    analyze_traffic, 
    quick_scan
)

# Advanced imports (may require external files/internet)
try:
    from .database.database import VulnerabilityDatabase
    from .analysis.pcap import PcapAnalyzer
    from .analysis.nmap import NmapAnalyzer
    from .vulnerability.matcher import VulnerabilityMatcher
    from .vulnerability.assessment import VulnerabilityAssessment
    ADVANCED_FEATURES_AVAILABLE = True
except ImportError:
    ADVANCED_FEATURES_AVAILABLE = False

# Optional data source imports
try:
    from .data.cve_sources import CVEDataManager
    from .data.database_manager import DatabaseManager, initialize_pysploit_database, update_vulnerability_database, get_database_status
    from .data.routersploit_data import RouterSploitVulnerabilityDatabase, RouterSploitIntegration, get_routersploit_database, assess_device_routersploit_compatibility
    DATA_SOURCES_AVAILABLE = True
except ImportError:
    DATA_SOURCES_AVAILABLE = False

# Optional utility imports
try:
    from .utils.reports import generate_report, ReportGenerator
    from .utils.filters import TrafficFilter, VulnerabilityFilter
    from .utils.helpers import load_config, setup_logging
    UTILS_AVAILABLE = True
except ImportError:
    UTILS_AVAILABLE = False

# Define what gets imported with "from pysploit import *"
__all__ = [
    # Core classes
    'VulnerabilityDatabase',
    'PcapAnalyzer', 
    'NmapAnalyzer',
    'VulnerabilityMatcher',
    'PySploitVulnerabilityAssessment',
    
    # Data sources
    'CVEDataManager',
    'DatabaseManager',
    'RouterSploitVulnerabilityDatabase',
    'RouterSploitIntegration',
    'initialize_pysploit_database',
    'update_vulnerability_database',
    'get_database_status',
    'get_routersploit_database',
    'assess_device_routersploit_compatibility',
    
    # Utilities
    'generate_report',
    'ReportGenerator',
    'TrafficFilter',
    'VulnerabilityFilter',
    'load_config',
    'setup_logging',
    
    # Package info
    '__version__',
    '__author__',
    '__license__'
]

# Package-level convenience functions
def quick_analyze(pcap_path, output_format="json"):
    """
    Quick vulnerability analysis of a PCAP file.
    
    Args:
        pcap_path (str): Path to PCAP file
        output_format (str): Output format ('json', 'csv', 'html')
    
    Returns:
        dict: Analysis results with vulnerability matches
    """
    db = VulnerabilityDatabase()
    analyzer = PcapAnalyzer()
    matcher = VulnerabilityMatcher(db)
    
    # Analyze traffic
    traffic_data = analyzer.analyze(pcap_path)
    
    # Match vulnerabilities
    matches = matcher.match_traffic(traffic_data)
    
    # Generate report
    report = generate_report(matches, format=output_format)
    
    return {
        'traffic_summary': traffic_data,
        'vulnerability_matches': matches,
        'report': report
    }

# Standalone convenience functions (work offline with embedded data)
def search_vulnerabilities(query: str, severity: str = None):
    """
    Search embedded vulnerability database (works offline)
    
    Args:
        query (str): Search term (vendor, CVE ID, etc.)
        severity (str, optional): Filter by severity (CRITICAL, HIGH, MEDIUM, LOW)
    
    Returns:
        list: Matching vulnerability records
    """
    db = get_embedded_database()
    
    if query.upper().startswith('CVE-'):
        return db.search_by_cve(query)
    elif severity:
        return [v for v in db.search_by_vendor(query) if v['severity'] == severity.upper()]
    else:
        return db.search_by_vendor(query)

def get_critical_vulnerabilities():
    """
    Get all critical vulnerabilities from embedded database
    
    Returns:
        list: Critical vulnerability records
    """
    db = get_embedded_database()
    return db.get_critical_vulnerabilities()

def get_database_info():
    """
    Get information about the embedded vulnerability database
    
    Returns:
        dict: Database statistics and information
    """
    analyzer = PySploitAnalyzer()
    return analyzer.get_database_info()

def get_router_vulnerabilities():
    """
    Get router-specific vulnerabilities from embedded RouterSploit data
    
    Returns:
        list: Router vulnerability records
    """
    db = get_embedded_database()
    return db.search_routersploit()

# Legacy function compatibility
def search_cves(query: str, limit: int = 20):
    """Legacy CVE search function - uses embedded database"""
    return search_vulnerabilities(query)[:limit]

# Initialize logging when package is imported
import logging
logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s - %(name)s - %(levelname)s - %(message)s'
)