#!/usr/bin/env python3
"""
PySploit PCAP Analyzer
Network traffic analysis with vulnerability detection capabilities.
"""

import subprocess
import pandas as pd
import os
import re
from typing import Optional, List, Dict, Any
import json


class PcapAnalyzer:
    """
    Analyze PCAP files for vulnerability indicators and network patterns.
    """
    
    def __init__(self):
        self.tshark_available = self._check_tshark()
        
    def _check_tshark(self) -> bool:
        """Check if tshark is available in system PATH."""
        try:
            subprocess.run(["tshark", "--version"], 
                         capture_output=True, check=True)
            return True
        except (subprocess.CalledProcessError, FileNotFoundError):
            return False
    
    def analyze(self, pcap_path: str, filter_type: str = "all") -> Dict[str, Any]:
        """
        Analyze PCAP file and extract vulnerability-relevant data.
        
        Args:
            pcap_path (str): Path to PCAP file
            filter_type (str): Analysis filter ('all', 'router', 'web', 'iot', 'suspicious')
        
        Returns:
            dict: Analysis results with extracted network data
        """
        if not self.tshark_available:
            raise RuntimeError("tshark is not available. Please install Wireshark/tshark.")
        
        if not os.path.exists(pcap_path):
            raise FileNotFoundError(f"PCAP file not found: {pcap_path}")
        
        # Extract data using tshark
        csv_data = self._extract_to_dataframe(pcap_path, filter_type)
        
        # Analyze the extracted data
        analysis_results = self._analyze_traffic(csv_data, filter_type)
        
        return {
            'pcap_file': pcap_path,
            'filter_type': filter_type,
            'packet_count': len(csv_data) if csv_data is not None else 0,
            'analysis_results': analysis_results,
            'raw_data': csv_data.to_dict('records') if csv_data is not None else []
        }
    
    def _extract_to_dataframe(self, pcap_path: str, filter_type: str) -> Optional[pd.DataFrame]:
        """Extract PCAP data to pandas DataFrame using tshark."""
        
        # Build tshark command based on filter type
        fields = self._get_fields_for_filter(filter_type)
        
        tshark_command = [
            "tshark", "-r", pcap_path, "-T", "fields"
        ]
        
        # Add field extraction parameters
        for field in fields:
            tshark_command.extend(["-e", field])
        
        # Add formatting options
        tshark_command.extend([
            "-E", "header=y",
            "-E", "separator=,", 
            "-E", "quote=d"
        ])
        
        try:
            # Run tshark and capture output
            result = subprocess.run(
                tshark_command, 
                capture_output=True, 
                text=True, 
                check=True
            )
            
            # Convert output to DataFrame
            if result.stdout:
                from io import StringIO
                df = pd.read_csv(StringIO(result.stdout))
                return df
            
        except subprocess.CalledProcessError as e:
            print(f"tshark error: {e}")
            return None
        except Exception as e:
            print(f"Error processing PCAP: {e}")
            return None
        
        return None
    
    def _get_fields_for_filter(self, filter_type: str) -> List[str]:
        """Get tshark fields based on analysis filter type."""
        
        # Base fields always included
        base_fields = [
            "frame.time_epoch", "frame.number", "frame.len",
            "ip.src", "ip.dst", "tcp.srcport", "tcp.dstport", 
            "udp.srcport", "udp.dstport", "_ws.col.Protocol", "_ws.col.Info"
        ]
        
        # Additional fields by filter type
        filter_fields = {
            "router": [
                "http.host", "http.request.uri", "http.user_agent", 
                "http.authorization", "telnet.data", "ssh.protocol",
                "snmp.community", "dhcp.option.hostname"
            ],
            "web": [
                "http.request.method", "http.request.uri", "http.host",
                "http.user_agent", "http.cookie", "http.response.code",
                "http.file_data", "tls.handshake.extensions.server_name"
            ],
            "iot": [
                "http.host", "dns.qry.name", "dhcp.option.hostname",
                "mdns.queries.name", "upnp.method", "coap.code"
            ],
            "weak_auth": [
                "ftp.request.command", "ftp.response.code", "telnet.data",
                "http.authorization", "snmp.community", "pop.request.command",
                "imap.request.command", "smtp.command"
            ],
            "suspicious": [
                "dns.qry.name", "http.request.uri", "tls.alert.description",
                "icmp.type", "tcp.analysis", "smb.cmd", "nbns.name"
            ]
        }
        
        if filter_type == "all":
            # Include all field types
            all_fields = base_fields[:]
            for fields in filter_fields.values():
                all_fields.extend(fields)
            # Remove duplicates while preserving order
            return list(dict.fromkeys(all_fields))
        
        return base_fields + filter_fields.get(filter_type, [])
    
    def _analyze_traffic(self, df: pd.DataFrame, filter_type: str) -> Dict[str, Any]:
        """Analyze extracted traffic data for vulnerability indicators."""
        
        if df is None or df.empty:
            return {"error": "No data to analyze"}
        
        analysis = {
            "summary": self._get_traffic_summary(df),
            "protocols": self._analyze_protocols(df),
            "hosts": self._analyze_hosts(df),
            "vulnerabilities": self._detect_vulnerabilities(df, filter_type)
        }
        
        return analysis
    
    def _get_traffic_summary(self, df: pd.DataFrame) -> Dict[str, Any]:
        """Generate traffic summary statistics."""
        return {
            "total_packets": len(df),
            "unique_src_ips": df['ip.src'].nunique() if 'ip.src' in df.columns else 0,
            "unique_dst_ips": df['ip.dst'].nunique() if 'ip.dst' in df.columns else 0,
            "protocols": df['_ws.col.Protocol'].value_counts().to_dict() if '_ws.col.Protocol' in df.columns else {},
            "time_span": {
                "start": df['frame.time_epoch'].min() if 'frame.time_epoch' in df.columns else None,
                "end": df['frame.time_epoch'].max() if 'frame.time_epoch' in df.columns else None
            }
        }
    
    def _analyze_protocols(self, df: pd.DataFrame) -> Dict[str, Any]:
        """Analyze protocol usage and patterns."""
        protocols = {}
        
        if '_ws.col.Protocol' in df.columns:
            protocol_counts = df['_ws.col.Protocol'].value_counts()
            protocols['distribution'] = protocol_counts.to_dict()
            
            # Identify potentially risky protocols
            risky_protocols = ['TELNET', 'FTP', 'HTTP', 'SNMP', 'SMB']
            detected_risky = [p for p in risky_protocols if p in protocol_counts.index]
            protocols['risky_protocols'] = detected_risky
        
        return protocols
    
    def _analyze_hosts(self, df: pd.DataFrame) -> Dict[str, Any]:
        """Analyze host communication patterns."""
        hosts = {}
        
        if 'ip.src' in df.columns and 'ip.dst' in df.columns:
            # Most active source IPs
            src_counts = df['ip.src'].value_counts().head(10)
            hosts['top_sources'] = src_counts.to_dict()
            
            # Most contacted destinations
            dst_counts = df['ip.dst'].value_counts().head(10)
            hosts['top_destinations'] = dst_counts.to_dict()
            
            # Communication pairs
            pairs = df.groupby(['ip.src', 'ip.dst']).size().sort_values(ascending=False).head(10)
            hosts['top_communications'] = pairs.to_dict()
        
        return hosts
    
    def _detect_vulnerabilities(self, df: pd.DataFrame, filter_type: str) -> List[Dict[str, Any]]:
        """Detect potential vulnerability indicators in traffic."""
        vulnerabilities = []
        
        # HTTP-based vulnerability indicators
        if 'http.request.uri' in df.columns:
            http_df = df[df['http.request.uri'].notna()]
            
            for _, row in http_df.iterrows():
                uri = str(row['http.request.uri'])
                
                # Check for common attack patterns
                vuln_indicators = self._check_http_vulnerabilities(uri, row)
                vulnerabilities.extend(vuln_indicators)
        
        # DNS-based indicators
        if 'dns.qry.name' in df.columns:
            dns_df = df[df['dns.qry.name'].notna()]
            
            for _, row in dns_df.iterrows():
                dns_name = str(row['dns.qry.name'])
                
                # Check for suspicious DNS queries
                dns_indicators = self._check_dns_vulnerabilities(dns_name, row)
                vulnerabilities.extend(dns_indicators)
        
        # Authentication-based indicators
        auth_indicators = self._check_auth_vulnerabilities(df)
        vulnerabilities.extend(auth_indicators)
        
        return vulnerabilities
    
    def _check_http_vulnerabilities(self, uri: str, row: pd.Series) -> List[Dict[str, Any]]:
        """Check HTTP requests for vulnerability patterns."""
        indicators = []
        
        # SQL Injection patterns
        sql_patterns = [
            r"('|(\\')|(;)|(\\;))", r"(\\x27)|(\\x2D)|(%27)|(%2D)",
            r"(union.*select)|(select.*from)|(insert.*into)",
            r"(drop.*table)|(update.*set)|(delete.*from)"
        ]
        
        for pattern in sql_patterns:
            if re.search(pattern, uri, re.IGNORECASE):
                indicators.append({
                    'type': 'sql_injection',
                    'severity': 'HIGH',
                    'description': f'Potential SQL injection in URI: {uri}',
                    'source_ip': row.get('ip.src', 'Unknown'),
                    'destination_ip': row.get('ip.dst', 'Unknown'),
                    'timestamp': row.get('frame.time_epoch', 'Unknown')
                })
                break
        
        # XSS patterns
        xss_patterns = [
            r"<script", r"javascript:", r"onload=", r"onerror=",
            r"alert\\(", r"document\\.cookie", r"<iframe"
        ]
        
        for pattern in xss_patterns:
            if re.search(pattern, uri, re.IGNORECASE):
                indicators.append({
                    'type': 'xss',
                    'severity': 'MEDIUM', 
                    'description': f'Potential XSS in URI: {uri}',
                    'source_ip': row.get('ip.src', 'Unknown'),
                    'destination_ip': row.get('ip.dst', 'Unknown'),
                    'timestamp': row.get('frame.time_epoch', 'Unknown')
                })
                break
        
        # Path traversal
        if '../' in uri or '..\\\\' in uri:
            indicators.append({
                'type': 'path_traversal',
                'severity': 'HIGH',
                'description': f'Potential path traversal in URI: {uri}',
                'source_ip': row.get('ip.src', 'Unknown'),
                'destination_ip': row.get('ip.dst', 'Unknown'),
                'timestamp': row.get('frame.time_epoch', 'Unknown')
            })
        
        return indicators
    
    def _check_dns_vulnerabilities(self, dns_name: str, row: pd.Series) -> List[Dict[str, Any]]:
        """Check DNS queries for suspicious patterns."""
        indicators = []
        
        # DNS tunneling indicators
        if len(dns_name) > 63:  # Unusually long DNS names
            indicators.append({
                'type': 'dns_tunneling',
                'severity': 'MEDIUM',
                'description': f'Unusually long DNS query (potential tunneling): {dns_name}',
                'source_ip': row.get('ip.src', 'Unknown'),
                'timestamp': row.get('frame.time_epoch', 'Unknown')
            })
        
        # Suspicious TLDs or patterns
        suspicious_patterns = [
            r'\\.[a-z0-9]{10,}$',  # Long random TLD
            r'\\.tk$', r'\\.ml$', r'\\.ga$',  # Common malware TLDs
            r'[0-9]{1,3}-[0-9]{1,3}-[0-9]{1,3}-[0-9]{1,3}'  # IP-like patterns
        ]
        
        for pattern in suspicious_patterns:
            if re.search(pattern, dns_name, re.IGNORECASE):
                indicators.append({
                    'type': 'suspicious_dns',
                    'severity': 'LOW',
                    'description': f'Suspicious DNS pattern: {dns_name}',
                    'source_ip': row.get('ip.src', 'Unknown'),
                    'timestamp': row.get('frame.time_epoch', 'Unknown')
                })
                break
        
        return indicators
    
    def _check_auth_vulnerabilities(self, df: pd.DataFrame) -> List[Dict[str, Any]]:
        """Check for authentication-related vulnerabilities."""
        indicators = []
        
        # Clear text authentication protocols
        if 'telnet.data' in df.columns:
            telnet_df = df[df['telnet.data'].notna()]
            if not telnet_df.empty:
                indicators.append({
                    'type': 'cleartext_protocol',
                    'severity': 'HIGH',
                    'description': 'Telnet traffic detected (cleartext authentication)',
                    'count': len(telnet_df)
                })
        
        if 'ftp.request.command' in df.columns:
            ftp_df = df[df['ftp.request.command'].notna()]
            if not ftp_df.empty:
                indicators.append({
                    'type': 'cleartext_protocol',
                    'severity': 'MEDIUM',
                    'description': 'FTP traffic detected (potential cleartext authentication)',
                    'count': len(ftp_df)
                })
        
        # HTTP Basic Auth
        if 'http.authorization' in df.columns:
            auth_df = df[df['http.authorization'].notna()]
            basic_auth = auth_df[auth_df['http.authorization'].str.startswith('Basic', na=False)]
            if not basic_auth.empty:
                indicators.append({
                    'type': 'weak_authentication',
                    'severity': 'MEDIUM',
                    'description': 'HTTP Basic Authentication detected',
                    'count': len(basic_auth)
                })
        
        return indicators
    
    def extract_http(self, pcap_path: str) -> Dict[str, Any]:
        """Extract only HTTP traffic from PCAP."""
        return self.analyze(pcap_path, filter_type="web")
    
    def extract_dns(self, pcap_path: str) -> Dict[str, Any]:
        """Extract only DNS traffic from PCAP.""" 
        return self.analyze(pcap_path, filter_type="suspicious")
    
    def filter_router_traffic(self, pcap_path: str) -> Dict[str, Any]:
        """Extract router-related traffic patterns."""
        return self.analyze(pcap_path, filter_type="router")
    
    def detect_suspicious_patterns(self, analysis_results: Dict[str, Any]) -> List[Dict[str, Any]]:
        """Extract suspicious patterns from analysis results."""
        if 'analysis_results' in analysis_results:
            return analysis_results['analysis_results'].get('vulnerabilities', [])
        return []