"""
PySploit Command Line Interface
Simple CLI for common PySploit operations
"""

import argparse
import sys
from pathlib import Path
from typing import Optional

import pysploit
from pysploit.utils.reports import ReportGenerator


def search_cves_command(args):
    """Search CVEs command"""
    print(f"Searching for CVEs matching: {args.query}")
    
    try:
        results = pysploit.search_cves(args.query, limit=args.limit)
        
        if not results:
            print("No CVEs found matching your query.")
            return
        
        print(f"\nFound {len(results)} CVEs:")
        print("-" * 80)
        
        for i, cve in enumerate(results, 1):
            print(f"{i}. {cve.get('id', 'N/A')}")
            print(f"   Score: {cve.get('base_score', 'N/A')}")
            print(f"   Severity: {cve.get('severity', 'N/A')}")
            print(f"   Description: {cve.get('description', 'No description')[:100]}...")
            if i < len(results):
                print()
        
    except Exception as e:
        print(f"Error searching CVEs: {e}")
        sys.exit(1)


def analyze_pcap_command(args):
    """Analyze PCAP command"""
    pcap_file = Path(args.pcap_file)
    
    if not pcap_file.exists():
        print(f"Error: PCAP file '{pcap_file}' not found.")
        sys.exit(1)
    
    print(f"Analyzing PCAP file: {pcap_file}")
    
    try:
        analyzer = pysploit.PcapAnalyzer()
        
        with analyzer:
            results = analyzer.analyze(str(pcap_file))
        
        print(f"\nAnalysis Results:")
        print(f"Total packets: {results.get('total_packets', 0)}")
        print(f"Suspicious packets: {len(results.get('suspicious_packets', []))}")
        print(f"Vulnerabilities detected: {len(results.get('vulnerabilities', []))}")
        
        # Show vulnerabilities
        vulnerabilities = results.get('vulnerabilities', [])
        if vulnerabilities:
            print(f"\nDetected Vulnerabilities:")
            print("-" * 40)
            for i, vuln in enumerate(vulnerabilities[:10], 1):  # Show first 10
                print(f"{i}. {vuln.get('type', 'Unknown')} - {vuln.get('description', 'No description')}")
        
        # Save results if requested
        if args.output:
            output_file = Path(args.output)
            generator = ReportGenerator()
            
            if output_file.suffix.lower() == '.json':
                report = generator.generate_json_report(results)
            elif output_file.suffix.lower() == '.html':
                report = generator.generate_html_report(results)
            elif output_file.suffix.lower() == '.csv':
                report = generator.generate_csv_report(results)
            else:
                report = generator.generate_json_report(results)
            
            generator.save_report(report, str(output_file))
            print(f"Results saved to: {output_file}")
        
    except Exception as e:
        print(f"Error analyzing PCAP: {e}")
        sys.exit(1)


def analyze_nmap_command(args):
    """Analyze Nmap XML command"""
    nmap_file = Path(args.nmap_file)
    
    if not nmap_file.exists():
        print(f"Error: Nmap file '{nmap_file}' not found.")
        sys.exit(1)
    
    print(f"Analyzing Nmap XML: {nmap_file}")
    
    try:
        analyzer = pysploit.NmapAnalyzer()
        results = analyzer.parse_xml(str(nmap_file))
        
        hosts = results.get('hosts', [])
        print(f"\nScan Results:")
        print(f"Hosts discovered: {len(hosts)}")
        
        # Show host information
        for i, host in enumerate(hosts[:5], 1):  # Show first 5 hosts
            print(f"\nHost {i}: {host.get('address', 'Unknown')}")
            print(f"  Status: {host.get('status', 'Unknown')}")
            print(f"  OS: {host.get('os', {}).get('name', 'Unknown')}")
            
            ports = host.get('ports', [])
            print(f"  Open ports: {len(ports)}")
            
            for port in ports[:5]:  # Show first 5 ports per host
                service = port.get('service', 'Unknown')
                version = port.get('version', '')
                print(f"    {port.get('port')}/{port.get('protocol')} - {service} {version}")
        
        # Save results if requested
        if args.output:
            output_file = Path(args.output)
            generator = ReportGenerator()
            
            if output_file.suffix.lower() == '.json':
                report = generator.generate_json_report(results)
            elif output_file.suffix.lower() == '.html':
                report = generator.generate_html_report(results)
            elif output_file.suffix.lower() == '.csv':
                report = generator.generate_csv_report(results)
            else:
                report = generator.generate_json_report(results)
            
            generator.save_report(report, str(output_file))
            print(f"Results saved to: {output_file}")
        
    except Exception as e:
        print(f"Error analyzing Nmap XML: {e}")
        sys.exit(1)


def quick_analyze_command(args):
    """Quick analysis command"""
    pcap_file = Path(args.pcap_file) if args.pcap_file else None
    nmap_file = Path(args.nmap_file) if args.nmap_file else None
    
    # Validate files exist
    if pcap_file and not pcap_file.exists():
        print(f"Error: PCAP file '{pcap_file}' not found.")
        sys.exit(1)
    
    if nmap_file and not nmap_file.exists():
        print(f"Error: Nmap file '{nmap_file}' not found.")
        sys.exit(1)
    
    if not pcap_file and not nmap_file:
        print("Error: At least one of --pcap or --nmap must be provided.")
        sys.exit(1)
    
    print("Performing quick security analysis...")
    
    try:
        results = pysploit.quick_analyze(
            pcap_file=str(pcap_file) if pcap_file else None,
            nmap_file=str(nmap_file) if nmap_file else None
        )
        
        print(f"\nQuick Analysis Results:")
        print(f"Network vulnerabilities: {len(results.get('network_vulns', []))}")
        print(f"Service vulnerabilities: {len(results.get('service_vulns', []))}")
        print(f"High-risk findings: {len(results.get('high_risk', []))}")
        
        # Show high-risk findings
        high_risk = results.get('high_risk', [])
        if high_risk:
            print(f"\nHigh-Risk Findings:")
            print("-" * 40)
            for i, finding in enumerate(high_risk[:5], 1):
                print(f"{i}. {finding.get('title', 'Unknown Risk')}")
                print(f"   Severity: {finding.get('severity', 'Unknown')}")
                print(f"   Description: {finding.get('description', 'No description')[:80]}...")
        
        # Save results if requested
        if args.output:
            output_file = Path(args.output)
            generator = ReportGenerator()
            
            if output_file.suffix.lower() == '.json':
                report = generator.generate_json_report(results)
            elif output_file.suffix.lower() == '.html':
                report = generator.generate_html_report(results)
            elif output_file.suffix.lower() == '.csv':
                report = generator.generate_csv_report(results)
            else:
                report = generator.generate_json_report(results)
            
            generator.save_report(report, str(output_file))
            print(f"Results saved to: {output_file}")
        
    except Exception as e:
        print(f"Error performing analysis: {e}")
        sys.exit(1)


def main():
    """Main CLI entry point"""
    parser = argparse.ArgumentParser(
        description="PySploit - Vulnerability Analysis Library CLI",
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Examples:
  pysploit search "buffer overflow" --limit 10
  pysploit pcap analyze traffic.pcap --output report.html
  pysploit nmap analyze scan.xml --output results.json
  pysploit quick --pcap traffic.pcap --nmap scan.xml --output assessment.html
        """
    )
    
    subparsers = parser.add_subparsers(dest='command', help='Available commands')
    
    # Search CVEs command
    search_parser = subparsers.add_parser('search', help='Search CVE database')
    search_parser.add_argument('query', help='Search query (CVE ID, keyword, etc.)')
    search_parser.add_argument('--limit', type=int, default=20, help='Maximum results to return')
    search_parser.set_defaults(func=search_cves_command)
    
    # PCAP analysis command
    pcap_parser = subparsers.add_parser('pcap', help='PCAP analysis commands')
    pcap_subparsers = pcap_parser.add_subparsers(dest='pcap_command')
    
    pcap_analyze_parser = pcap_subparsers.add_parser('analyze', help='Analyze PCAP file')
    pcap_analyze_parser.add_argument('pcap_file', help='Path to PCAP file')
    pcap_analyze_parser.add_argument('--output', '-o', help='Output file path')
    pcap_analyze_parser.set_defaults(func=analyze_pcap_command)
    
    # Nmap analysis command
    nmap_parser = subparsers.add_parser('nmap', help='Nmap analysis commands')
    nmap_subparsers = nmap_parser.add_subparsers(dest='nmap_command')
    
    nmap_analyze_parser = nmap_subparsers.add_parser('analyze', help='Analyze Nmap XML file')
    nmap_analyze_parser.add_argument('nmap_file', help='Path to Nmap XML file')
    nmap_analyze_parser.add_argument('--output', '-o', help='Output file path')
    nmap_analyze_parser.set_defaults(func=analyze_nmap_command)
    
    # Quick analysis command
    quick_parser = subparsers.add_parser('quick', help='Quick vulnerability analysis')
    quick_parser.add_argument('--pcap', dest='pcap_file', help='PCAP file to analyze')
    quick_parser.add_argument('--nmap', dest='nmap_file', help='Nmap XML file to analyze')
    quick_parser.add_argument('--output', '-o', help='Output file path')
    quick_parser.set_defaults(func=quick_analyze_command)
    
    # Parse arguments
    args = parser.parse_args()
    
    if not args.command:
        parser.print_help()
        sys.exit(1)
    
    # Handle subcommands that have their own subparsers
    if args.command == 'pcap' and not args.pcap_command:
        pcap_parser.print_help()
        sys.exit(1)
    
    if args.command == 'nmap' and not args.nmap_command:
        nmap_parser.print_help()
        sys.exit(1)
    
    # Execute command
    try:
        args.func(args)
    except KeyboardInterrupt:
        print("\nOperation cancelled by user.")
        sys.exit(130)
    except Exception as e:
        print(f"Unexpected error: {e}")
        sys.exit(1)


if __name__ == '__main__':
    main()