"""
Enhanced PySploit Vulnerability Data Loader
Loads comprehensive vulnerability data from Analysis/Vulnerability project
"""

import csv
import json
import os
from pathlib import Path

class ComprehensiveVulnerabilityData:
    """
    Comprehensive vulnerability database with data from multiple sources
    """
    
    def __init__(self):
        """Initialize comprehensive vulnerability data from collected sources"""
        
        # Critical vulnerabilities from our analysis
        self.critical_cves = [
            {
                'id': 'CVE-2021-44228',
                'description': 'Apache Log4j2 JNDI features do not protect against attacker controlled LDAP and other JNDI related endpoints',
                'base_score': 10.0,
                'severity': 'CRITICAL',
                'published': '2021-12-10',
                'vendors': ['Apache'],
                'products': ['Log4j'],
                'references': ['https://nvd.nist.gov/vuln/detail/CVE-2021-44228'],
                'source': 'NVD',
                'exploit_available': True
            },
            {
                'id': 'CVE-2020-1472',
                'description': 'Microsoft Windows Netlogon elevation of privilege vulnerability (Zerologon)',
                'base_score': 10.0,
                'severity': 'CRITICAL', 
                'published': '2020-09-11',
                'vendors': ['Microsoft'],
                'products': ['Windows Server'],
                'references': ['https://nvd.nist.gov/vuln/detail/CVE-2020-1472'],
                'source': 'NVD',
                'exploit_available': True
            },
            {
                'id': 'CVE-2019-19781',
                'description': 'Citrix Application Delivery Controller and Citrix Gateway remote code execution',
                'base_score': 9.8,
                'severity': 'CRITICAL',
                'published': '2019-12-17',
                'vendors': ['Citrix'],
                'products': ['Application Delivery Controller', 'Gateway'],
                'references': ['https://nvd.nist.gov/vuln/detail/CVE-2019-19781'],
                'source': 'NVD',
                'exploit_available': True
            },
            {
                'id': 'CVE-2021-34527',
                'description': 'Windows Print Spooler Remote Code Execution Vulnerability (PrintNightmare)',
                'base_score': 8.8,
                'severity': 'HIGH',
                'published': '2021-07-01',
                'vendors': ['Microsoft'],
                'products': ['Windows'],
                'references': ['https://nvd.nist.gov/vuln/detail/CVE-2021-34527'],
                'source': 'NVD',
                'exploit_available': True
            },
            {
                'id': 'CVE-2021-26855',
                'description': 'Microsoft Exchange Server Remote Code Execution Vulnerability',
                'base_score': 9.8,
                'severity': 'CRITICAL',
                'published': '2021-03-02',
                'vendors': ['Microsoft'],
                'products': ['Exchange Server'],
                'references': ['https://nvd.nist.gov/vuln/detail/CVE-2021-26855'],
                'source': 'NVD',
                'exploit_available': True
            },
            {
                'id': 'CVE-2022-0778',
                'description': 'OpenSSL infinite loop vulnerability in BN_mod_sqrt() function',
                'base_score': 7.5,
                'severity': 'HIGH',
                'published': '2022-03-15',
                'vendors': ['OpenSSL'],
                'products': ['OpenSSL'],
                'references': ['https://nvd.nist.gov/vuln/detail/CVE-2022-0778'],
                'source': 'NVD',
                'exploit_available': False
            }
        ]
        
        # RouterSploit vulnerabilities from Analysis/Vulnerability
        self.routersploit_vulnerabilities = [
            {
                'id': 'ROUTERSPLOIT-001',
                'module_name': 'linksys_eseries_themoon_rce',
                'description': 'Linksys E-Series TheMoon malware RCE via tmUnblock.cgi',
                'vendor': 'Linksys',
                'devices': ['E-Series (E900, E1000, E1200, E1500, E1550, E2000, E2100L, E2500, E3000, E3200, E4200)'],
                'cve_references': ['CVE-2014-2321'],
                'vulnerability_type': 'Remote Code Execution',
                'severity': 'CRITICAL',
                'base_score': 9.8,
                'category': 'router_exploit',
                'target_ports': [80, 8080],
                'fingerprint_patterns': ['tmUnblock.cgi', 'Linksys E-Series'],
                'source': 'RouterSploit'
            },
            {
                'id': 'ROUTERSPLOIT-002',
                'module_name': 'netgear_r7000_r6400_rce',
                'description': 'Netgear R7000 and R6400 remote command execution vulnerability',
                'vendor': 'Netgear',
                'devices': ['R7000', 'R6400'],
                'cve_references': ['CVE-2016-6277'],
                'vulnerability_type': 'Remote Code Execution',
                'severity': 'CRITICAL',
                'base_score': 9.8,
                'category': 'router_exploit',
                'target_ports': [80, 8080],
                'fingerprint_patterns': ['Netgear', 'R7000', 'R6400'],
                'source': 'RouterSploit'
            },
            {
                'id': 'ROUTERSPLOIT-003',
                'module_name': 'cisco_ios_http_authorization_bypass',
                'description': 'Cisco IOS HTTP server authentication bypass allowing admin access',
                'vendor': 'Cisco',
                'devices': ['IOS 11.3-12.2'],
                'cve_references': ['CVE-2001-0537'],
                'vulnerability_type': 'Authentication Bypass',
                'severity': 'HIGH',
                'base_score': 8.1,
                'category': 'router_exploit',
                'target_ports': [80, 8080],
                'fingerprint_patterns': ['Cisco-IOS', 'IOS'],
                'source': 'RouterSploit'
            },
            {
                'id': 'ROUTERSPLOIT-004',
                'module_name': 'zte_zxv10_rce',
                'description': 'ZTE ZXV10 H108L remote code execution vulnerability',
                'vendor': 'ZTE',
                'devices': ['ZXV10 H108L'],
                'cve_references': ['CVE-2014-2321'],
                'vulnerability_type': 'Remote Code Execution',
                'severity': 'CRITICAL',
                'base_score': 9.8,
                'category': 'router_exploit',
                'target_ports': [80, 8080],
                'fingerprint_patterns': ['ZTE', 'ZXV10', 'H108L'],
                'source': 'RouterSploit'
            },
            {
                'id': 'ROUTERSPLOIT-005',
                'module_name': 'huawei_hg532_rce',
                'description': 'Huawei HG532 series routers remote code execution vulnerability',
                'vendor': 'Huawei',
                'devices': ['HG532'],
                'cve_references': ['CVE-2017-17215'],
                'vulnerability_type': 'Remote Code Execution',
                'severity': 'CRITICAL',
                'base_score': 9.8,
                'category': 'router_exploit',
                'target_ports': [37215],
                'fingerprint_patterns': ['Huawei', 'HG532'],
                'source': 'RouterSploit'
            },
            {
                'id': 'ROUTERSPLOIT-006',
                'module_name': 'mikrotik_routeros_jailbreak',
                'description': 'MikroTik RouterOS jailbreak vulnerability allowing arbitrary code execution',
                'vendor': 'MikroTik',
                'devices': ['RouterOS'],
                'cve_references': ['CVE-2018-14847'],
                'vulnerability_type': 'Remote Code Execution',
                'severity': 'CRITICAL',
                'base_score': 9.8,
                'category': 'router_exploit',
                'target_ports': [8291],
                'fingerprint_patterns': ['MikroTik', 'RouterOS'],
                'source': 'RouterSploit'
            },
            {
                'id': 'ROUTERSPLOIT-007',
                'module_name': 'zyxel_d1000_rce',
                'description': 'ZyXEL/Eir D1000 SOAP remote code execution vulnerability',
                'vendor': 'ZyXEL',
                'devices': ['EIR D1000'],
                'cve_references': ['CVE-2016-10401'],
                'vulnerability_type': 'Remote Code Execution',
                'severity': 'CRITICAL',
                'base_score': 9.8,
                'category': 'router_exploit',
                'target_ports': [7547],
                'fingerprint_patterns': ['ZyXEL', 'D1000'],
                'source': 'RouterSploit'
            }
        ]
        
        # High-impact NVD vulnerabilities from our analysis
        self.nvd_high_impact = [
            {
                'id': 'CVE-2010-3765',
                'description': 'Mozilla Firefox memory corruption via nsCSSFrameConstructor::ContentAppended',
                'base_score': 9.8,
                'severity': 'CRITICAL',
                'published': '2010-10-28',
                'vendors': ['Mozilla'],
                'products': ['Firefox', 'Thunderbird', 'SeaMonkey'],
                'references': ['https://nvd.nist.gov/vuln/detail/CVE-2010-3765'],
                'source': 'NVD'
            },
            {
                'id': 'CVE-2013-3918',
                'description': 'Microsoft InformationCardSigninHelper ActiveX control vulnerability',
                'base_score': 8.8,
                'severity': 'HIGH',
                'published': '2013-11-12',
                'vendors': ['Microsoft'],
                'products': ['Windows', 'Internet Explorer'],
                'references': ['https://nvd.nist.gov/vuln/detail/CVE-2013-3918'],
                'source': 'NVD'
            },
            {
                'id': 'CVE-2014-0781',
                'description': 'Yokogawa CENTUM CS 3000 BKCLogSvr.exe heap buffer overflow',
                'base_score': 9.8,
                'severity': 'CRITICAL',
                'published': '2014-03-14',
                'vendors': ['Yokogawa'],
                'products': ['CENTUM CS 3000'],
                'references': ['https://nvd.nist.gov/vuln/detail/CVE-2014-0781'],
                'source': 'NVD'
            }
        ]
        
        # ExploitDB critical entries
        self.exploitdb_critical = [
            {
                'id': 'EDB-51854',
                'description': 'GL.iNet AR300M v3.216 Remote Code Execution - CVE-2023-46456 Exploit',
                'author': 'cyberaz0r',
                'type': 'remote',
                'platform': 'hardware',
                'cve_references': ['CVE-2023-46456'],
                'severity': 'CRITICAL',
                'base_score': 9.8,
                'published': '2023',
                'source': 'ExploitDB'
            },
            {
                'id': 'EDB-51851',
                'description': 'GL.iNet AR300M v4.3.7 Arbitrary File Read - CVE-2023-46455 Exploit',
                'author': 'cyberaz0r',
                'type': 'remote',
                'platform': 'hardware',
                'cve_references': ['CVE-2023-46455'],
                'severity': 'HIGH',
                'base_score': 7.5,
                'published': '2023',
                'source': 'ExploitDB'
            },
            {
                'id': 'EDB-51852',
                'description': 'GL.iNet AR300M v4.3.7 Remote Code Execution - CVE-2023-46454 Exploit',
                'author': 'cyberaz0r',
                'type': 'remote',
                'platform': 'hardware',
                'cve_references': ['CVE-2023-46454'],
                'severity': 'CRITICAL',
                'base_score': 9.8,
                'published': '2023',
                'source': 'ExploitDB'
            }
        ]
        
        # Vulnerability patterns for detection
        self.vulnerability_patterns = {
            'web_vulnerabilities': [
                {'pattern': r'(?i)sql.*(injection|inject)', 'severity': 'HIGH', 'category': 'sql_injection'},
                {'pattern': r'(?i)cross.?site.?script', 'severity': 'MEDIUM', 'category': 'xss'},
                {'pattern': r'(?i)buffer.?overflow', 'severity': 'HIGH', 'category': 'buffer_overflow'},
                {'pattern': r'(?i)remote.?code.?execution', 'severity': 'CRITICAL', 'category': 'rce'},
                {'pattern': r'(?i)privilege.?escalation', 'severity': 'HIGH', 'category': 'privilege_escalation'},
                {'pattern': r'(?i)directory.?traversal', 'severity': 'MEDIUM', 'category': 'directory_traversal'},
            ],
            'network_vulnerabilities': [
                {'pattern': r'(?i)denial.?of.?service', 'severity': 'HIGH', 'category': 'dos'},
                {'pattern': r'(?i)man.?in.?the.?middle', 'severity': 'HIGH', 'category': 'mitm'},
                {'pattern': r'(?i)authentication.?bypass', 'severity': 'HIGH', 'category': 'auth_bypass'},
                {'pattern': r'(?i)information.?disclosure', 'severity': 'MEDIUM', 'category': 'info_disclosure'},
            ],
            'malware_signatures': [
                {'pattern': r'(?i)themoon', 'severity': 'CRITICAL', 'category': 'malware'},
                {'pattern': r'(?i)mirai', 'severity': 'CRITICAL', 'category': 'botnet'},
                {'pattern': r'(?i)gafgyt', 'severity': 'CRITICAL', 'category': 'botnet'},
                {'pattern': r'(?i)bashlite', 'severity': 'CRITICAL', 'category': 'botnet'},
            ]
        }
    
    def get_all_vulnerabilities(self):
        """Get all vulnerabilities from all sources (deduplicated)"""
        all_vulns = []
        seen_ids = set()
        
        # Add CVEs first (highest priority)
        for vuln in self.critical_cves:
            if vuln['id'] not in seen_ids:
                all_vulns.append(vuln)
                seen_ids.add(vuln['id'])
        
        # Add NVD high impact
        for vuln in self.nvd_high_impact:
            if vuln['id'] not in seen_ids:
                all_vulns.append(vuln)
                seen_ids.add(vuln['id'])
        
        # Convert ExploitDB format
        for edb in self.exploitdb_critical:
            if edb['id'] not in seen_ids:
                vuln = {
                    'id': edb['id'],
                    'description': edb['description'],
                    'base_score': edb['base_score'],
                    'severity': edb['severity'],
                    'published': edb['published'],
                    'vendors': ['Multiple'],  # ExploitDB doesn't always specify vendors
                    'products': [edb['platform']],
                    'references': [],
                    'source': edb['source']
                }
                all_vulns.append(vuln)
                seen_ids.add(edb['id'])
        
        # Convert RouterSploit format
        for rs in self.routersploit_vulnerabilities:
            if rs['id'] not in seen_ids:
                vuln = {
                    'id': rs['id'],
                    'description': rs['description'],
                    'base_score': rs['base_score'],
                    'severity': rs['severity'],
                    'published': '2024-01-01',  # Default date for RouterSploit
                    'vendors': [rs['vendor']],
                    'products': rs['devices'],
                    'references': [],
                    'source': rs['source']
                }
                all_vulns.append(vuln)
                seen_ids.add(rs['id'])
        
        return all_vulns
    
    def get_vulnerability_count(self):
        """Get total vulnerability count"""
        return (len(self.critical_cves) + 
                len(self.routersploit_vulnerabilities) + 
                len(self.nvd_high_impact) + 
                len(self.exploitdb_critical))