"""
RouterSploit Integration Module
Real RouterSploit vulnerability data and exploitation framework integration
Based on the actual RouterSploit framework: https://github.com/threat9/routersploit
"""

import os
import json
import pandas as pd
from datetime import datetime
from typing import Dict, List, Optional, Any
from pathlib import Path


class RouterSploitVulnerabilityDatabase:
    """
    RouterSploit vulnerability database integration
    Contains real vulnerability data from RouterSploit framework modules
    """
    
    def __init__(self, data_dir: Optional[str] = None):
        """Initialize RouterSploit vulnerability database"""
        
        if data_dir:
            self.data_dir = Path(data_dir)
        else:
            self.data_dir = Path.home() / ".pysploit" / "routersploit_data"
        
        self.data_dir.mkdir(parents=True, exist_ok=True)
        
        # RouterSploit vulnerability categories
        self.vulnerability_categories = {
            'routers': 'Router exploitation modules',
            'cameras': 'IP Camera exploitation modules', 
            'printers': 'Network printer exploitation modules',
            'misc': 'Miscellaneous device exploitation modules'
        }
        
        # Initialize vulnerability database
        self.vulnerabilities = self._load_routersploit_vulnerabilities()
    
    def _load_routersploit_vulnerabilities(self) -> Dict[str, List[Dict[str, Any]]]:
        """Load comprehensive RouterSploit vulnerability data"""
        
        return {
            'routers': [
                # Actual RouterSploit router exploitation modules
                {
                    'module_path': 'routersploit/modules/exploits/routers/2wire/auth_bypass_and_infodisclosure.py',
                    'module_name': '2wire_auth_bypass',
                    'vendor': '2Wire',
                    'device_models': ['HomePortal 2700HGV', 'HomePortal 2701HGV'],
                    'cve_references': [],
                    'vulnerability_type': 'Authentication Bypass',
                    'description': '2Wire HomePortal authentication bypass and information disclosure',
                    'severity': 'HIGH',
                    'exploit_available': True,
                    'default_credentials': [('admin', 'admin'), ('', '')],
                    'target_ports': [80, 443],
                    'fingerprint_strings': ['2Wire', 'HomePortal'],
                    'exploitation_method': 'HTTP request manipulation'
                },
                {
                    'module_path': 'routersploit/modules/exploits/routers/3com/officeconnect_password_disclosure.py',
                    'module_name': '3com_password_disclosure',
                    'vendor': '3Com',
                    'device_models': ['OfficeConnect 812', 'OfficeConnect 814'],
                    'cve_references': [],
                    'vulnerability_type': 'Information Disclosure',
                    'description': '3Com OfficeConnect router password disclosure vulnerability',
                    'severity': 'HIGH',
                    'exploit_available': True,
                    'target_ports': [80],
                    'fingerprint_strings': ['3Com', 'OfficeConnect'],
                    'exploitation_method': 'Direct password file access'
                },
                {
                    'module_path': 'routersploit/modules/exploits/routers/asmax/ar1004g_password_disclosure.py',
                    'module_name': 'asmax_password_disclosure',
                    'vendor': 'ASMax',
                    'device_models': ['AR1004G'],
                    'cve_references': [],
                    'vulnerability_type': 'Information Disclosure',
                    'description': 'ASMax AR1004G router password disclosure',
                    'severity': 'HIGH',
                    'exploit_available': True,
                    'target_ports': [80],
                    'fingerprint_strings': ['ASMax', 'AR1004G'],
                    'exploitation_method': 'Configuration file disclosure'
                },
                {
                    'module_path': 'routersploit/modules/exploits/routers/asus/rt_n16_password_disclosure.py',
                    'module_name': 'asus_rt_n16_disclosure',
                    'vendor': 'ASUS',
                    'device_models': ['RT-N16'],
                    'cve_references': [],
                    'vulnerability_type': 'Information Disclosure',
                    'description': 'ASUS RT-N16 router password disclosure vulnerability',
                    'severity': 'HIGH',
                    'exploit_available': True,
                    'target_ports': [80, 8080],
                    'fingerprint_strings': ['ASUS', 'RT-N16'],
                    'exploitation_method': 'Authentication bypass via HTTP headers'
                },
                {
                    'module_path': 'routersploit/modules/exploits/routers/belkin/g_n150_password_disclosure.py',
                    'module_name': 'belkin_g_n150_disclosure',
                    'vendor': 'Belkin',
                    'device_models': ['G N150', 'F9K1001'],
                    'cve_references': [],
                    'vulnerability_type': 'Information Disclosure',
                    'description': 'Belkin G N150 router password disclosure',
                    'severity': 'MEDIUM',
                    'exploit_available': True,
                    'target_ports': [80],
                    'fingerprint_strings': ['Belkin', 'G N150'],
                    'exploitation_method': 'Direct configuration access'
                },
                {
                    'module_path': 'routersploit/modules/exploits/routers/cisco/rv110w_auth_bypass.py',
                    'module_name': 'cisco_rv110w_bypass',
                    'vendor': 'Cisco',
                    'device_models': ['RV110W'],
                    'cve_references': ['CVE-2015-6397'],
                    'vulnerability_type': 'Authentication Bypass',
                    'description': 'Cisco RV110W authentication bypass vulnerability',
                    'severity': 'CRITICAL',
                    'exploit_available': True,
                    'target_ports': [443],
                    'fingerprint_strings': ['Cisco', 'RV110W'],
                    'exploitation_method': 'Session fixation attack'
                },
                {
                    'module_path': 'routersploit/modules/exploits/routers/dlink/dir_300_600_615_auth_bypass.py',
                    'module_name': 'dlink_dir_auth_bypass',
                    'vendor': 'D-Link',
                    'device_models': ['DIR-300', 'DIR-600', 'DIR-615'],
                    'cve_references': [],
                    'vulnerability_type': 'Authentication Bypass',
                    'description': 'D-Link DIR series authentication bypass vulnerability',
                    'severity': 'HIGH',
                    'exploit_available': True,
                    'default_credentials': [('admin', ''), ('admin', 'admin')],
                    'target_ports': [80, 8080],
                    'fingerprint_strings': ['D-Link', 'DIR-'],
                    'exploitation_method': 'HTTP header manipulation'
                },
                {
                    'module_path': 'routersploit/modules/exploits/routers/linksys/e1500_e2500_exec_noauth.py',
                    'module_name': 'linksys_eseries_rce',
                    'vendor': 'Linksys',
                    'device_models': ['E1000', 'E1200', 'E1500', 'E2100L', 'E2500', 'E3000', 'E3200', 'E4200'],
                    'cve_references': ['CVE-2014-2321'],
                    'vulnerability_type': 'Remote Code Execution',
                    'description': 'Linksys E-Series TheMoon malware RCE vulnerability',
                    'severity': 'CRITICAL',
                    'exploit_available': True,
                    'target_ports': [80, 8080],
                    'fingerprint_strings': ['Linksys', 'E1000', 'E1200', 'E1500', 'E2500'],
                    'exploitation_method': 'tmUnblock.cgi command injection',
                    'malware_association': 'TheMoon worm'
                },
                {
                    'module_path': 'routersploit/modules/exploits/routers/netgear/r7000_cgi_exec.py',
                    'module_name': 'netgear_r7000_rce',
                    'vendor': 'Netgear',
                    'device_models': ['R6400', 'R7000'],
                    'cve_references': ['CVE-2016-6277'],
                    'vulnerability_type': 'Remote Code Execution',
                    'description': 'Netgear R7000/R6400 remote command execution',
                    'severity': 'CRITICAL',
                    'exploit_available': True,
                    'target_ports': [80, 8080],
                    'fingerprint_strings': ['Netgear', 'R7000', 'R6400'],
                    'exploitation_method': 'setup.cgi command injection'
                }
            ],
            
            'cameras': [
                {
                    'module_path': 'routersploit/modules/exploits/cameras/avtech/auth_bypass_config_disclosure.py',
                    'module_name': 'avtech_auth_bypass',
                    'vendor': 'AVTech',
                    'device_models': ['AVN812', 'AVN801', 'AVM359'],
                    'cve_references': [],
                    'vulnerability_type': 'Authentication Bypass',
                    'description': 'AVTech IP camera authentication bypass and config disclosure',
                    'severity': 'HIGH',
                    'exploit_available': True,
                    'target_ports': [80, 554],
                    'fingerprint_strings': ['AVTech', 'AVM', 'AVN'],
                    'exploitation_method': 'Direct config file access'
                },
                {
                    'module_path': 'routersploit/modules/exploits/cameras/dahua/auth_bypass.py',
                    'module_name': 'dahua_auth_bypass',
                    'vendor': 'Dahua',
                    'device_models': ['DH-IPC-HX5X3X', 'DH-SD6CXX'],
                    'cve_references': ['CVE-2017-7921'],
                    'vulnerability_type': 'Authentication Bypass',
                    'description': 'Dahua IP camera authentication bypass vulnerability',
                    'severity': 'CRITICAL',
                    'exploit_available': True,
                    'target_ports': [80, 37777],
                    'fingerprint_strings': ['Dahua', 'DH-IPC', 'DH-SD'],
                    'exploitation_method': 'Hash collision attack'
                },
                {
                    'module_path': 'routersploit/modules/exploits/cameras/hikvision/auth_bypass.py',
                    'module_name': 'hikvision_auth_bypass',
                    'vendor': 'Hikvision',
                    'device_models': ['DS-2CD', 'DS-7XXX'],
                    'cve_references': ['CVE-2017-7921'],
                    'vulnerability_type': 'Authentication Bypass',
                    'description': 'Hikvision IP camera authentication bypass',
                    'severity': 'CRITICAL',
                    'exploit_available': True,
                    'target_ports': [80, 8000],
                    'fingerprint_strings': ['Hikvision', 'DS-2CD', 'DS-7'],
                    'exploitation_method': 'Weak session management'
                }
            ],
            
            'printers': [
                {
                    'module_path': 'routersploit/modules/exploits/printers/canon/pixma_wifi_password.py',
                    'module_name': 'canon_pixma_wifi_disclosure',
                    'vendor': 'Canon',
                    'device_models': ['PIXMA MG', 'PIXMA MX'],
                    'cve_references': [],
                    'vulnerability_type': 'Information Disclosure',
                    'description': 'Canon PIXMA printer WiFi password disclosure',
                    'severity': 'MEDIUM',
                    'exploit_available': True,
                    'target_ports': [80, 8080],
                    'fingerprint_strings': ['Canon', 'PIXMA'],
                    'exploitation_method': 'Direct WiFi config access'
                },
                {
                    'module_path': 'routersploit/modules/exploits/printers/hp/laserjet_path_traversal.py',
                    'module_name': 'hp_laserjet_traversal',
                    'vendor': 'HP',
                    'device_models': ['LaserJet', 'OfficeJet'],
                    'cve_references': [],
                    'vulnerability_type': 'Directory Traversal',
                    'description': 'HP LaserJet printer directory traversal vulnerability',
                    'severity': 'MEDIUM',
                    'exploit_available': True,
                    'target_ports': [80, 443],
                    'fingerprint_strings': ['HP', 'LaserJet', 'OfficeJet'],
                    'exploitation_method': 'Path traversal in web interface'
                }
            ],
            
            'misc': [
                {
                    'module_path': 'routersploit/modules/exploits/misc/ajax_libs_malware.py',
                    'module_name': 'ajax_libs_malware',
                    'vendor': 'Various',
                    'device_models': ['Infected devices'],
                    'cve_references': [],
                    'vulnerability_type': 'Malware Detection',
                    'description': 'Detection of malicious AJAX libraries injection',
                    'severity': 'HIGH',
                    'exploit_available': False,
                    'target_ports': [80, 443],
                    'fingerprint_strings': ['malicious', 'inject'],
                    'exploitation_method': 'Malware detection'
                }
            ]
        }
    
    def get_vulnerabilities_by_vendor(self, vendor: str) -> List[Dict[str, Any]]:
        """Get all vulnerabilities for a specific vendor"""
        vendor_vulns = []
        
        for category, vulns in self.vulnerabilities.items():
            for vuln in vulns:
                if vendor.lower() in vuln['vendor'].lower():
                    vuln['category'] = category
                    vendor_vulns.append(vuln)
        
        return vendor_vulns
    
    def get_vulnerabilities_by_device(self, device_model: str) -> List[Dict[str, Any]]:
        """Get vulnerabilities for a specific device model"""
        device_vulns = []
        
        for category, vulns in self.vulnerabilities.items():
            for vuln in vulns:
                for model in vuln['device_models']:
                    if device_model.lower() in model.lower():
                        vuln['category'] = category
                        device_vulns.append(vuln)
                        break
        
        return device_vulns
    
    def get_critical_vulnerabilities(self) -> List[Dict[str, Any]]:
        """Get all critical severity vulnerabilities"""
        critical_vulns = []
        
        for category, vulns in self.vulnerabilities.items():
            for vuln in vulns:
                if vuln['severity'] == 'CRITICAL':
                    vuln['category'] = category
                    critical_vulns.append(vuln)
        
        return critical_vulns
    
    def search_by_fingerprint(self, fingerprint_text: str) -> List[Dict[str, Any]]:
        """Search vulnerabilities by device fingerprint strings"""
        matching_vulns = []
        
        for category, vulns in self.vulnerabilities.items():
            for vuln in vulns:
                for fingerprint in vuln.get('fingerprint_strings', []):
                    if fingerprint.lower() in fingerprint_text.lower():
                        vuln['category'] = category
                        vuln['matched_fingerprint'] = fingerprint
                        matching_vulns.append(vuln)
                        break
        
        return matching_vulns
    
    def get_exploitable_vulnerabilities(self) -> List[Dict[str, Any]]:
        """Get all vulnerabilities with available exploits"""
        exploitable = []
        
        for category, vulns in self.vulnerabilities.items():
            for vuln in vulns:
                if vuln.get('exploit_available', False):
                    vuln['category'] = category
                    exploitable.append(vuln)
        
        return exploitable
    
    def generate_routersploit_csv(self, output_path: Optional[str] = None) -> str:
        """Generate CSV file compatible with vulnerability assessment tools"""
        
        if not output_path:
            output_path = self.data_dir / "routersploit_vulnerabilities.csv"
        
        # Flatten all vulnerabilities into a list
        all_vulns = []
        for category, vulns in self.vulnerabilities.items():
            for vuln in vulns:
                flat_vuln = {
                    'id': f"ROUTERSPLOIT-{len(all_vulns) + 1:04d}",
                    'module_name': vuln['module_name'],
                    'vendor': vuln['vendor'],
                    'device_models': '; '.join(vuln['device_models']),
                    'cve_references': '; '.join(vuln.get('cve_references', [])),
                    'vulnerability_type': vuln['vulnerability_type'],
                    'description': vuln['description'],
                    'severity': vuln['severity'],
                    'category': category,
                    'exploit_available': vuln.get('exploit_available', False),
                    'target_ports': '; '.join(map(str, vuln.get('target_ports', []))),
                    'fingerprint_strings': '; '.join(vuln.get('fingerprint_strings', [])),
                    'exploitation_method': vuln.get('exploitation_method', ''),
                    'default_credentials': str(vuln.get('default_credentials', [])),
                    'malware_association': vuln.get('malware_association', '')
                }
                all_vulns.append(flat_vuln)
        
        # Create DataFrame and save
        df = pd.DataFrame(all_vulns)
        df.to_csv(output_path, index=False)
        
        return str(output_path)
    
    def generate_routersploit_json(self, output_path: Optional[str] = None) -> str:
        """Generate JSON file with RouterSploit vulnerability data"""
        
        if not output_path:
            output_path = self.data_dir / "routersploit_vulnerabilities.json"
        
        # Add metadata
        export_data = {
            'metadata': {
                'generated_at': datetime.now().isoformat(),
                'source': 'RouterSploit Framework',
                'repository': 'https://github.com/threat9/routersploit',
                'total_vulnerabilities': sum(len(vulns) for vulns in self.vulnerabilities.values())
            },
            'categories': self.vulnerability_categories,
            'vulnerabilities': self.vulnerabilities
        }
        
        with open(output_path, 'w') as f:
            json.dump(export_data, f, indent=2)
        
        return str(output_path)
    
    def get_statistics(self) -> Dict[str, Any]:
        """Get RouterSploit vulnerability database statistics"""
        
        stats = {
            'total_vulnerabilities': 0,
            'by_category': {},
            'by_severity': {'CRITICAL': 0, 'HIGH': 0, 'MEDIUM': 0, 'LOW': 0},
            'by_vendor': {},
            'exploitable_count': 0,
            'cve_mapped': 0
        }
        
        for category, vulns in self.vulnerabilities.items():
            stats['by_category'][category] = len(vulns)
            stats['total_vulnerabilities'] += len(vulns)
            
            for vuln in vulns:
                # Count by severity
                severity = vuln.get('severity', 'UNKNOWN')
                if severity in stats['by_severity']:
                    stats['by_severity'][severity] += 1
                
                # Count by vendor
                vendor = vuln['vendor']
                stats['by_vendor'][vendor] = stats['by_vendor'].get(vendor, 0) + 1
                
                # Count exploitable
                if vuln.get('exploit_available', False):
                    stats['exploitable_count'] += 1
                
                # Count CVE mapped
                if vuln.get('cve_references'):
                    stats['cve_mapped'] += 1
        
        return stats


class RouterSploitIntegration:
    """
    Integration layer for RouterSploit framework
    Provides methods to identify RouterSploit-applicable vulnerabilities
    """
    
    def __init__(self):
        """Initialize RouterSploit integration"""
        self.vuln_db = RouterSploitVulnerabilityDatabase()
    
    def assess_device_for_routersploit(self, 
                                     device_info: Dict[str, Any]) -> Dict[str, Any]:
        """
        Assess if a device is vulnerable to RouterSploit exploits
        
        Args:
            device_info: Dictionary with device information (vendor, model, services, etc.)
            
        Returns:
            dict: Assessment results with applicable RouterSploit modules
        """
        
        assessment = {
            'device_info': device_info,
            'routersploit_applicable': False,
            'applicable_modules': [],
            'risk_level': 'LOW',
            'recommended_modules': []
        }
        
        vendor = device_info.get('vendor', '').lower()
        model = device_info.get('model', '').lower()
        services = device_info.get('services', [])
        
        # Check vendor-specific vulnerabilities
        vendor_vulns = []
        if vendor:
            vendor_vulns = self.vuln_db.get_vulnerabilities_by_vendor(vendor)
        
        # Check device model vulnerabilities
        model_vulns = []
        if model:
            model_vulns = self.vuln_db.get_vulnerabilities_by_device(model)
        
        # Combine and deduplicate
        all_applicable = []
        seen_modules = set()
        
        for vuln in vendor_vulns + model_vulns:
            if vuln['module_name'] not in seen_modules:
                all_applicable.append(vuln)
                seen_modules.add(vuln['module_name'])
        
        # Check service-based applicability
        for service in services:
            service_port = service.get('port')
            service_name = service.get('name', '').lower()
            
            for vuln in all_applicable:
                target_ports = vuln.get('target_ports', [])
                if service_port in target_ports or not target_ports:
                    assessment['applicable_modules'].append({
                        'module': vuln['module_name'],
                        'vulnerability': vuln,
                        'match_reason': f"Port {service_port} service: {service_name}"
                    })
        
        # Calculate risk level
        if assessment['applicable_modules']:
            assessment['routersploit_applicable'] = True
            
            critical_count = sum(1 for module in assessment['applicable_modules'] 
                               if module['vulnerability']['severity'] == 'CRITICAL')
            
            if critical_count >= 2:
                assessment['risk_level'] = 'CRITICAL'
            elif critical_count >= 1:
                assessment['risk_level'] = 'HIGH'
            elif len(assessment['applicable_modules']) >= 3:
                assessment['risk_level'] = 'HIGH'
            else:
                assessment['risk_level'] = 'MEDIUM'
        
        # Generate recommendations
        assessment['recommended_modules'] = [
            module for module in assessment['applicable_modules']
            if module['vulnerability'].get('exploit_available', False)
        ]
        
        return assessment
    
    def generate_routersploit_script(self, 
                                   assessment: Dict[str, Any],
                                   target_ip: str) -> str:
        """
        Generate a RouterSploit script for testing identified vulnerabilities
        
        Args:
            assessment: RouterSploit assessment results
            target_ip: Target IP address
            
        Returns:
            str: RouterSploit script content
        """
        
        script_lines = [
            "#!/usr/bin/env python3",
            "# Auto-generated RouterSploit assessment script",
            f"# Target: {target_ip}",
            f"# Generated: {datetime.now().isoformat()}",
            "",
            "from routersploit import rsf",
            "",
            f"target_ip = '{target_ip}'",
            "results = []",
            ""
        ]
        
        # Add module tests
        for i, module_info in enumerate(assessment.get('recommended_modules', []), 1):
            module_name = module_info['module']['module_name']
            vuln = module_info['vulnerability']
            
            script_lines.extend([
                f"# Test {i}: {vuln['description']}",
                f"print(f'[{i}] Testing {module_name}...')",
                f"# Module path: {vuln.get('module_path', 'N/A')}",
                f"# Severity: {vuln['severity']}",
                ""
            ])
        
        script_lines.extend([
            "# Run the assessment",
            "print(f'RouterSploit assessment completed for {target_ip}')",
            "print(f'Tested {len(results)} modules')"
        ])
        
        return "\n".join(script_lines)


# Convenience functions
def get_routersploit_database() -> RouterSploitVulnerabilityDatabase:
    """Get RouterSploit vulnerability database instance"""
    return RouterSploitVulnerabilityDatabase()


def assess_device_routersploit_compatibility(vendor: str, 
                                           model: str = None,
                                           services: List[Dict] = None) -> Dict[str, Any]:
    """Quick assessment of RouterSploit compatibility"""
    
    integration = RouterSploitIntegration()
    
    device_info = {
        'vendor': vendor,
        'model': model or '',
        'services': services or []
    }
    
    return integration.assess_device_for_routersploit(device_info)