"""
PySploit Comprehensive Vulnerability Database
Advanced vulnerability database with integrated CVE, ExploitDB, NVD, and RouterSploit data
"""

import sqlite3
import json
import os
from datetime import datetime
from pathlib import Path
from typing import Dict, List, Optional, Any

# Import comprehensive vulnerability data
try:
    from ..data.comprehensive_vulns_massive import MassiveVulnerabilityData, MASSIVE_DATA_AVAILABLE
    COMPREHENSIVE_DATA_AVAILABLE = MASSIVE_DATA_AVAILABLE
    DATA_SOURCE = "massive"
except ImportError:
    try:
        from ..data.comprehensive_vulns import ComprehensiveVulnerabilityData
        COMPREHENSIVE_DATA_AVAILABLE = True
        DATA_SOURCE = "comprehensive"
    except ImportError:
        COMPREHENSIVE_DATA_AVAILABLE = False
        DATA_SOURCE = "basic"
        print("Warning: Comprehensive vulnerability data not available, using basic vulnerability data")


class EmbeddedVulnerabilityData:
    """
    Advanced vulnerability database with comprehensive security intelligence
    Contains 55,000+ vulnerabilities from CVE, ExploitDB, NVD, and RouterSploit sources
    """
    
    def __init__(self):
        """Initialize embedded vulnerability data"""
        
        # Load data based on availability
        if COMPREHENSIVE_DATA_AVAILABLE and DATA_SOURCE == "massive":
            # Load comprehensive vulnerability dataset
            print("[+] Loading comprehensive vulnerability dataset...")
            self.comprehensive_data = MassiveVulnerabilityData()
            self.embedded_cves = self.comprehensive_data.get_all_vulnerabilities()
            self.embedded_routersploit = self.comprehensive_data.routersploit_vulnerabilities
            self.vulnerability_patterns = getattr(self.comprehensive_data, 'vulnerability_patterns', 
                                                 self._get_massive_patterns())
            stats = self.comprehensive_data.get_loading_statistics()
            print(f"[+] Loaded {self.comprehensive_data.get_vulnerability_count()} vulnerabilities from comprehensive dataset")
            print(f"    Sources: ExploitDB({stats['sources']['ExploitDB']}), NVD({stats['sources']['NVD']}), RouterSploit({stats['sources']['RouterSploit']}), Index({stats['sources']['Index']})")
        elif COMPREHENSIVE_DATA_AVAILABLE and DATA_SOURCE == "comprehensive":
            # Load standard comprehensive data
            self.comprehensive_data = ComprehensiveVulnerabilityData()
            self.embedded_cves = self.comprehensive_data.get_all_vulnerabilities()
            self.embedded_routersploit = self.comprehensive_data.routersploit_vulnerabilities
            self.vulnerability_patterns = self.comprehensive_data.vulnerability_patterns
            print(f"[+] Loaded {self.comprehensive_data.get_vulnerability_count()} vulnerabilities from comprehensive database")
        else:
            # Fallback to basic embedded data
            self._load_basic_data()
    
    def _load_basic_data(self):
        """Load basic embedded data as fallback"""
        
        # Essential CVE data (high-impact vulnerabilities)
        self.embedded_cves = [
            {
                'id': 'CVE-2021-44228',
                'description': 'Apache Log4j2 JNDI features do not protect against attacker controlled LDAP and other JNDI related endpoints',
                'base_score': 10.0,
                'severity': 'CRITICAL',
                'published': '2021-12-10',
                'vendors': ['Apache'],
                'products': ['Log4j'],
                'references': ['https://nvd.nist.gov/vuln/detail/CVE-2021-44228'],
                'source': 'NVD'
            },
            {
                'id': 'CVE-2021-4034',
                'description': 'A local privilege escalation vulnerability was found on polkit\'s pkexec utility',
                'base_score': 7.8,
                'severity': 'HIGH',
                'published': '2022-01-28',
                'vendors': ['Polkit Project'],
                'products': ['Polkit'],
                'references': ['https://nvd.nist.gov/vuln/detail/CVE-2021-4034'],
                'source': 'NVD'
            },
            {
                'id': 'CVE-2014-2321',
                'description': 'Linksys E-Series routers vulnerable to TheMoon worm via tmUnblock.cgi',
                'base_score': 9.8,
                'severity': 'CRITICAL',
                'published': '2014-03-20',
                'vendors': ['Linksys'],
                'products': ['E1000', 'E1200', 'E1500', 'E2500'],
                'references': ['https://nvd.nist.gov/vuln/detail/CVE-2014-2321'],
                'source': 'NVD'
            },
            {
                'id': 'CVE-2016-6277',
                'description': 'Netgear R7000 and R6400 remote command execution via setup.cgi',
                'base_score': 9.8,
                'severity': 'CRITICAL',
                'published': '2016-08-17',
                'vendors': ['Netgear'],
                'products': ['R7000', 'R6400'],
                'references': ['https://nvd.nist.gov/vuln/detail/CVE-2016-6277'],
                'source': 'NVD'
            },
            {
                'id': 'CVE-2017-7921',
                'description': 'Hikvision and Dahua IP cameras authentication bypass vulnerability',
                'base_score': 9.8,
                'severity': 'CRITICAL',
                'published': '2017-04-24',
                'vendors': ['Hikvision', 'Dahua'],
                'products': ['IP Cameras'],
                'references': ['https://nvd.nist.gov/vuln/detail/CVE-2017-7921'],
                'source': 'NVD'
            },
            {
                'id': 'CVE-2019-19781',
                'description': 'Citrix Application Delivery Controller and Gateway directory traversal vulnerability',
                'base_score': 9.8,
                'severity': 'CRITICAL',
                'published': '2019-12-17',
                'vendors': ['Citrix'],
                'products': ['Application Delivery Controller', 'Gateway', 'SD-WAN WANOP'],
                'references': ['https://nvd.nist.gov/vuln/detail/CVE-2019-19781'],
                'source': 'NVD'
            },
            {
                'id': 'CVE-2020-1472',
                'description': 'Microsoft Windows Netlogon elevation of privilege vulnerability (Zerologon)',
                'base_score': 10.0,
                'severity': 'CRITICAL',
                'published': '2020-08-17',
                'vendors': ['Microsoft'],
                'products': ['Windows Server'],
                'references': ['https://nvd.nist.gov/vuln/detail/CVE-2020-1472'],
                'source': 'NVD'
            },
            {
                'id': 'CVE-2021-34527',
                'description': 'Windows Print Spooler remote code execution vulnerability (PrintNightmare)',
                'base_score': 8.8,
                'severity': 'HIGH',
                'published': '2021-07-02',
                'vendors': ['Microsoft'],
                'products': ['Windows'],
                'references': ['https://nvd.nist.gov/vuln/detail/CVE-2021-34527'],
                'source': 'NVD'
            },
            {
                'id': 'CVE-2021-26855',
                'description': 'Microsoft Exchange Server remote code execution vulnerability',
                'base_score': 9.8,
                'severity': 'CRITICAL',
                'published': '2021-03-02',
                'vendors': ['Microsoft'],
                'products': ['Exchange Server'],
                'references': ['https://nvd.nist.gov/vuln/detail/CVE-2021-26855'],
                'source': 'NVD'
            },
            {
                'id': 'CVE-2022-0778',
                'description': 'OpenSSL infinite loop in BN_mod_sqrt() reachable when parsing certificates',
                'base_score': 7.5,
                'severity': 'HIGH',
                'published': '2022-03-15',
                'vendors': ['OpenSSL Project'],
                'products': ['OpenSSL'],
                'references': ['https://nvd.nist.gov/vuln/detail/CVE-2022-0778'],
                'source': 'NVD'
            }
        ]
        
        # RouterSploit vulnerabilities (embedded)
        self.embedded_routersploit = [
            {
                'id': 'ROUTERSPLOIT-001',
                'module_name': 'linksys_eseries_rce',
                'vendor': 'Linksys',
                'devices': ['E1000', 'E1200', 'E1500', 'E2500', 'E3000'],
                'cve_references': ['CVE-2014-2321'],
                'description': 'Linksys E-Series TheMoon malware RCE via tmUnblock.cgi',
                'base_score': 9.8,
                'severity': 'CRITICAL',
                'category': 'router',
                'target_ports': [80, 8080],
                'fingerprint_patterns': ['linksys', 'e1000', 'e1200', 'e1500'],
                'source': 'RouterSploit'
            },
            {
                'id': 'ROUTERSPLOIT-002',
                'module_name': 'netgear_r7000_rce',
                'vendor': 'Netgear',
                'devices': ['R7000', 'R6400'],
                'cve_references': ['CVE-2016-6277'],
                'description': 'Netgear R7000/R6400 command injection via setup.cgi',
                'base_score': 9.8,
                'severity': 'CRITICAL',
                'category': 'router',
                'target_ports': [80, 8080],
                'fingerprint_patterns': ['netgear', 'r7000', 'r6400'],
                'source': 'RouterSploit'
            },
            {
                'id': 'ROUTERSPLOIT-003',
                'module_name': 'dlink_dir_auth_bypass',
                'vendor': 'D-Link',
                'devices': ['DIR-300', 'DIR-600', 'DIR-615', 'DIR-645'],
                'cve_references': [],
                'description': 'D-Link DIR series authentication bypass vulnerability',
                'base_score': 8.5,
                'severity': 'HIGH',
                'category': 'router',
                'target_ports': [80],
                'fingerprint_patterns': ['d-link', 'dir-'],
                'source': 'RouterSploit'
            },
            {
                'id': 'ROUTERSPLOIT-004',
                'module_name': 'hikvision_auth_bypass',
                'vendor': 'Hikvision',
                'devices': ['DS-2CD', 'DS-7XXX'],
                'cve_references': ['CVE-2017-7921'],
                'description': 'Hikvision IP camera authentication bypass',
                'base_score': 9.8,
                'severity': 'CRITICAL',
                'category': 'camera',
                'target_ports': [80, 8000],
                'fingerprint_patterns': ['hikvision', 'ds-2cd'],
                'source': 'RouterSploit'
            },
            {
                'id': 'ROUTERSPLOIT-005',
                'module_name': 'dahua_auth_bypass',
                'vendor': 'Dahua',
                'devices': ['DH-IPC', 'DH-SD'],
                'cve_references': ['CVE-2017-7921'],
                'description': 'Dahua IP camera authentication bypass vulnerability',
                'base_score': 9.8,
                'severity': 'CRITICAL',
                'category': 'camera',
                'target_ports': [80, 37777],
                'fingerprint_patterns': ['dahua', 'dh-ipc'],
                'source': 'RouterSploit'
            },
            {
                'id': 'ROUTERSPLOIT-006',
                'module_name': 'cisco_rv110w_bypass',
                'vendor': 'Cisco',
                'devices': ['RV110W'],
                'cve_references': ['CVE-2015-6397'],
                'description': 'Cisco RV110W authentication bypass vulnerability',
                'base_score': 9.1,
                'severity': 'CRITICAL',
                'category': 'router',
                'target_ports': [443],
                'fingerprint_patterns': ['cisco', 'rv110w'],
                'source': 'RouterSploit'
            },
            {
                'id': 'ROUTERSPLOIT-007',
                'module_name': 'asus_rt_disclosure',
                'vendor': 'ASUS',
                'devices': ['RT-N16', 'RT-AC', 'RT-AX'],
                'cve_references': [],
                'description': 'ASUS router password disclosure vulnerability',
                'base_score': 7.5,
                'severity': 'HIGH',
                'category': 'router',
                'target_ports': [80, 8080],
                'fingerprint_patterns': ['asus', 'rt-'],
                'source': 'RouterSploit'
            }
        ]
        
        # Common vulnerability patterns for traffic analysis
        self.vulnerability_patterns = {
            'web_attacks': [
                {'pattern': r'\.\./', 'type': 'Directory Traversal', 'severity': 'HIGH'},
                {'pattern': r'<script', 'type': 'XSS Attack', 'severity': 'MEDIUM'},
                {'pattern': r'union.*select', 'type': 'SQL Injection', 'severity': 'HIGH'},
                {'pattern': r'/etc/passwd', 'type': 'File Inclusion', 'severity': 'HIGH'},
                {'pattern': r'cmd\.exe|/bin/sh', 'type': 'Command Injection', 'severity': 'CRITICAL'}
            ],
            'router_attacks': [
                {'pattern': r'tmUnblock\.cgi', 'type': 'TheMoon Malware', 'severity': 'CRITICAL'},
                {'pattern': r'setup\.cgi.*submit_button', 'type': 'Netgear RCE', 'severity': 'CRITICAL'},
                {'pattern': r'/cgi-bin/.*\.cgi', 'type': 'CGI Exploitation', 'severity': 'HIGH'}
            ],
            'iot_attacks': [
                {'pattern': r'mirai|gafgyt|bashlite', 'type': 'IoT Botnet', 'severity': 'CRITICAL'},
                {'pattern': r'default.*password', 'type': 'Default Credentials', 'severity': 'MEDIUM'}
            ]
        }
    
    def _get_massive_patterns(self):
        """Get vulnerability patterns for massive dataset"""
        from ..data.comprehensive_vulns_massive import MASSIVE_VULNERABILITY_PATTERNS
        return MASSIVE_VULNERABILITY_PATTERNS


class EmbeddedVulnerabilityDatabase:
    """
    Advanced vulnerability database with comprehensive security intelligence
    Features high-performance in-memory SQLite database with advanced indexing
    """
    
    def __init__(self):
        """Initialize embedded database"""
        self.embedded_data = EmbeddedVulnerabilityData()
        
        # Create in-memory SQLite database
        self.conn = sqlite3.connect(':memory:')
        self.conn.row_factory = sqlite3.Row
        
        # Initialize database schema
        self._create_tables()
        
        # Load embedded data
        self._load_embedded_data()
        
        print("[+] PySploit vulnerability database initialized successfully")
    
    def _create_tables(self):
        """Create database tables"""
        cursor = self.conn.cursor()
        
        # Main vulnerabilities table
        cursor.execute('''
            CREATE TABLE vulnerabilities (
                id TEXT PRIMARY KEY,
                cve_id TEXT,
                description TEXT,
                base_score REAL,
                severity TEXT,
                published_date TEXT,
                vendors TEXT,
                products TEXT,
                ref_links TEXT,
                source TEXT,
                category TEXT,
                target_ports TEXT,
                fingerprint_patterns TEXT
            )
        ''')
        
        # Vulnerability patterns table
        cursor.execute('''
            CREATE TABLE vulnerability_patterns (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                pattern TEXT,
                pattern_type TEXT,
                severity TEXT,
                category TEXT
            )
        ''')
        
        # Create indexes for massive dataset performance
        cursor.execute('CREATE INDEX idx_cve_id ON vulnerabilities(cve_id)')
        cursor.execute('CREATE INDEX idx_severity ON vulnerabilities(severity)')
        cursor.execute('CREATE INDEX idx_vendor ON vulnerabilities(vendors)')
        cursor.execute('CREATE INDEX idx_source ON vulnerabilities(source)')
        cursor.execute('CREATE INDEX idx_category ON vulnerabilities(category)')
        cursor.execute('CREATE INDEX idx_description_fts ON vulnerabilities(description)')
        
        # Pattern indexes
        cursor.execute('CREATE INDEX idx_pattern_type ON vulnerability_patterns(pattern_type)')
        cursor.execute('CREATE INDEX idx_pattern_severity ON vulnerability_patterns(severity)')
        
        self.conn.commit()
    
    def _load_embedded_data(self):
        """Load embedded vulnerability data into database with batch processing"""
        cursor = self.conn.cursor()
        
        # Optimize for large datasets
        cursor.execute('PRAGMA synchronous = OFF')
        cursor.execute('PRAGMA journal_mode = MEMORY') 
        cursor.execute('PRAGMA cache_size = 10000')
        
        print(f"[+] Loading {len(self.embedded_data.embedded_cves)} vulnerabilities...")
        
        # Batch load CVE data for performance
        batch_size = 1000
        cve_data = []
        
        for i, cve in enumerate(self.embedded_data.embedded_cves):
            cve_row = (
                cve['id'],
                cve.get('id', ''), 
                cve.get('description', ''),
                cve.get('base_score', 0.0),
                cve.get('severity', 'MEDIUM'),
                cve.get('published_date', cve.get('published', '')),
                json.dumps(cve.get('vendors', [])),
                json.dumps(cve.get('products', [])),
                json.dumps(cve.get('references', [])),
                cve.get('source', 'Unknown'),
                cve.get('category', 'general')
            )
            cve_data.append(cve_row)
            
            # Batch insert for performance
            if len(cve_data) >= batch_size:
                cursor.executemany('''
                    INSERT OR REPLACE INTO vulnerabilities 
                    (id, cve_id, description, base_score, severity, published_date, 
                     vendors, products, ref_links, source, category)
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
                ''', cve_data)
                cve_data = []
                
                if (i + 1) % 5000 == 0:
                    print(f"    Processed {i + 1} vulnerabilities...")
        
        # Insert remaining CVE data
        if cve_data:
            cursor.executemany('''
                INSERT OR REPLACE INTO vulnerabilities 
                (id, cve_id, description, base_score, severity, published_date, 
                 vendors, products, ref_links, source, category)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
            ''', cve_data)
        
        # Load RouterSploit data with batch processing
        rs_data = []
        for rs in self.embedded_data.embedded_routersploit:
            rs_row = (
                rs.get('id', ''),
                ','.join(rs.get('cve_references', [])) if rs.get('cve_references') else '',
                rs.get('description', ''),
                rs.get('base_score', 0.0),
                rs.get('severity', 'HIGH'),
                datetime.now().date().isoformat(),
                json.dumps([rs.get('vendor', '')]) if rs.get('vendor') else json.dumps([]),
                json.dumps(rs.get('devices', [])) if 'devices' in rs else json.dumps([rs.get('device', '')]),
                json.dumps([]),
                rs.get('source', 'RouterSploit'),
                rs.get('category', 'router_exploit'),
                json.dumps(rs.get('target_ports', [])),
                json.dumps(rs.get('fingerprint_patterns', []))
            )
            rs_data.append(rs_row)
        
        if rs_data:
            cursor.executemany('''
                INSERT OR REPLACE INTO vulnerabilities 
                (id, cve_id, description, base_score, severity, published_date,
                 vendors, products, ref_links, source, category, target_ports, fingerprint_patterns)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
            ''', rs_data)
        
        # Load vulnerability patterns
        for category, patterns in self.embedded_data.vulnerability_patterns.items():
            for pattern in patterns:
                cursor.execute('''
                    INSERT INTO vulnerability_patterns (pattern, pattern_type, severity, category)
                    VALUES (?, ?, ?, ?)
                ''', (
                    pattern['pattern'],
                    pattern['category'], 
                    pattern['severity'],
                    category
                ))
        
        self.conn.commit()
        print(f"[+] Database integration complete: {len(self.embedded_data.embedded_cves)} vulnerabilities and {len(self.embedded_data.embedded_routersploit)} security modules")
    
    def search_by_cve(self, cve_id: str) -> List[Dict[str, Any]]:
        """Search by CVE ID"""
        cursor = self.conn.cursor()
        cursor.execute('SELECT * FROM vulnerabilities WHERE cve_id = ? OR cve_id LIKE ?', 
                      (cve_id, f'%{cve_id}%'))
        
        results = cursor.fetchall()
        return [dict(row) for row in results]
    
    def search_by_vendor(self, vendor: str) -> List[Dict[str, Any]]:
        """Search by vendor"""
        cursor = self.conn.cursor()
        cursor.execute('SELECT * FROM vulnerabilities WHERE vendors LIKE ?', (f'%{vendor}%',))
        
        results = cursor.fetchall()
        return [dict(row) for row in results]
    
    def search_by_severity(self, severity: str) -> List[Dict[str, Any]]:
        """Search by severity level"""
        cursor = self.conn.cursor()
        cursor.execute('SELECT * FROM vulnerabilities WHERE severity = ?', (severity.upper(),))
        
        results = cursor.fetchall()
        return [dict(row) for row in results]
    
    def search_routersploit(self, device_info: str = None) -> List[Dict[str, Any]]:
        """Search RouterSploit vulnerabilities"""
        cursor = self.conn.cursor()
        
        if device_info:
            cursor.execute('''
                SELECT * FROM vulnerabilities 
                WHERE source = "RouterSploit" 
                AND (vendors LIKE ? OR products LIKE ? OR fingerprint_patterns LIKE ?)
            ''', (f'%{device_info}%', f'%{device_info}%', f'%{device_info}%'))
        else:
            cursor.execute('SELECT * FROM vulnerabilities WHERE source = "RouterSploit"')
        
        results = cursor.fetchall()
        return [dict(row) for row in results]
    
    def get_critical_vulnerabilities(self) -> List[Dict[str, Any]]:
        """Get all critical vulnerabilities"""
        cursor = self.conn.cursor()
        cursor.execute('SELECT * FROM vulnerabilities WHERE severity = "CRITICAL" ORDER BY base_score DESC')
        
        results = cursor.fetchall()
        return [dict(row) for row in results]
    
    def get_vulnerability_patterns(self, category: str = None) -> List[Dict[str, Any]]:
        """Get vulnerability detection patterns"""
        cursor = self.conn.cursor()
        
        if category:
            cursor.execute('SELECT * FROM vulnerability_patterns WHERE category = ?', (category,))
        else:
            cursor.execute('SELECT * FROM vulnerability_patterns')
        
        results = cursor.fetchall()
        return [dict(row) for row in results]
    
    def get_statistics(self) -> Dict[str, Any]:
        """Get database statistics"""
        cursor = self.conn.cursor()
        
        # Total vulnerabilities
        cursor.execute('SELECT COUNT(*) FROM vulnerabilities')
        total_vulns = cursor.fetchone()[0]
        
        # By severity
        cursor.execute('SELECT severity, COUNT(*) FROM vulnerabilities GROUP BY severity')
        severity_counts = dict(cursor.fetchall())
        
        # By source
        cursor.execute('SELECT source, COUNT(*) FROM vulnerabilities GROUP BY source')
        source_counts = dict(cursor.fetchall())
        
        return {
            'total_vulnerabilities': total_vulns,
            'by_severity': severity_counts,
            'by_source': source_counts,
            'database_type': 'Embedded',
            'requires_internet': False
        }
    
    def close(self):
        """Close database connection"""
        if self.conn:
            self.conn.close()
    
    def __enter__(self):
        """Context manager entry"""
        return self
    
    def __exit__(self, exc_type, exc_val, exc_tb):
        """Context manager exit"""
        self.close()


# Global instance for embedded database
_embedded_db = None

def get_embedded_database() -> EmbeddedVulnerabilityDatabase:
    """Get or create embedded database instance"""
    global _embedded_db
    if _embedded_db is None:
        _embedded_db = EmbeddedVulnerabilityDatabase()
    return _embedded_db