#!/usr/bin/env python3
"""
PySploit Report Generator
Generate various report formats from vulnerability analysis results.
"""

import json
import pandas as pd
from datetime import datetime
from typing import Dict, List, Any, Optional


def generate_report(matches: List[Dict[str, Any]], format: str = "json", 
                   output_path: Optional[str] = None) -> str:
    """
    Generate vulnerability assessment report.
    
    Args:
        matches (list): Vulnerability matches from matcher
        format (str): Output format ('json', 'csv', 'html', 'txt')
        output_path (str, optional): Output file path
    
    Returns:
        str: Report content or file path if saved
    """
    generator = ReportGenerator()
    
    if format.lower() == 'json':
        return generator.generate_json_report(matches, output_path)
    elif format.lower() == 'csv':
        return generator.generate_csv_report(matches, output_path)
    elif format.lower() == 'html':
        return generator.generate_html_report(matches, output_path)
    elif format.lower() == 'txt':
        return generator.generate_text_report(matches, output_path)
    else:
        raise ValueError(f"Unsupported format: {format}")


class ReportGenerator:
    """Generate vulnerability assessment reports in various formats."""
    
    def generate_json_report(self, matches: List[Dict[str, Any]], 
                           output_path: Optional[str] = None) -> str:
        """Generate JSON format report."""
        report = {
            'metadata': {
                'generated_at': datetime.now().isoformat(),
                'total_matches': len(matches),
                'generator': 'PySploit v1.0.0'
            },
            'summary': self._generate_summary(matches),
            'matches': matches
        }
        
        json_content = json.dumps(report, indent=2, default=str)
        
        if output_path:
            with open(output_path, 'w') as f:
                f.write(json_content)
            return output_path
        
        return json_content
    
    def generate_csv_report(self, matches: List[Dict[str, Any]], 
                          output_path: Optional[str] = None) -> str:
        """Generate CSV format report."""
        if not matches:
            return ""
        
        # Flatten matches for CSV format
        flattened = []
        for match in matches:
            flat_match = self._flatten_match(match)
            flattened.append(flat_match)
        
        df = pd.DataFrame(flattened)
        
        if output_path:
            df.to_csv(output_path, index=False)
            return output_path
        
        return df.to_csv(index=False)
    
    def generate_html_report(self, matches: List[Dict[str, Any]], 
                           output_path: Optional[str] = None) -> str:
        """Generate HTML format report."""
        summary = self._generate_summary(matches)
        
        html_content = f"""
<!DOCTYPE html>
<html>
<head>
    <title>PySploit Vulnerability Assessment Report</title>
    <style>
        body {{ font-family: Arial, sans-serif; margin: 20px; }}
        .header {{ background-color: #f0f0f0; padding: 20px; border-radius: 5px; }}
        .summary {{ margin: 20px 0; }}
        .match {{ border: 1px solid #ddd; margin: 10px 0; padding: 15px; border-radius: 5px; }}
        .severity-HIGH {{ border-left: 5px solid #ff4444; }}
        .severity-MEDIUM {{ border-left: 5px solid #ffaa00; }}
        .severity-LOW {{ border-left: 5px solid #44ff44; }}
        .severity-CRITICAL {{ border-left: 5px solid #880000; }}
        table {{ border-collapse: collapse; width: 100%; }}
        th, td {{ border: 1px solid #ddd; padding: 8px; text-align: left; }}
        th {{ background-color: #f2f2f2; }}
    </style>
</head>
<body>
    <div class="header">
        <h1>PySploit Vulnerability Assessment Report</h1>
        <p><strong>Generated:</strong> {datetime.now().strftime('%Y-%m-%d %H:%M:%S')}</p>
        <p><strong>Total Matches:</strong> {len(matches)}</p>
    </div>
    
    <div class="summary">
        <h2>Summary</h2>
        <table>
            <tr><th>Metric</th><th>Value</th></tr>
            <tr><td>Total Vulnerabilities</td><td>{summary.get('total_vulnerabilities', 0)}</td></tr>
            <tr><td>Critical Severity</td><td>{summary.get('by_severity', {}).get('CRITICAL', 0)}</td></tr>
            <tr><td>High Severity</td><td>{summary.get('by_severity', {}).get('HIGH', 0)}</td></tr>
            <tr><td>Medium Severity</td><td>{summary.get('by_severity', {}).get('MEDIUM', 0)}</td></tr>
            <tr><td>Low Severity</td><td>{summary.get('by_severity', {}).get('LOW', 0)}</td></tr>
        </table>
    </div>
    
    <div class="matches">
        <h2>Vulnerability Matches</h2>
        {self._generate_html_matches(matches)}
    </div>
</body>
</html>
        """
        
        if output_path:
            with open(output_path, 'w') as f:
                f.write(html_content)
            return output_path
        
        return html_content
    
    def generate_text_report(self, matches: List[Dict[str, Any]], 
                           output_path: Optional[str] = None) -> str:
        """Generate plain text format report."""
        summary = self._generate_summary(matches)
        
        text_content = f"""
PySploit Vulnerability Assessment Report
=======================================

Generated: {datetime.now().strftime('%Y-%m-%d %H:%M:%S')}
Total Matches: {len(matches)}

Summary:
--------
Total Vulnerabilities: {summary.get('total_vulnerabilities', 0)}
Critical Severity: {summary.get('by_severity', {}).get('CRITICAL', 0)}
High Severity: {summary.get('by_severity', {}).get('HIGH', 0)}
Medium Severity: {summary.get('by_severity', {}).get('MEDIUM', 0)}
Low Severity: {summary.get('by_severity', {}).get('LOW', 0)}

Vulnerability Matches:
---------------------
{self._generate_text_matches(matches)}
        """
        
        if output_path:
            with open(output_path, 'w') as f:
                f.write(text_content)
            return output_path
        
        return text_content
    
    def _generate_summary(self, matches: List[Dict[str, Any]]) -> Dict[str, Any]:
        """Generate summary statistics from matches."""
        summary = {
            'total_vulnerabilities': len(matches),
            'by_severity': {},
            'by_match_type': {},
            'by_confidence': {}
        }
        
        for match in matches:
            # Count by severity
            severity = match.get('severity', 'UNKNOWN')
            summary['by_severity'][severity] = summary['by_severity'].get(severity, 0) + 1
            
            # Count by match type
            match_type = match.get('match_type', 'unknown')
            summary['by_match_type'][match_type] = summary['by_match_type'].get(match_type, 0) + 1
            
            # Count by confidence
            confidence = match.get('confidence', 'unknown')
            summary['by_confidence'][confidence] = summary['by_confidence'].get(confidence, 0) + 1
        
        return summary
    
    def _flatten_match(self, match: Dict[str, Any]) -> Dict[str, Any]:
        """Flatten nested match data for CSV export."""
        flat = {}
        
        # Basic match info
        flat['match_type'] = match.get('match_type', '')
        flat['confidence'] = match.get('confidence', '')
        flat['severity'] = match.get('severity', '')
        
        # Host information
        flat['host'] = match.get('host', match.get('source_ip', ''))
        flat['port'] = match.get('port', '')
        flat['service'] = match.get('service', '')
        
        # Vulnerability information
        vuln = match.get('vulnerability', {})
        if vuln:
            flat['cve_id'] = vuln.get('cve_id', '')
            flat['vulnerability_description'] = vuln.get('description', '')
            flat['cvss_score'] = vuln.get('cvss_score', '')
            flat['source'] = vuln.get('source', '')
        
        # Additional context
        flat['pattern'] = match.get('pattern', '')
        flat['protocol'] = match.get('protocol', '')
        
        return flat
    
    def _generate_html_matches(self, matches: List[Dict[str, Any]]) -> str:
        """Generate HTML for vulnerability matches."""
        html_matches = []
        
        for i, match in enumerate(matches):
            severity = match.get('severity', 'UNKNOWN')
            match_type = match.get('match_type', 'unknown')
            confidence = match.get('confidence', 'unknown')
            
            vuln = match.get('vulnerability', {})
            cve_id = vuln.get('cve_id', 'N/A')
            description = vuln.get('description', 'No description available')[:200]
            
            html_match = f"""
            <div class="match severity-{severity}">
                <h3>Match #{i+1} - {match_type.replace('_', ' ').title()}</h3>
                <p><strong>Severity:</strong> {severity}</p>
                <p><strong>Confidence:</strong> {confidence}</p>
                <p><strong>CVE ID:</strong> {cve_id}</p>
                <p><strong>Description:</strong> {description}</p>
                <p><strong>Host:</strong> {match.get('host', match.get('source_ip', 'N/A'))}</p>
                <p><strong>Port:</strong> {match.get('port', 'N/A')}</p>
            </div>
            """
            html_matches.append(html_match)
        
        return '\\n'.join(html_matches)
    
    def _generate_text_matches(self, matches: List[Dict[str, Any]]) -> str:
        """Generate plain text for vulnerability matches."""
        text_matches = []
        
        for i, match in enumerate(matches):
            severity = match.get('severity', 'UNKNOWN')
            match_type = match.get('match_type', 'unknown')
            confidence = match.get('confidence', 'unknown')
            
            vuln = match.get('vulnerability', {})
            cve_id = vuln.get('cve_id', 'N/A')
            description = vuln.get('description', 'No description available')[:150]
            
            text_match = f"""
Match #{i+1}: {match_type.replace('_', ' ').title()}
  Severity: {severity}
  Confidence: {confidence}
  CVE ID: {cve_id}
  Description: {description}
  Host: {match.get('host', match.get('source_ip', 'N/A'))}
  Port: {match.get('port', 'N/A')}
            """
            text_matches.append(text_match)
        
        return '\\n'.join(text_matches)