# Chaturbate Events – Copilot Instructions

## Overview

Async Python wrapper for Chaturbate Events API with real-time event streaming, strong typing, and rate limiting.

---

## Core Components

- **EventClient**: Async context manager for API polling with retry logic and rate limiting
- **EventRouter**: Decorator-based event handling (`@router.on`, `@router.on_any`)
- **Models**: Pydantic-based with `Event`, `EventType`, `User`, `Message`, `Tip`, `RoomSubject`

---

## Development Standards

### Code Style (Google Python Style Guide)

- **Type hints**: Required on all functions/methods with return types
- **Docstrings**: Google style with Args/Returns/Raises for all public APIs
- **Constants**: Module-level constants need inline documentation
- **Performance**: Use `__slots__` for performance-critical classes
- **Arguments**: Prefer keyword-only args for functions with >2 parameters
- **Type hints**: Use `Self` for methods returning instance (not string annotation)

### Testing

- **Framework**: pytest with asyncio, aioresponses for mocking
- **Philosophy**: Modify tests to match code changes, not the reverse

### File Management

- **Never create**: IMPROVEMENTS.md, CHANGES.md, NOTES.md, TODO.md, ROADMAP.md, or summary documents
- **CHANGELOG.md**: Auto-generated by semantic-release - never edit manually
- **Work directly**: Edit source files, tests, and docs - no intermediate documents

### Tool Usage

- **Always use tools**: Use run_in_terminal, replace_string_in_file, runTests - never just suggest commands
- **Validate**: Run get_errors after editing, runTests after changes

### Commit Messages

Follow conventional commits with imperative mood and specific verbs:

- `feat(client):` - New features
- `fix(router):` - Bug fixes
- `docs(README):` - Documentation
- `tests(client):` - Test updates
- `chore(deps):` - Dependencies

**Good verbs**: add, remove, replace, rename, move, extract, inline, merge, split, convert
**Bad verbs**: update, improve, enhance, refine, optimize, tweak, clean up

Examples:
- ✅ `fix(client): add AuthError for 401 responses`
- ❌ `fix(client): improve error handling`
- ✅ `chore(deps): bump aiohttp from 3.9.0 to 3.9.1`
- ❌ `chore(deps): update dependencies`

---

## Usage Pattern

```python
from cb_events import EventClient, EventRouter, EventType, Event

router = EventRouter()

@router.on(EventType.TIP)
async def handle_tip(event: Event) -> None:
    if event.tip and event.user:
        print(f"{event.user.username}: {event.tip.tokens} tokens")

async with EventClient(username, token, config=config) as client:
    async for event in client:
        await router.dispatch(event)
```

---

## Key Details

- **Python**: >=3.11 required
- **Dependencies**: aiohttp, aiohttp-retry, aiolimiter, pydantic
- **Rate limit**: 2000 req/min built-in
- **Security**: Token masking (last 4 chars only)
- **Environment**: `CB_USERNAME`, `CB_TOKEN` for credentials
