#!/usr/bin/env python3
from nicegui import ui
from biblemategui import config, BIBLEMATEGUI_DATA, BIBLEMATEGUI_APP_DIR, USER_DEFAULT_SETTINGS
from biblemategui.pages.home import *
from biblemategui.js.tooltip import TOOLTIP_JS
from biblemategui.css.tooltip import get_tooltip_css
import os

def get_tooltip_data(word):
    """Fetch tooltip data from database"""
    import apsw, json
    result = None
    db = os.path.join(BIBLEMATEGUI_DATA, "morphology.sqlite")
    with apsw.Connection(db) as connn:
        query = "SELECT * FROM morphology WHERE WordID=?" if word.startswith("wh") else "SELECT * FROM morphology WHERE WordID=? and Book > 39"
        cursor = connn.cursor()
        cursor.execute(query, (int(word[2:] if word.startswith("wh") else word[1:]),))
        result = cursor.fetchone()    
    if result:
        wordID, clauseID, book, chapter, verse, word, lexicalEntry, morphologyCode, morphology, lexeme, transliteration, pronunciation, interlinear, translation, gloss = result
        description = f'''<{'heb' if word.startswith("wh") else 'grk'}>{word}</{'heb' if word.startswith("wh") else 'grk'}> | <{'heb' if word.startswith("wh") else 'grk'}>{lexeme}</{'heb' if word.startswith("wh") else 'grk'}><br>
<clid>{morphology[:-1].replace(",", ", ")}</clid><br>
<wgloss>{interlinear}</wgloss><br>
<wtrans>{translation}</wtrans>'''
        #links = json.loads(links_json)
        return {'description': description, 'links': ""}
    return None

# API endpoint for tooltip data
@app.get('/api/tooltip/{word}')
async def tooltip_api(word: str):
    data = get_tooltip_data(word)
    if data:
        return data
    return {'error': 'Not found'}, 404

# Home Page
@ui.page('/')
def page_home(
    pc: str | None = None, # primary color
    sc: str | None = None, # secondary color
    fs: int | None = None, # font size in %
    s: bool | None = None, # sync
    d: bool | None = None, # dark mode
    t: str | None = None, # Token for using custom data: allow users to pass a custom token, which won't be stored, via a parameter when using public devices. For personal devices, enable persistent settings using `custom_token`.
    k: bool | None = True, # keep valid specified parameters in history
    m: bool | None = True, # display menu
    l: int | None = None, # layout; either: 1 (bible area only) or 2 (bible & tool areas) or 3 (tool area only)
    bbt: str | None = None, # bible bible text
    bb: int | None = None, # bible book
    bc: int | None = None, # bible chapter
    bv: int | None = None, # bible verse
    tbt: str | None = None, # tool bible text
    tb: int | None = None, # tool book
    tc: int | None = None, # tool chapter
    tv: int | None = None, # tool verse
    tool: str | None = None, # supported options: bible, audio, chronology, search ...
    bq: str | None = None, # bible query; currently not in use
    tq: str | None = None, # tool query
):
    """
    Home page that accepts optional parameters.
    Example: /?bb=1&bc=1&bv=1
    """
    def set_default_settings():
        """Sets the default settings in app.storage.user if they don't already exist."""
        for key, value in USER_DEFAULT_SETTINGS.items():
            if key not in app.storage.user:
                app.storage.user[key] = value
    # Call this once on startup to populate the default user storage
    set_default_settings()

    # spacing
    ui.query('.nicegui-content').classes('w-full h-full !p-0 !b-0 !m-0 !gap-0')

    # font-size
    if fs:
        app.storage.user["font_size"] = fs
    ui.add_css(f"""
        /* This targets the root HTML element and sets its font size */
        html {{
            font-size: {app.storage.user['font_size']}%;
        }}
    """)

    # colors
    if pc:
        app.storage.user["primary_color"] = pc
    if sc:
        app.storage.user["secondary_color"] = sc
    ui.colors(primary=app.storage.user["primary_color"], secondary=app.storage.user["secondary_color"])

    # sync
    if s is not None:
        app.storage.user['sync'] = s
    else:
        s = app.storage.user.setdefault('sync', True)

    # Bind app state to user storage
    app.storage.user["fullscreen"] = False
    ui.fullscreen().bind_value(app.storage.user, 'fullscreen')
    ui.dark_mode().bind_value(app.storage.user, 'dark_mode')
    if d is not None:
        app.storage.user['dark_mode'] = d

    # manage custom resources
    if not config.custom_token or (t and t == config.custom_token) or (app.storage.user.setdefault('custom_token', "") == config.custom_token):
        app.storage.client["custom"] = True # short-term memory (single page visit)
    else:
        app.storage.client["custom"] = False

    if l is not None and l in (1, 2, 3):
        app.storage.user['layout'] = l
    else:
        l = app.storage.user.setdefault('layout', 2)

    if bq is not None:
        app.storage.user['bible_query'] = bq
    else:
        bq = app.storage.user.setdefault('bible_query', '')    
    if tq is not None:
        app.storage.user['tool_query'] = tq
    else:
        tq = app.storage.user.setdefault('tool_query', '')

    load_bible_at_start = False
    if bbt is not None:
        load_bible_at_start = True
        app.storage.user['bible_book_text'] = bbt
    else:
        bbt = app.storage.user.setdefault('bible_book_text', "NET")
    if bb is not None:
        app.storage.user['bible_book_number'] = bb
    else:
        bb = app.storage.user.setdefault('bible_book_number', 1)
    if bc is not None:
        app.storage.user['bible_chapter_number'] = bc
    else:
        bc = app.storage.user.setdefault('bible_chapter_number', 1)
    if bv is not None:
        app.storage.user['bible_verse_number'] = bv
    else:
        bv = app.storage.user.setdefault('bible_verse_number', 1)
    if tbt is not None:
        app.storage.user['tool_book_text'] = tbt
    else:
        tbt = app.storage.user.setdefault('tool_book_text', "KJV")
    if tb is not None:
        app.storage.user['tool_book_number'] = tb
    else:
        tb = app.storage.user.setdefault('tool_book_number', 1)
    if tc is not None:
        app.storage.user['tool_chapter_number'] = tc
    else:
        tc = app.storage.user.setdefault('tool_chapter_number', 1)
    if tv is not None:
        app.storage.user['tool_verse_number'] = tv
    else:
        tv = app.storage.user.setdefault('tool_verse_number', 1)

    ui.add_head_html(get_tooltip_css(app.storage.user["dark_mode"]))
    ui.add_body_html(TOOLTIP_JS)

    # GUI object
    gui = BibleMateGUI()
    # navigation menu
    if m:
        gui.create_menu() # Add the shared menu
    # main content
    gui.create_home_layout()

    # load bible content at start
    if load_bible_at_start:
        next_tab_num = gui.area1_tab_loaded + 1
        if next_tab_num > gui.area1_tab_counter:
            gui.add_tab_area1()
        gui.load_area_1_content(title=bbt, keep=k)
    elif not gui.area1_tab_loaded: # when nothing is loaded
        gui.load_area_1_content(title="NET")
    
    # load tool content at start
    if tool:
        next_tab_num = gui.area2_tab_loaded + 1
        if next_tab_num > gui.area2_tab_counter:
            gui.add_tab_area2()
        gui.load_area_2_content(title=tbt if tool == "bible" else tool, keep=k, sync=s)
    elif not gui.area2_tab_loaded: # when nothing is loaded
        gui.load_area_2_content(title="Audio", sync=True)

    if k:
        # update storage based on latest loaded content
        args = app.storage.user[gui.get_active_area1_tab()]
        app.storage.user['bible_book_text'] = args.get('bt', "NET")
        app.storage.user['bible_book_number'] = args.get('b', 1)
        app.storage.user['bible_chapter_number'] = args.get('c', 1)
        app.storage.user['bible_verse_number'] = args.get('v', 1)
        app.storage.user['bible_query'] = args.get('q', '')

        args = app.storage.user[gui.get_active_area2_tab()]
        app.storage.user['tool_book_text'] = args.get('bt', "NET")
        app.storage.user['tool_book_number'] = args.get('b', 1)
        app.storage.user['tool_chapter_number'] = args.get('c', 1)
        app.storage.user['tool_verse_number'] = args.get('v', 1)
        app.storage.user['tool_query'] = args.get('q', '')
    
    if l is not None and l in (1, 2, 3):
        gui.swap_layout(l)

    # update the URL to reflect current settings without reloading
    '''def update_url():
        params = []
        params.append(f'pc={app.storage.user["primary_color"]}')
        params.append(f'sc={app.storage.user["secondary_color"]}')
        params.append(f'fs={app.storage.user["font_size"]}')
        params.append(f's={str(app.storage.user["sync"]).lower()}')
        params.append(f'd={str(app.storage.user["dark_mode"]).lower()}')
        if config.custom_token:
            params.append(f't={config.custom_token}')
        params.append(f'k={str(k).lower()}')
        params.append(f'm={str(m).lower()}')
        params.append(f'l={app.storage.user["layout"]}')
        params.append(f'bbt={app.storage.user["bible_book_text"]}')
        params.append(f'bb={app.storage.user["bible_book_number"]}')
        params.append(f'bc={app.storage.user["bible_chapter_number"]}')
        params.append(f'bv={app.storage.user["bible_verse_number"]}')
        params.append(f'tbt={app.storage.user["tool_book_text"]}')
        params.append(f'tb={app.storage.user["tool_book_number"]}')
        params.append(f'tc={app.storage.user["tool_chapter_number"]}')
        params.append(f'tv={app.storage.user["tool_verse_number"]}')
        params.append(f'bq={app.storage.user["bible_query"]}')
        params.append(f'tq={app.storage.user["tool_query"]}')
        new_url = '/?' + '&'.join(params)
        ui.run_javascript(f"history.replaceState(null, '', '{new_url}');")''' # TODO - update url when content is loaded

# Settings
@ui.page('/settings')
def page_Settings():
    """The main settings page for the BibleMate AI app."""
    def set_default_settings():
        """Sets the default settings in app.storage.user if they don't already exist."""
        for key, value in USER_DEFAULT_SETTINGS.items():
            if key not in app.storage.user:
                app.storage.user[key] = value
    # We can call this again to be safe, especially if new settings are added in updates.
    set_default_settings()

    # Adjust font-size
    ui.run_javascript(f"document.documentElement.style.fontSize = '{app.storage.user['font_size']}%'")
    def set_font_size(value):
        # Update the storage (automatic via bind, but good for explicit logic if needed later)
        app.storage.user['font_size'] = value
        # Update the HTML root element immediately via JS
        ui.run_javascript(f"document.documentElement.style.fontSize = '{value}%'")

    # primary color
    ui.colors(primary=app.storage.user["primary_color"], secondary=app.storage.user["secondary_color"])

    # Bind app state to user storage
    ui.dark_mode().bind_value(app.storage.user, 'dark_mode')
    app.storage.user["fullscreen"] = False
    ui.fullscreen().bind_value(app.storage.user, 'fullscreen')

    with ui.card().classes('w-full max-w-2xl mx-auto p-6 shadow-xl rounded-lg'):
        ui.label('BibleMate AI Settings').classes('text-3xl font-bold text-secondary mb-6')
        
        # --- Appearance Section ---
        with ui.expansion('Appearance', icon='palette').classes('w-full rounded-lg'):
            with ui.column().classes('w-full p-4'):
                # font-size
                with ui.row().classes('w-full items-center'):
                    ui.label("Font Size").classes('flex items-center font-bold mr-4')
                    # We display the current % value next to the label for clarity
                    ui.label().bind_text_from(app.storage.user, 'font_size', backward=lambda v: f'{int(v)}%').classes('text-sm text-gray-500')
                ui.slider(min=50, max=200, step=5, value=app.storage.user['font_size']) \
                    .bind_value(app.storage.user, 'font_size') \
                    .on_value_change(lambda e: set_font_size(e.value)) \
                    .props('label-always color=primary') \
                    .classes('w-full mb-4') \
                    .tooltip('Adjust the global font size (50% to 200%)')
                # colors
                ui.color_input(label='Primary Color') \
                    .bind_value(app.storage.user, 'primary_color') \
                    .tooltip('Manual hex code or color picker for app theme.') \
                    .on_value_change(lambda e: ui.colors(primary=e.value))
                ui.color_input(label='Secondary Color') \
                    .bind_value(app.storage.user, 'secondary_color') \
                    .tooltip('Manual hex code or color picker for app theme.') \
                    .on_value_change(lambda e: ui.colors(secondary=e.value))
                # dark mode
                with ui.row().classes('w-full'):
                    ui.label("Dark Mode").classes('flex items-center')
                    ui.space()
                    ui.switch().bind_value(app.storage.user, 'dark_mode').tooltip('Toggle dark mode for the app.').on_value_change(lambda: ui.run_javascript('location.reload()'))
                # fullscreen
                with ui.row().classes('w-full'):
                    ui.label("Fullscreen").classes('flex items-center')
                    ui.space()
                    ui.switch().bind_value(app.storage.user, 'fullscreen').tooltip('Toggle fullscreen mode for the app.')

        # --- User & Custom Data Section ---
        with ui.expansion('User & Custom Data', icon='person').classes('w-full rounded-lg'):
            with ui.column().classes('w-full p-4 gap-4'):
                ui.input(label='Avatar URL', placeholder='https://example.com/avatar.png') \
                    .bind_value(app.storage.user, 'avatar') \
                    .classes('w-full') \
                    .tooltip('URL for your profile picture (leave blank for default).')
                
                ui.input(label='Custom Token', password=True, password_toggle_button=True) \
                    .bind_value(app.storage.user, 'custom_token') \
                    .classes('w-full') \
                    .tooltip('Token for using custom data sources or personal APIs.')

        # --- Default Resources Section ---
        with ui.expansion('Default Resources', icon='book', value=True).classes('w-full rounded-lg'):
            # Use a grid for a more compact layout
            with ui.grid(columns=2).classes('w-full p-4 gap-4'):
                ui.select(label='Default Bible',
                          options=['NET', 'NIV', 'ESV', 'KJV']) \
                    .bind_value(app.storage.user, 'default_bible')

                ui.select(label='Default Commentary',
                          options=['CBSC', 'CBC', 'Calvin']) \
                    .bind_value(app.storage.user, 'default_commentary')

                ui.select(label='Default Encyclopedia',
                          options=['ISBE', 'Hasting', 'Kitto']) \
                    .bind_value(app.storage.user, 'default_encyclopedia')

                ui.select(label='Default Lexicon',
                          options=['Morphology', 'Strong', 'HALOT', 'BDAG']) \
                    .bind_value(app.storage.user, 'default_lexicon')

        # --- AI Backend Section ---
        with ui.expansion('AI Backend', icon='memory').classes('w-full rounded-lg'):
            with ui.column().classes('w-full p-4 gap-4'):
                ui.select(label='AI Backend',
                          options=['googleai', 'openai', 'azure', 'xai']) \
                    .bind_value(app.storage.user, 'ai_backend') \
                    .tooltip('Select the AI service provider.')

                ui.input(label='API Endpoint', placeholder='(Optional) Custom API endpoint') \
                    .bind_value(app.storage.user, 'api_endpoint') \
                    .classes('w-full') \
                    .tooltip('The custom API endpoint URL (if not using default).')

                ui.input(label='API Key', password=True, password_toggle_button=True) \
                    .bind_value(app.storage.user, 'api_key') \
                    .classes('w-full') \
                    .tooltip('Your API key for the selected backend.')

        # --- Localization Section ---
        with ui.expansion('Language', icon='language').classes('w-full rounded-lg'):
            with ui.column().classes('w-full p-4'):
                ui.select(label='Language',
                          options=['English', 'Traditional Chinese', 'Simplified Chinese']) \
                    .bind_value(app.storage.user, 'language')

        # --- Save Feedback ---
        ui.button('Home', on_click=lambda: ui.navigate.to('/')) \
            .classes('mt-6 w-full py-3 bg-blue-600 text-white rounded-lg font-semibold') \
            .tooltip('All settings are saved automatically as you change them. Click this to open the home page.')

# Entry_point

def main():
    # --- Run the App ---
    # Make sure to replace the secret!
    ui.run(
        reload=config.hot_reload,
        storage_secret=config.storage_secret, # e.g. generate one by running `openssl rand -hex 32` or `openssl rand -base64 32`
        port=config.port,
        title='BibleMate AI',
        favicon=os.path.expanduser(config.avatar) if config.avatar else os.path.join(BIBLEMATEGUI_APP_DIR, 'eliranwong.jpg')
    )

main()