"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.BackupVault = exports.BackupVaultEvents = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const backup_generated_1 = require("./backup.generated");
/**
 * Backup vault events
 */
var BackupVaultEvents;
(function (BackupVaultEvents) {
    /** BACKUP_JOB_STARTED */
    BackupVaultEvents["BACKUP_JOB_STARTED"] = "BACKUP_JOB_STARTED";
    /** BACKUP_JOB_COMPLETED */
    BackupVaultEvents["BACKUP_JOB_COMPLETED"] = "BACKUP_JOB_COMPLETED";
    /** BACKUP_JOB_SUCCESSFUL */
    BackupVaultEvents["BACKUP_JOB_SUCCESSFUL"] = "BACKUP_JOB_SUCCESSFUL";
    /** BACKUP_JOB_FAILED */
    BackupVaultEvents["BACKUP_JOB_FAILED"] = "BACKUP_JOB_FAILED";
    /** BACKUP_JOB_EXPIRED */
    BackupVaultEvents["BACKUP_JOB_EXPIRED"] = "BACKUP_JOB_EXPIRED";
    /** RESTORE_JOB_STARTED */
    BackupVaultEvents["RESTORE_JOB_STARTED"] = "RESTORE_JOB_STARTED";
    /** RESTORE_JOB_COMPLETED */
    BackupVaultEvents["RESTORE_JOB_COMPLETED"] = "RESTORE_JOB_COMPLETED";
    /** RESTORE_JOB_SUCCESSFUL */
    BackupVaultEvents["RESTORE_JOB_SUCCESSFUL"] = "RESTORE_JOB_SUCCESSFUL";
    /** RESTORE_JOB_FAILED */
    BackupVaultEvents["RESTORE_JOB_FAILED"] = "RESTORE_JOB_FAILED";
    /** COPY_JOB_STARTED */
    BackupVaultEvents["COPY_JOB_STARTED"] = "COPY_JOB_STARTED";
    /** COPY_JOB_SUCCESSFUL */
    BackupVaultEvents["COPY_JOB_SUCCESSFUL"] = "COPY_JOB_SUCCESSFUL";
    /** COPY_JOB_FAILED */
    BackupVaultEvents["COPY_JOB_FAILED"] = "COPY_JOB_FAILED";
    /** RECOVERY_POINT_MODIFIED */
    BackupVaultEvents["RECOVERY_POINT_MODIFIED"] = "RECOVERY_POINT_MODIFIED";
    /** BACKUP_PLAN_CREATED */
    BackupVaultEvents["BACKUP_PLAN_CREATED"] = "BACKUP_PLAN_CREATED";
    /** BACKUP_PLAN_MODIFIED */
    BackupVaultEvents["BACKUP_PLAN_MODIFIED"] = "BACKUP_PLAN_MODIFIED";
})(BackupVaultEvents = exports.BackupVaultEvents || (exports.BackupVaultEvents = {}));
class BackupVaultBase extends core_1.Resource {
    /**
     * Grant the actions defined in actions to the given grantee
     * on this Backup Vault resource.
     *
     * @param grantee Principal to grant right to
     * @param actions The actions to grant
     */
    grant(grantee, ...actions) {
        for (const action of actions) {
            if (action.indexOf('*') >= 0) {
                throw new Error("AWS Backup access policies don't support a wildcard in the Action key.");
            }
        }
        return iam.Grant.addToPrincipal({
            grantee: grantee,
            actions: actions,
            resourceArns: [this.backupVaultArn],
        });
    }
}
/**
 * A backup vault
 */
class BackupVault extends BackupVaultBase {
    constructor(scope, id, props = {}) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_backup_BackupVaultProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, BackupVault);
            }
            throw error;
        }
        if (props.backupVaultName && !/^[a-zA-Z0-9\-_]{2,50}$/.test(props.backupVaultName)) {
            throw new Error('Expected vault name to match pattern `^[a-zA-Z0-9\-_]{2,50}$`');
        }
        let notifications;
        if (props.notificationTopic) {
            notifications = {
                backupVaultEvents: props.notificationEvents || Object.values(BackupVaultEvents),
                snsTopicArn: props.notificationTopic.topicArn,
            };
            props.notificationTopic.grantPublish(new iam.ServicePrincipal('backup.amazonaws.com'));
        }
        this.accessPolicy = props.accessPolicy ?? new iam.PolicyDocument();
        if (props.blockRecoveryPointDeletion) {
            this.blockRecoveryPointDeletion();
        }
        const vault = new backup_generated_1.CfnBackupVault(this, 'Resource', {
            backupVaultName: props.backupVaultName || this.uniqueVaultName(),
            accessPolicy: core_1.Lazy.any({ produce: () => this.accessPolicy.toJSON() }),
            encryptionKeyArn: props.encryptionKey && props.encryptionKey.keyArn,
            notifications,
        });
        vault.applyRemovalPolicy(props.removalPolicy);
        this.backupVaultName = vault.attrBackupVaultName;
        this.backupVaultArn = vault.attrBackupVaultArn;
    }
    /**
     * Import an existing backup vault by name
     */
    static fromBackupVaultName(scope, id, backupVaultName) {
        const backupVaultArn = core_1.Stack.of(scope).formatArn({
            service: 'backup',
            resource: 'backup-vault',
            resourceName: backupVaultName,
            arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME,
        });
        return BackupVault.fromBackupVaultArn(scope, id, backupVaultArn);
    }
    /**
     * Import an existing backup vault by arn
     */
    static fromBackupVaultArn(scope, id, backupVaultArn) {
        const parsedArn = core_1.Stack.of(scope).splitArn(backupVaultArn, core_1.ArnFormat.SLASH_RESOURCE_NAME);
        if (!parsedArn.resourceName) {
            throw new Error(`Backup Vault Arn ${backupVaultArn} does not have a resource name.`);
        }
        class Import extends BackupVaultBase {
            constructor() {
                super(...arguments);
                this.backupVaultName = parsedArn.resourceName;
                this.backupVaultArn = backupVaultArn;
            }
        }
        return new Import(scope, id, {
            account: parsedArn.account,
            region: parsedArn.region,
        });
    }
    /**
     * Adds a statement to the vault access policy
     */
    addToAccessPolicy(statement) {
        this.accessPolicy.addStatements(statement);
    }
    /**
     * Adds a statement to the vault access policy that prevents anyone
     * from deleting a recovery point.
     */
    blockRecoveryPointDeletion() {
        this.addToAccessPolicy(new iam.PolicyStatement({
            effect: iam.Effect.DENY,
            actions: [
                'backup:DeleteRecoveryPoint',
                'backup:UpdateRecoveryPointLifecycle',
            ],
            principals: [new iam.AnyPrincipal()],
            resources: ['*'],
        }));
    }
    uniqueVaultName() {
        // Max length of 50 chars, get the last 50 chars
        const id = core_1.Names.uniqueId(this);
        return id.substring(Math.max(id.length - 50, 0), id.length);
    }
}
exports.BackupVault = BackupVault;
_a = JSII_RTTI_SYMBOL_1;
BackupVault[_a] = { fqn: "@aws-cdk/aws-backup.BackupVault", version: "1.174.0" };
//# sourceMappingURL=data:application/json;base64,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