import logging
from typing import Callable, Tuple

from pyimport import type_converter
from pyimport.fieldfile import FieldFile
from pyimport.logger import ErrorResponse, ErrorHandler
from pyimport.type_converter import convert_it


class EnricherException(Exception):
    pass


class Enricher:

    def __init__(self,
                 field_file: FieldFile,
                 locator: bool = True,
                 timestamp_func: Callable = None,
                 onerror: ErrorResponse = ErrorResponse.Warn,
                 filename: str = None):

        self._logger = logging.getLogger(__name__)

        self._onerror = onerror
        self._eh = ErrorHandler(onerror)
        self._line_count = 0
        self._timestamp = None
        self._idField = None  # section on which filename == _id
        self._log = logging.getLogger(__name__)
        #self._converter = Converter(self._log)
        self._field_file = field_file
        self._locator = locator
        self._timestamp_func = timestamp_func
        if filename is None:
            self._filename = "Unknown"
        else:
            self._filename = filename

    def enrich_value(self, k, v) -> str:

        if k.startswith("blank-"):
            self._eh.warning(f"Field {k} is blank [blank-] : ignoring")
            return None
        # try:
        t = self._field_file.type_value(k)

        try:
            return convert_it(t, v, self._field_file.format_value(k))

        except ValueError as e:
            self._eh.error(f"Parse failure at field '{k}'\ntype conversion error: Cannot convert '{v}' to type {t}")

    def enrich_doc(self, csv_doc: dict, new_field: Tuple[str, str]=None, cut_list:list[str]=None, line_number: int = None) -> dict:
        """
        Make a new doc from a dictEntry generated by the csv.DictReader.

        :param csv_doc: the line to be parsed (dict of strs)
        :param new_field: a tuple of a new field and value to be added to the doc
        :param line_number: the location of the line in the input file
        :return: the new doc

        WIP
        Do we make gen id generate a compound key or another field instead of ID
        TODO: Need to get the filename being parsed in at this level to allow use to report the right fil
        when an error occurs.
        """

        fields = self._field_file.fields()
        if line_number is None:
            line_number = "Unknown"

        if len(csv_doc) == 1:
            line = ",".join(csv_doc.values())
            self._eh.warning(f"Warning: only one field in input line. Do you have the right delimiter set ?")
            self._eh.warning(f"input line : {line}")

        if self._locator:
            csv_doc['locator'] = {"line": line_number}

        if new_field and new_field not in csv_doc:
            type_str, _ = type_converter.guess_type(new_field[1])
            csv_doc[new_field[0]] = type_converter.convert_it(type_str, new_field[1])

        return self._timestamp_func(csv_doc) if self._timestamp_func else csv_doc


