import os
import sys
from datetime import datetime, timezone

from pyimport.audit import Audit
from pyimport.csvreader import CSVReader
from pyimport.doctimestamp import DocTimeStamp
from pyimport.enricher import Enricher
from pyimport.fieldfile import FieldFile, FieldFileException
from pyimport.filereader import FileReader
from pyimport.importresult import ImportResults
from pyimport.linereader import is_url, RemoteLineReader
from pyimport.logger import Log, eh
from pyimport.version import __VERSION__


class ImportCommand:
    def __init__(self, args):

        self._args = args
        self._log = Log().log
        if args.audit:
            self._audit = Audit(args.audithost, database_name=args.auditdatabase, collection_name=args.auditcollection)
        else:
            self._audit = None

    @staticmethod
    def parse_new_field(new_field):
        if new_field:
            k, v = new_field.split("=")
            return k.strip(), v.strip()
        else:
            return None

    @staticmethod
    def prep_field_file(args, csv_filename=None) -> FieldFile:
        """
        Prepares the field file for the import process. If a field file is specified it is loaded. If not, a field file
        is autogenerated from the csv file. If the CSV file has not header line, a FieldFileException is raised.
        Args:
            args: list of command line args
            csv_filename: the input CSV file.

        Returns:
            FieldFile object
        """

        if args.fieldfile:
            if os.path.isfile(args.fieldfile):
                field_file = FieldFile.load(args.fieldfile)
            else:
                raise FieldFileException(f"No such field file:'{args.fieldfile}'")
        else:
            field_filename = FieldFile.make_default_tff_name(args.filenames[0])
            if os.path.isfile(field_filename):
                field_file = FieldFile.load(field_filename)
            elif FileReader.sniff_header(csv_filename):
                field_file = FieldFile.generate_field_file(csv_filename=csv_filename,
                                                           ff_filename=field_filename,
                                                           delimiter=args.delimiter,
                                                           has_header=args.hasheader)
                eh.info(f"Auto generated field file:'{field_filename}'")
            else:
                raise FieldFileException(f"'{csv_filename}' has no header line, so we cannot generate a field file")

        return field_file

    @staticmethod
    def prep_csv_reader(args, filename, field_info) -> CSVReader:
        if is_url(filename):
            csv_file = RemoteLineReader(url=filename)
        else:
            csv_file = open(filename, "r")

        return CSVReader(file=csv_file, limit=args.limit, field_file=field_info,
                         has_header=args.hasheader,
                         cut_fields=args.cut,
                         delimiter=args.delimiter)

    @staticmethod
    def prep_parser(args, field_info, filename) -> Enricher:
        if args.addtimestamp == DocTimeStamp.DOC_TIMESTAMP:
            ts_func = args.addtimestamp
        elif args.addtimestamp == DocTimeStamp.BATCH_TIMESTAMP:
            ts_func = datetime.now(timezone.utc)
        else:
            ts_func = None

        parser = Enricher(field_info, locator=args.locator, timestamp_func=ts_func, onerror=args.onerror,
                          filename=filename)
        return parser

    @staticmethod
    def process_mode(args):
        mode = {}
        if args.multi or args.asyncpro or args.threads:
            if args.multi:
                mode["multi"] = args.poolsize
            if args.asyncpro:
                mode["asyncpro"] = True
            if args.threads:
                mode["threads"] = args.threads
        else:
            mode["single"] = True
        return mode

    def report_process_one_file(self, args, result):
        audit_doc = None
        if self._audit:
            audit_doc = {"command": "process one file",
                         "version": __VERSION__,
                         "filename": result.filename,
                         "elapsed_time": result.elapsed_time,
                         "total_written": result.total_written,
                         "mode": ImportCommand.process_mode(args),
                         "avg_records_per_sec": result.avg_records_per_sec,
                         "cmd_line": " ".join(sys.argv)}

            self._audit.add_batch_info(audit_doc)
        return audit_doc

    def report_process_files(self, args, results: ImportResults):

        audit_doc = None
        if self._audit:
            audit_doc = {"command": "process files",
                         "filenames": results.filenames,
                         "elapsed_time": results.elapsed_time,
                         "total_written": results.total_written,
                         "avg_records_per_sec": results.avg_records_per_sec,
                         "mode": ImportCommand.process_mode(args),
                         "cmd_line": " ".join(sys.argv)}
            self._audit.add_batch_info(audit_doc)
        return audit_doc

