"""Models for Archer questionnaire operations."""

from typing import List, Optional, Dict, Any
from pydantic import BaseModel, Field


class QuestionnaireRule(BaseModel):
    """Represents a questionnaire rule in Archer."""
    
    Id: int = Field(..., description="Questionnaire rule ID")
    Name: Optional[str] = Field(None, description="Rule name")
    QuestionnaireId: Optional[int] = Field(None, description="Parent questionnaire ID")
    LevelId: Optional[int] = Field(None, description="Target application/level ID")
    
    class Config:
        json_schema_extra = {
            "example": {
                "Id": 501,
                "Name": "Auto-populate Rule",
                "QuestionnaireId": 100,
                "LevelId": 75
            }
        }


class Questionnaire(BaseModel):
    """Represents an Archer questionnaire."""
    
    Id: int = Field(..., description="Questionnaire ID")
    Name: str = Field(..., description="Questionnaire name")
    Alias: Optional[str] = Field(None, description="Questionnaire alias")
    Type: Optional[int] = Field(None, description="Questionnaire type")
    LanguageId: Optional[int] = Field(None, description="Language ID")
    
    # Target application reference
    TargetApplicationId: Optional[int] = Field(
        None,
        description="ID of the target application this questionnaire links to"
    )
    TargetApplicationName: Optional[str] = Field(
        None,
        description="Name of the target application"
    )
    
    # Optional metadata
    fields: Optional[List[Dict[str, Any]]] = Field(
        None,
        description="Questionnaire fields"
    )
    rules: Optional[List[QuestionnaireRule]] = Field(
        None,
        description="Questionnaire rules"
    )
    
    class Config:
        json_schema_extra = {
            "example": {
                "Id": 100,
                "Name": "Security Assessment Questionnaire",
                "Alias": "SEC_ASSESS_Q",
                "Type": 1,
                "LanguageId": 1,
                "TargetApplicationId": 75,
                "TargetApplicationName": "Security Incidents"
            }
        }
    
    def get_rule_by_id(self, rule_id: int) -> Optional[QuestionnaireRule]:
        """
        Get questionnaire rule by ID.
        
        Args:
            rule_id: ID of the rule
            
        Returns:
            QuestionnaireRule if found, None otherwise
        """
        if not self.rules:
            return None
        
        return next((r for r in self.rules if r.Id == rule_id), None)
    
    def get_rules_by_level(self, level_id: int) -> List[QuestionnaireRule]:
        """
        Get all rules for a specific level/application.
        
        Args:
            level_id: ID of the level/application
            
        Returns:
            List of QuestionnaireRule for the level
        """
        if not self.rules:
            return []
        
        return [r for r in self.rules if r.LevelId == level_id]
