# Import necessary modules and handle potential import errors
try:
    from sage_lib.FileManager import FileManager
except ImportError as e:
    import sys
    # Write an error message to standard error if the import fails
    sys.stderr.write(f"An error occurred while importing FileManager: {str(e)}\n")
    # Clean up the namespace by deleting the failed import
    del sys

# Define the WaveFileManager class, inheriting from FileManager
class WaveFileManager(FileManager):
    """
    WaveFileManager Class
    
    A specialized FileManager designed to handle wave function data files, commonly
    used in computational physics and electrochemistry. This class extends the 
    functionality of FileManager to include operations specific to handling 
    large wave function files, such as those generated by VASP (Vienna Ab initio 
    Simulation Package). 

    Attributes:
        data: Stores any additional data associated with the file. This attribute 
              can be utilized to hold metadata or file-specific information.
    """

    def __init__(self, file_location: str = None, name: str = None, **kwargs):
        """
        Initializes the WaveFileManager instance.

        Parameters:
            file_location (str, optional): The location of the file to be managed.
            name (str, optional): The name of the file.
            **kwargs: Arbitrary keyword arguments passed to the FileManager.
        """
        super().__init__(name=name, file_location=file_location)
        self.data = None  # Initialize additional data attribute

    def exportFile(self, file_location: str = None, original_file_location: str = None) -> bool:
        """
        Exports the managed file to a specified location.

        Parameters:
            file_location (str, optional): The destination location for the file.
            original_file_location (str, optional): The original location of the file. 
                                                    If not provided, uses the instance's file location.

        Returns:
            bool: True if the file is exported successfully, False otherwise.
        """
        original_file_location = original_file_location if original_file_location is not None else self.file_location
        self.copy(file_location, original_file_location)
        return True

    def importFileLocation(self, file_location: str = None) -> bool:
        """
        Imports and sets the file location.

        Parameters:
            file_location (str, optional): The location of the file to be imported. 
                                            If not a string, the instance's file location is used.

        Returns:
            bool: True if the file location is set successfully, False otherwise.
        """
        file_location = file_location if type(file_location) == str else self.file_location
        self._file_location = file_location
        return True
