#ifndef FILE_SVDTREFFTZ_HPP
#define FILE_SVDTREFFTZ_HPP
#include <comp.hpp>
#include <python_comp.hpp>

#ifndef FILE_INTEGRATORCFHPP
#include <integratorcf.hpp>
#define FILE_INTEGRATORCFHPP
#endif

namespace ngcomp
{
  /// Copies `source` to the beginning of `target`.
  ///
  /// For `source.Size() > target.Size()`,
  /// the behaviour is undefined.
  void copyBitArray (const shared_ptr<BitArray> target,
                     const shared_ptr<const BitArray> source);

  class TrefftzEmbedding
  {
    shared_ptr<FESpace> fes = nullptr;
    shared_ptr<FESpace> fes_test = nullptr;
    shared_ptr<FESpace> fes_conformity = nullptr;
    const shared_ptr<SumOfIntegrals> top;
    const shared_ptr<SumOfIntegrals> trhs;
    const shared_ptr<SumOfIntegrals> cop;
    const shared_ptr<SumOfIntegrals> crhs;
    const shared_ptr<SumOfIntegrals> fes_ip;
    shared_ptr<MeshAccess> ma;
    shared_ptr<BitArray> ignoredofs;

    /// contains the mapping of Element Number to the associated dofs
    /// of the conforming Trefftz space
    Table<DofId> tdof_nrs;

    bool compute_elmat_T_inv = true;

    std::variant<size_t, double> ndof_trefftz;

    shared_ptr<std::map<std::string, Vector<double>>> stats = nullptr;

    /// elmat = (elmat_conforming | elmat_trefftz)
    Array<optional<Matrix<double>>> etmats;
    Array<optional<Matrix<Complex>>> etmatsc;

    /// (pseudo-)inverse matrices of the Trefftz part of etmats
    Array<optional<Matrix<double>>> etmats_trefftz_inv;
    /// (pseudo-)inverse matrices of the Trefftz part of etmatsc
    Array<optional<Matrix<Complex>>> etmatsc_trefftz_inv;

    /// elmat_trefftz.width == local_ndof_trefftz
    /// elmat_conforming.width == elmat.width - local_ndof_trefftz
    ///                        == local_ndof_conforming
    Array<size_t> local_ndofs_trefftz;

    shared_ptr<const BaseVector> psol;

    /// Executes the embedded Trefftz procedure
    /// and stores the results in this class.
    ///
    /// @tparam SCAL either `double` or `Complex`
    template <typename SCAL> void EmbTrefftz ();

  public:
    TrefftzEmbedding (shared_ptr<SumOfIntegrals> _top,
                      shared_ptr<SumOfIntegrals> _trhs,
                      shared_ptr<SumOfIntegrals> _cop,
                      shared_ptr<SumOfIntegrals> _crhs, size_t _ndof_trefftz,
                      double _eps, shared_ptr<FESpace> _fes = nullptr,
                      shared_ptr<FESpace> _fes_test = nullptr,
                      shared_ptr<FESpace> _fes_conformity = nullptr,
                      shared_ptr<SumOfIntegrals> _fes_ip = nullptr,
                      shared_ptr<BitArray> _ignoredofs = nullptr,
                      shared_ptr<std::map<std::string, Vector<double>>> _stats
                      = nullptr);

    shared_ptr<BaseVector>
    Embed (const shared_ptr<const BaseVector> tgfu) const;
    shared_ptr<GridFunction>
    Embed (const shared_ptr<const GridFunction> tgfu) const;
    shared_ptr<const BaseVector> GetParticularSolution () const;
    shared_ptr<const BaseVector>
    GetParticularSolution (shared_ptr<SumOfIntegrals> _trhs) const;
    shared_ptr<const BaseVector>
    GetParticularSolution (shared_ptr<const BaseVector> _trhsvec) const;
    shared_ptr<const BaseMatrix> GetEmbedding () const;
    shared_ptr<FESpace> GetFES () const noexcept { return fes; }
    shared_ptr<FESpace> GetFEStest () const noexcept { return fes_test; }
    shared_ptr<FESpace> GetFESconf () const noexcept { return fes_conformity; }
    shared_ptr<const BitArray> GetIgnoredDofs () const { return ignoredofs; }
    const Array<optional<Matrix<double>>> &GetEtmats () const noexcept
    {
      return etmats;
    }
    const Array<optional<Matrix<Complex>>> &GetEtmatsC () const noexcept
    {
      return etmatsc;
    }
    optional<Matrix<double>> GetEtmat (size_t i) const { return etmats[i]; }
    optional<Matrix<Complex>> GetEtmatC (size_t i) const { return etmatsc[i]; }
    const Array<size_t> &GetLocalNodfsTrefftz () const noexcept
    {
      return local_ndofs_trefftz;
    }
    const FlatArray<DofId> GetTDofNrs (const size_t elnr) const
    {
      return tdof_nrs[elnr];
    }
  };

  /// Represents the FESpace, that is generated by the embedded Trefftz method.
  /// Use the \ref EmbTrefftzFESpace(shared_ptr<T>) constructor to build the
  /// space from a given (trial) FESpcae, use \ref EmbTrefftzFESpace::SetOp to
  /// specify the operation that the Trefftz space should be build from.
  ///
  /// @tparam T must be a FESpace
  template <typename T>
  class EmbTrefftzFESpace
      : public T //, public std::enable_shared_from_this<EmbTrefftzFESpace>
  {
    const shared_ptr<TrefftzEmbedding> emb;
    static_assert (std::is_base_of_v<FESpace, T>, "T must be a FESpace");
    const FlatArray<optional<Matrix<double>>> etmats;
    const FlatArray<optional<Matrix<Complex>>> etmatsc;
    const shared_ptr<const FESpace> fes_conformity;
    const shared_ptr<const BitArray> ignoredofs;

    /// (pseudo-)inverse matrices of etmats
    mutable Array<optional<Matrix<double>>> etmats_inv;
    mutable Array<optional<Matrix<Complex>>> etmatsc_inv;
    mutable once_flag etmats_inv_computed;

  public:
    EmbTrefftzFESpace (shared_ptr<MeshAccess> ama, const Flags &flags,
                       bool parseflags = false)
        : T (ama, flags, parseflags)
    {
      throw Exception ("Please construct via an embedding");
    }

    EmbTrefftzFESpace (shared_ptr<TrefftzEmbedding> aemb)
        : T (aemb->GetFES ()->GetMeshAccess (), aemb->GetFES ()->GetFlags (),
             false),
          emb (aemb), etmats (emb->GetEtmats ()), etmatsc (emb->GetEtmatsC ()),
          fes_conformity (emb->GetFESconf ()),
          ignoredofs (emb->GetIgnoredDofs ())
    {
      this->name
          = "EmbTrefftzFESpace(" + emb->GetFES ()->GetClassName () + ")";
      this->type = "embt";
      this->needs_transform_vec = true;
      this->iscomplex = emb->GetFES ()->IsComplex ();
      if constexpr (std::is_same_v<CompoundFESpace, T>)
        for (auto space : static_pointer_cast<T> (emb->GetFES ())->Spaces ())
          this->AddSpace (space);

      if (this->IsComplex ())
        etmatsc_inv.SetSize (this->ma->GetNE (VOL));
      else
        etmats_inv.SetSize (this->ma->GetNE (VOL));

      this->Update ();
      // this->UpdateDofTables();
      // this->UpdateCouplingDofArray();

      if (this->order_policy == ORDER_POLICY::VARIABLE_ORDER)
        {
          this->GetMeshAccess ()->IterateElements (VOL, [&] (ElementId ei) {
            NodeId ni (NODE_TYPE::NT_ELEMENT, ei.Nr ());
            this->SetOrder (ni, emb->GetFES ()->GetOrder (ni));
          });
          this->Update ();
        }

      this->FinalizeUpdate ();

      // needs previous FinalizeUpdate to construct free_dofs for `this`
      if (fes_conformity)
        copyBitArray (this->GetFreeDofs (), fes_conformity->GetFreeDofs ());
    }

    void GetDofNrs (ElementId ei, Array<DofId> &dnums) const override;

    virtual void VTransformMR (ElementId ei, const SliceMatrix<double> mat,
                               TRANSFORM_TYPE type) const override;

    virtual void VTransformMC (ElementId ei, const SliceMatrix<Complex> mat,
                               TRANSFORM_TYPE type) const override;

    virtual void VTransformVR (ElementId ei, const SliceVector<double> vec,
                               TRANSFORM_TYPE type) const override;

    virtual void VTransformVC (ElementId ei, const SliceVector<Complex> vec,
                               TRANSFORM_TYPE type) const override;

    virtual string GetClassName () const override;

    shared_ptr<TrefftzEmbedding> GetEmbedding () const noexcept { return emb; }

    virtual void UpdateCouplingDofArray () override;

  private:
    optional<FlatMatrix<double>> GetEtmatInv (size_t idx) const;
    optional<FlatMatrix<Complex>> GetEtmatCInv (size_t idx) const;
  };
}

#ifdef NGS_PYTHON
#include <python_ngstd.hpp>
void ExportEmbTrefftz (py::module m);
#endif // NGS_PYTHON

#endif
