import time
from typing import Final, List

import dg
import numpy as np
import scipy as sp
import scipy.sparse
from dg import dgell, exactpoi
from netgen.geom2d import unit_square
from ngsolve import (
    BBND,
    BND,
    H1,
    L2,
    BilinearForm,
    CoefficientFunction,
    Draw,
    FacetFESpace,
    FESpace,
    GridFunction,
    Integrate,
    Mesh,
    NormalFacetFESpace,
    TaskManager,
    comp,
    dx,
    grad,
    specialcf,
    sqrt,
    x,
    y,
)
from ngstrefftz import EmbeddedTrefftzFES, L2EmbTrefftzFESpace, TrefftzEmbedding

# import matplotlib.pyplot as plt


# SetNumThreads(3)


def PySVDConformingTrefftz(
    top: comp.SumOfIntegrals,
    fes: FESpace,
    cop: comp.SumOfIntegrals,
    crhs: comp.SumOfIntegrals,
    fes_conformity: FESpace,
    trefftzndof: int,
    debug: bool = False,
) -> np.ndarray:
    """
    produces an embedding matrix P

    `top`: the differential operation

    `fes`: the finite element space of `top`

    `cop`: left hand side of the conformity operation

    `crhs`: right hand side of the conformity operation

    `fes_conformity`: finite element space of the conformity operation

    `trefftzndof`: number of degrees of freedom per element
        in the Trefftz finite element space on `fes`, generated by `top`
        (i.e. the local dimension of the kernel of `top` on one element)

    `debug`: if `True`, print debug messages. Default: `False`

    returns: P

    raises: LinAlgError: if the imput is not sound, a non-invertible matrix may
        be tried to be inverted
    """
    mesh = fes.mesh

    # let L be the matrix corrensponding to
    # the differential operator top
    opbf = BilinearForm(fes)
    opbf += top
    opbf.Assemble()
    rows, cols, vals = opbf.mat.COO()
    L = sp.sparse.csr_matrix((vals, (rows, cols)))
    L = L.todense()

    # let B1 be the matrix corrensponding to
    # the left hand side conformity operator cop
    copbf_lhs = BilinearForm(trialspace=fes, testspace=fes_conformity)
    copbf_lhs += cop
    copbf_lhs.Assemble()
    rows, cols, vals = copbf_lhs.mat.COO()
    B1 = sp.sparse.csr_matrix((vals, (rows, cols)))
    B1 = B1.todense()
    if debug:
        print("B1.shape", B1.shape)

    # let B2 be the matrix corrensponding to
    # the right hand side conformity operator crhs
    copbf_rhs = BilinearForm(fes_conformity)
    copbf_rhs += crhs
    copbf_rhs.Assemble()
    rows, cols, vals = copbf_rhs.mat.COO()
    B2 = sp.sparse.csr_matrix((vals, (rows, cols)))
    B2 = B2.todense()

    # localndof = int(fes.ndof/mesh.ne)
    # number of degrees of freedom per element
    # in the trefftz finite element space on fes
    # trefftzndof: Final[int] = 2 * order + 1 - 3

    # number of degrees of freedom of the contraint finite element space
    n_constr: Final[int] = fes_conformity.ndof

    # layout:
    # /    |    \
    # | P1 | P2 |
    # \    |    /
    # with P1 having shape (fes.ndof, n_constr),
    P = np.zeros([L.shape[1], (trefftzndof) * mesh.ne + n_constr])

    # solve the following linear system in an element-wise fashion:
    # L @ T1 = B for the unknown matrix T1,
    # with the given matrices:
    #     /   \    /   \
    #  A= |B_1| B= |B_2|
    #     | L |    | 0 |
    #     \   /    \   /
    for el, el_c in zip(fes.Elements(), fes_conformity.Elements()):
        nr: Final[int] = el.nr
        dofs: Final[List[int]] = el.dofs
        dofs_c: Final[List[int]] = el_c.dofs

        if debug:
            print("dofs:", dofs)
            print("dofs_c:", dofs_c)

        # produce local sub-matrices from the global matrices L, B1, B2
        elmat_l = L[dofs, :][:, dofs]
        elmat_b1 = B1[dofs_c, :][:, dofs]
        elmat_b2 = B2[dofs_c, :][:, dofs_c]

        if debug:
            print("elmat_b1", elmat_b1.shape)
            print("elmat_l", elmat_l.shape)

        #     /   \    /   \
        #  A= |B_1| B= |B_2|
        #     | L |    | 0 |
        #     \   /    \   /
        elmat_a = np.vstack([elmat_b1, elmat_l])
        elmat_b = np.vstack([elmat_b2, np.zeros([len(dofs), len(dofs_c)])])

        if debug:
            print("elmat_a", elmat_a)
            print("elmat_b", elmat_b)

        # A = U @ s @ V, singular value decomposition
        U, s, V = np.linalg.svd(elmat_a)

        # pseudo inverse of s
        s_inv = np.hstack(
            [np.diag(1.0 / s), np.zeros((V.shape[0], U.shape[0] - V.shape[0]))]
        )

        if debug:
            print("U", U)
            print("s_inv", s_inv)
            print("V", V)

        # solve A @ T1 = B
        # i.e. T1 = A^{-1} @ B
        # for the unknown T1
        T1 = V.T @ s_inv @ U.T @ elmat_b

        if debug:
            print("T1", T1)

        # Place the local solution T1
        # into the global solution P.
        P[np.ix_(dofs, dofs_c)] += T1

        ## matplotlib and netgen do not like to be opened at the same time,
        ## so the plotting of the singular values is disabled for now
        # if debug:
        #    plt.plot(s, marker="x")
        #    plt.title(f"singular values of A for element number {nr}")
        #    plt.xlabel("singular value number")
        #    plt.ylabel("singular value")
        #    plt.yscale("log")
        #    plt.show()

        # Place the basis of the kernel of A into P2.
        #
        # The basis of the kernel is found in V,
        # namely as the bottommost row-vectors.
        #
        # In P2, the kernel from one element is placed into
        # `trefftzndof` many neighbouring columns,
        # starting with element nr. 0 with columns
        # n_constr to n_constr + trefftzndof - 1 (inclusively).
        # Note, that the basis is transposed from row to column-vectors.
        nonzero_dofs: Final[int] = len(dofs) - trefftzndof
        P[
            np.ix_(
                dofs,
                range(
                    n_constr + nr * trefftzndof,
                    n_constr + (nr + 1) * trefftzndof,
                ),
            )
        ] += V[nonzero_dofs:, :].T

    if debug:
        import netgen.gui

        print(P)
        gfu = GridFunction(fes)
        for i in range(P.shape[1]):
            print("P slice:\n", P[:, i].flatten())
            gfu.vec.FV().NumPy()[:] = P[:, i]
            Draw(gfu)
            input("")
    return P


def test_PySVDConformingTrefftz(
    order: int = 2, debug: bool = False, maxh=0.4
) -> float:
    """
    simple test case for PySVDConstrainedTrefftz.

    `order`: polynomial oder of the underlying space

    `debug`: True: print debug info, default: False

    >>> test_PySVDConformingTrefftz(order=3, debug=False) # doctest:+ELLIPSIS
    3.6...e-05
    """
    mesh2d = Mesh(unit_square.GenerateMesh(maxh=maxh))

    fes = L2(mesh2d, order=order, dgjumps=True)  # ,all_dofs_together=True)
    u, v = fes.TnT()
    uh = u.Operator("hesse")
    vh = v.Operator("hesse")
    top = (uh[0, 0] + uh[1, 1]) * (vh[0, 0] + vh[1, 1]) * dx

    fes_conformity = FacetFESpace(mesh2d, order=0)  # ,all_dofs_together=True)
    uF, vF = fes_conformity.TnT()
    cop = u * vF * dx(element_boundary=True)
    crhs = uF * vF * dx(element_boundary=True)

    P = PySVDConformingTrefftz(
        top,
        fes,
        cop,
        crhs,
        fes_conformity,
        2 * order + 1 - 3,
        debug=False,
    )
    if debug:
        print("P:")
        print(P.shape)
        print(P)

        import matplotlib.pyplot as plt

        plt.spy(P)
        plt.show()

    a, f = dg.dgell(fes, dg.exactlap)
    rows, cols, vals = a.mat.COO()
    A = scipy.sparse.csr_matrix((vals, (rows, cols)))
    A = A.todense()

    TA = P.transpose() @ A @ P
    tsol = np.linalg.solve(TA, P.transpose() @ f.vec.FV())
    tpgfu = GridFunction(fes)
    tpgfu.vec.data = P @ tsol
    if debug:
        import netgen.gui

        Draw(tpgfu)
        input()
    return sqrt(Integrate((tpgfu - dg.exactlap) ** 2, mesh2d))


def test_ConstrainedTrefftzCpp(
    order: int = 2, debug: bool = False, maxh=0.4
) -> float:
    """
    simple test case for Constrained Trefftz C++ implementation.

    `order`: polynomial oder of the underlying space

    `debug`: True: print debug info, default: False

    >>> test_ConstrainedTrefftzCpp(order=3, debug=False) # doctest:+ELLIPSIS
    3.6...e-05

    >>> with TaskManager():
    ...   test_ConstrainedTrefftzCpp(order=3, debug=False) # doctest:+ELLIPSIS
    3.6...e-05
    """
    mesh2d = Mesh(unit_square.GenerateMesh(maxh=maxh))

    fes = L2(mesh2d, order=order, dgjumps=True)  # ,all_dofs_together=True)
    u, v = fes.TnT()
    uh = u.Operator("hesse")
    vh = v.Operator("hesse")
    top = (uh[0, 0] + uh[1, 1]) * (vh[0, 0] + vh[1, 1]) * dx

    fes_conformity = FacetFESpace(mesh2d, order=0)  # ,all_dofs_together=True)
    uF, vF = fes_conformity.TnT()
    cop = u * vF * dx(element_boundary=True)
    crhs = uF * vF * dx(element_boundary=True)

    emb = TrefftzEmbedding(
        top=top,
        fes=fes,
        cop=cop,
        crhs=crhs,
        fes_conformity=fes_conformity,
        ndof_trefftz=2 * order + 1 - 3,
    )
    P = emb.GetEmbedding()

    rows, cols, vals = P.COO()

    P = scipy.sparse.csr_matrix((vals, (rows, cols)))
    P.todense()
    if debug:
        print(P)

    if debug:
        import matplotlib.pyplot as plt

        plt.spy(P)
        plt.show()

    a, f = dg.dgell(fes, dg.exactlap)
    rows, cols, vals = a.mat.COO()
    A = scipy.sparse.csr_matrix((vals, (rows, cols)))
    A = A.todense()

    TA = P.transpose() @ A @ P
    tsol = np.linalg.solve(TA, P.transpose() @ f.vec.FV())
    tpgfu = GridFunction(fes)
    tpgfu.vec.data = P @ tsol
    if debug:
        import netgen.gui

        Draw(tpgfu)
        input()
    return sqrt(Integrate((tpgfu - dg.exactlap) ** 2, mesh2d))


def test_conformiting_trefftz_with_rhs(order, order_conformity):
    """
    >>> test_conformiting_trefftz_with_rhs(5, 2) # doctest:+ELLIPSIS
    3...e-09

    >>> with TaskManager():
    ...   test_conformiting_trefftz_with_rhs(5, 2) # doctest:+ELLIPSIS
    3...e-09
    """
    mesh = Mesh(unit_square.GenerateMesh(maxh=0.3))
    fes = L2(mesh, order=order, dgjumps=True)
    start = time.time()
    u, v = fes.TnT()
    uh = u.Operator("hesse")
    vh = v.Operator("hesse")
    top = (uh[0, 0] + uh[1, 1]) * (vh[0, 0] + vh[1, 1]) * dx
    rhs = -exactpoi.Diff(x).Diff(x) - exactpoi.Diff(y).Diff(y)
    lop = -rhs * (vh[0, 0] + vh[1, 1]) * dx

    fes_conformity = FacetFESpace(
        mesh, order=order_conformity
    )  # ,all_dofs_together=True)
    uF, vF = fes_conformity.TnT()
    cop = u * vF * dx(element_boundary=True)
    crhs = uF * vF * dx(element_boundary=True)

    emb = TrefftzEmbedding(top, lop, cop, crhs, ndof_trefftz=2 * order + 1)
    PP = emb.GetEmbedding()
    uf = emb.GetParticularSolution()
    PPT = PP.CreateTranspose()
    a, f = dgell(fes, exactpoi, rhs)
    TA = PPT @ a.mat @ PP
    TU = TA.Inverse() * (PPT * (f.vec - a.mat * uf))
    tpgfu = GridFunction(fes)
    tpgfu.vec.data = PP * TU + uf
    return sqrt(Integrate((tpgfu - exactpoi) ** 2, mesh))


def test_conformiting_trefftz_trivial_mixed_mode(order, order_conformity):
    """
    Test, if the conforming trefftz procedure takes the trial space as test space,
    if no test space is given.

    >>> test_conformiting_trefftz_trivial_mixed_mode(5, 3)

    >>> with TaskManager():
    ...   test_conformiting_trefftz_trivial_mixed_mode(5, 3)
    """
    mesh2d = Mesh(unit_square.GenerateMesh(maxh=0.3))
    fes = L2(mesh2d, order=order, dgjumps=True)
    mesh = fes.mesh
    start = time.time()
    u, v = fes.TnT()
    uh = u.Operator("hesse")
    vh = v.Operator("hesse")
    top = (uh[0, 0] + uh[1, 1]) * (vh[0, 0] + vh[1, 1]) * dx
    rhs = -exactpoi.Diff(x).Diff(x) - exactpoi.Diff(y).Diff(y)
    lop = -rhs * (vh[0, 0] + vh[1, 1]) * dx

    fes_conformity = FacetFESpace(
        mesh2d, order=order_conformity
    )  # ,all_dofs_together=True)
    uF, vF = fes_conformity.TnT()
    cop = u * vF * dx(element_boundary=True)
    crhs = uF * vF * dx(element_boundary=True)

    emba = TrefftzEmbedding(top, lop, cop, crhs, ndof_trefftz=2 * order + 3 - 1)
    ta = emba.GetEmbedding()
    va = emba.GetParticularSolution()
    embb = TrefftzEmbedding(
        top, lop, cop, crhs, ndof_trefftz=2 * order + 3 - 1, fes_test=fes
    )
    tb = embb.GetEmbedding()
    vb = embb.GetParticularSolution()

    import scipy.sparse as sp

    rows, cols, vals = ta.COO()
    Ta = sp.csr_matrix((vals, (rows, cols)))
    rows, cols, vals = tb.COO()
    Tb = sp.csr_matrix((vals, (rows, cols)))

    assert np.isclose(Ta.toarray(), Tb.toarray()).all(), (
        "The embedding matrices do not agree"
    )
    assert np.isclose(va.FV().NumPy(), vb.FV().NumPy()).all(), (
        "The particular solutions disagree"
    )


def test_conforming_trefftz_mixed_mode(order, order_conformity):
    """
    >>> test_conforming_trefftz_mixed_mode(6, 2) # doctest:+ELLIPSIS
    6...e-11

    >>> with TaskManager():
    ...   test_conforming_trefftz_mixed_mode(6, 2) # doctest:+ELLIPSIS
    6...e-11
    """
    mesh2d = Mesh(unit_square.GenerateMesh(maxh=0.3))
    fes = L2(mesh2d, order=order, dgjumps=True)
    fes_test = L2(mesh2d, order=order - 2, dgjumps=True)
    mesh = fes.mesh
    start = time.time()
    u = fes.TrialFunction()
    v = fes_test.TestFunction()
    uh = u.Operator("hesse")
    # vh = v.Operator("hesse")
    top = (uh[0, 0] + uh[1, 1]) * v * dx
    rhs = -exactpoi.Diff(x).Diff(x) - exactpoi.Diff(y).Diff(y)
    lop = -rhs * v * dx

    fes_conformity = FacetFESpace(
        mesh2d, order=order_conformity
    )  # ,all_dofs_together=True)
    uF, vF = fes_conformity.TnT()
    cop = u * vF * dx(element_boundary=True)
    crhs = uF * vF * dx(element_boundary=True)

    emb = TrefftzEmbedding(top, lop, cop, crhs, ndof_trefftz=2 * order + 3 - 1)
    PP = emb.GetEmbedding()
    PPT = PP.CreateTranspose()
    uf = emb.GetParticularSolution()
    a, f = dgell(fes, exactpoi, rhs)
    TA = PPT @ a.mat @ PP
    TU = TA.Inverse() * (PPT * (f.vec - a.mat * uf))
    tpgfu = GridFunction(fes)
    tpgfu.vec.data = PP * TU + uf
    return sqrt(Integrate((tpgfu - exactpoi) ** 2, mesh))


def test_conforming_trefftz_without_op(order, order_conformity):
    """
    ## >>> test_conforming_trefftz_without_op(5, 3) < 1e-07
    ## True
    """
    mesh2d = Mesh(unit_square.GenerateMesh(maxh=0.3))
    fes = L2(mesh2d, order=order, dgjumps=True)
    mesh = fes.mesh
    start = time.time()
    u, v = fes.TnT()
    uh = u.Operator("hesse")
    vh = v.Operator("hesse")
    top = None
    rhs = -exactpoi.Diff(x).Diff(x) - exactpoi.Diff(y).Diff(y)
    lop = -rhs * (vh[0, 0] + vh[1, 1]) * dx

    fes_conformity = FacetFESpace(
        mesh2d, order=order_conformity
    )  # ,all_dofs_together=True)
    uF, vF = fes_conformity.TnT()
    cop = u * vF * dx(element_boundary=True)
    crhs = uF * vF * dx(element_boundary=True)

    emb = TrefftzEmbedding(top, lop, cop, crhs, ndof_trefftz=2 * order + 3 - 1)
    PP = emb.GetEmbedding()
    PPT = PP.CreateTranspose()
    uf = emb.GetParticularSolution()
    a, f = dgell(fes, exactpoi, rhs)
    TA = PPT @ a.mat @ PP
    TU = TA.Inverse() * (PPT * (f.vec - a.mat * uf))
    tpgfu = GridFunction(fes)
    tpgfu.vec.data = PP * TU + uf
    return sqrt(Integrate((tpgfu - exactpoi) ** 2, mesh))


def test_ConstrainedTrefftzFESpace(
    order: int = 2, debug: bool = False, maxh=0.4
) -> float:
    """
    simple test case for EmbTrefftzFESpace with conforming Trefftz.

    `order`: polynomial oder of the underlying space

    `debug`: True: print debug info, default: False

    >>> test_ConstrainedTrefftzFESpace(order=3, debug=False) # doctest:+ELLIPSIS
    3...e-05

    >>> with TaskManager():
    ...   test_ConstrainedTrefftzFESpace(order=3, debug=False) # doctest:+ELLIPSIS
    3...e-05
    """
    mesh2d = Mesh(unit_square.GenerateMesh(maxh=maxh))

    fes = L2(mesh2d, order=order, dgjumps=True)  # ,all_dofs_together=True)

    u, v = fes.TnT()
    uh = u.Operator("hesse")
    vh = v.Operator("hesse")
    top = (uh[0, 0] + uh[1, 1]) * (vh[0, 0] + vh[1, 1]) * dx

    fes_conformity = FacetFESpace(mesh2d, order=0)  # ,all_dofs_together=True)
    uF, vF = fes_conformity.TnT()
    cop = u * vF * dx(element_boundary=True)
    crhs = uF * vF * dx(element_boundary=True)

    emb = TrefftzEmbedding(
        top=top, cop=cop, crhs=crhs, ndof_trefftz=2 * order + 1 - 3
    )
    fes_trefftz = EmbeddedTrefftzFES(emb)

    a, f = dg.dgell(fes_trefftz, dg.exactlap)
    a.Assemble()
    f.Assemble()

    A_inv = a.mat.Inverse()
    tpgfu = GridFunction(fes_trefftz)
    tpgfu.vec.data = A_inv * f.vec
    if debug:
        import netgen.gui

        Draw(tpgfu)
        input()
    return sqrt(Integrate((tpgfu - dg.exactlap) ** 2, mesh2d))


def test_ConstrainedTrefftzFESpaceDirichlet():
    """
    >>> test_ConstrainedTrefftzFESpaceDirichlet()

    >>> with TaskManager():
    ...   test_ConstrainedTrefftzFESpaceDirichlet()
    """
    mesh = Mesh(unit_square.GenerateMesh(maxh=0.2))
    fes_base = L2(mesh, order=4)
    fes_conformity = H1(mesh, order=2, dirichlet="left|right") * FacetFESpace(
        mesh, order=3
    )

    u = fes_base.TrialFunction()
    ((uc_h, uc_f), (vc_h, vc_f)) = fes_conformity.TnT()
    cop = u * vc_h * dx(element_vb=BBND)
    crhs = uc_h * vc_h * dx(element_vb=BBND)

    cop += u * vc_f * dx(element_vb=BND)
    crhs += uc_f * vc_f * dx(element_vb=BND)

    emb = TrefftzEmbedding(top=None, cop=cop, crhs=crhs)
    fes = EmbeddedTrefftzFES(emb)

    assert type(fes) is L2EmbTrefftzFESpace
    assert fes.ndof >= fes_conformity.ndof

    fes_freedofs = fes.FreeDofs()
    fes_conformity_freedofs = fes_conformity.FreeDofs()
    for i in range(fes_conformity.ndof):
        assert fes_freedofs[i] == fes_conformity_freedofs[i]


def test_ConstrainedTrefftzFESpaceEmbed():
    """
    Tests, whether the embedded gfu matches the gfu of the EmbeddedTrefftzFES.
    >>> [test_ConstrainedTrefftzFESpaceEmbed()] # doctest:+ELLIPSIS
    [...e-32]
    """

    def hesse(f: CoefficientFunction) -> CoefficientFunction:
        return f.Operator("hesse")

    def del_x(f: CoefficientFunction) -> CoefficientFunction:
        return grad(f)[0]

    def del_y(f: CoefficientFunction) -> CoefficientFunction:
        return grad(f)[1]

    def del_xx(f: CoefficientFunction) -> CoefficientFunction:
        return hesse(f)[0, 0]

    def del_xy(f: CoefficientFunction) -> CoefficientFunction:
        return hesse(f)[0, 1]

    def del_yy(f: CoefficientFunction) -> CoefficientFunction:
        return hesse(f)[1, 1]

    mesh = Mesh(unit_square.GenerateMesh(maxh=0.2))

    order = 7
    fes = L2(mesh, order=order)

    vertex_value_space = H1(mesh, order=1)
    deriv_x_value_space = H1(mesh, order=1)
    deriv_y_value_space = H1(mesh, order=1)
    deriv_xx_value_space = H1(mesh, order=1)
    deriv_xy_value_space = H1(mesh, order=1)
    deriv_yy_value_space = H1(mesh, order=1)
    normal_deriv_moment_space = NormalFacetFESpace(mesh, order=order - 5)
    facet_moment_space = FacetFESpace(mesh, order=order - 6)

    conformity_space = (
        vertex_value_space
        * deriv_x_value_space
        * deriv_y_value_space
        * deriv_xx_value_space
        * deriv_xy_value_space
        * deriv_yy_value_space
        * normal_deriv_moment_space
        * facet_moment_space
    )

    u = fes.TrialFunction()
    (u_, u_dx, u_dy, u_dxx, u_dxy, u_dyy, u_n, u_f) = (
        conformity_space.TrialFunction()
    )
    (v_, v_dx, v_dy, v_dxx, v_dxy, v_dyy, v_n, v_f) = (
        conformity_space.TestFunction()
    )

    dVertex = dx(element_vb=BBND)
    dFace = dx(element_vb=BND)
    n = specialcf.normal(2)

    cop_lhs = (
        u * v_ * dVertex
        + del_x(u) * v_dx * dVertex
        + del_y(u) * v_dy * dVertex
        + del_xx(u) * v_dxx * dVertex
        + del_xy(u) * v_dxy * dVertex
        + del_yy(u) * v_dyy * dVertex
        + grad(u) * n * v_n * n * dFace
        + u * v_f * dFace
    )
    cop_rhs = (
        u_ * v_ * dVertex
        + u_dx * v_dx * dVertex
        + u_dy * v_dy * dVertex
        + u_dxx * v_dxx * dVertex
        + u_dxy * v_dxy * dVertex
        + u_dyy * v_dyy * dVertex
        + u_n * n * v_n * n * dFace
        + u_f * v_f * dFace
    )

    embedding = TrefftzEmbedding(cop=cop_lhs, crhs=cop_rhs, ndof_trefftz=0)

    argyris = EmbeddedTrefftzFES(embedding)

    assert type(argyris) is L2EmbTrefftzFESpace

    gfu = GridFunction(argyris)
    for i in range(0, len(gfu.vec), 2):
        gfu.vec.data[i] = 1.0
    base_gfu = GridFunction(fes)
    base_gfu.vec.data[:] = embedding.Embed(gfu.vec)
    base_gfu_2 = embedding.Embed(gfu)

    # 1. Test, if Embed(gfu) gives the same result as Embed(gfu.vec)
    assert len(base_gfu.vec) == len(base_gfu_2.vec)
    for i in range(len(base_gfu.vec)):
        assert base_gfu.vec.data[i] == base_gfu_2.vec.data[i], (
            f"i: {i}, {base_gfu.vec.data[i]} != {base_gfu_2.vec.data[i]}"
        )

    # 2. Test, if the L2 gfu is the same function as the Argyris gfu
    return Integrate((gfu - base_gfu) ** 2, mesh)


if __name__ == "__main__":
    import doctest

    doctest.testmod()
