#!/usr/bin/env python3
"""Run DSPy MIPROv2 optimization on Banking77 intent classification."""

from __future__ import annotations

import asyncio
from pathlib import Path

from dspy_banking77_adapter import run_dspy_miprov2_banking77


async def main():
    """Main entry point for Banking77 MIPROv2 runner."""
    import argparse

    parser = argparse.ArgumentParser(description="Run DSPy MIPROv2 on Banking77")
    parser.add_argument(
        "--task-app-url",
        default="http://127.0.0.1:8102",
        help="Task app URL (for reference)",
    )
    parser.add_argument(
        "--rollout-budget",
        type=int,
        default=200,
        help="Rollout budget (default: 200)",
    )
    parser.add_argument(
        "--output-dir",
        type=Path,
        default=None,
        help="Output directory",
    )

    args = parser.parse_args()

    # Run MIPROv2 optimization with seeds matching synth_mipro_config.yaml (banking77_dspy)
    train_seeds = list(range(25))  # 0-24: 25 training examples (matches bootstrap_train_seeds)
    val_seeds = list(range(50, 250))  # 50-249: 200 validation examples (large valset for robust evaluation)

    results = await run_dspy_miprov2_banking77(
        task_app_url=args.task_app_url,
        train_seeds=train_seeds,
        val_seeds=val_seeds,
        rollout_budget=args.rollout_budget,
        output_dir=args.output_dir,
    )

    print(f"\n✅ MIPROv2 Optimization complete!")
    print(f"   Best score: {results['best_score']:.4f}")
    print(f"   Val score: {results.get('val_score', 'N/A')}")
    print(f"   Total rollouts: {results['total_rollouts']}")
    print(f"   Prompt file: {results.get('prompt_file', 'N/A')}")


if __name__ == "__main__":
    asyncio.run(main())
